(ns currency.core
  (:require [currency.currencies.core :refer [monetary-unit->string]]
            [currency.currencies.cad]
            [currency.currencies.usd]
            [currency.round :as round]
            [clojure.spec.alpha :as s]))

;;; Declarations

(declare validate)

;;; Records

(defrecord MonetaryUnit [currency amount]
  Object
  (toString [this]
    (monetary-unit->string this)))

;;; API

(defn monetary-unit
  [currency amount]
  {:pre [(validate
          (s/cat :currency ::currency
                 :amount ::amount)
          [currency amount])]
   :post [(validate ::monetary-unit %)]}
  (->MonetaryUnit currency amount))

(defn round
  [unit & {:keys [mode] :or {mode :half-up}}]
  {:pre [(validate (s/nilable ::monetary-unit) unit)
         (validate ::rounding-mode mode)]}
  (when unit
    (condp = mode
      :half-up (update unit :amount round/half-up)
      :ceil (update unit :amount round/ceil)
      :floor (update unit :amount round/floor)
      unit)))

(defn add
  "Sum the monetary units together. We try to be flexible and allow nil or empty
  inputs. The result of providing nil or empty inputs will be nil."
  [& units]
  {:pre [(validate ::add-inputs units)]}
  (when-let [units (not-empty (filter :currency units))]
    (monetary-unit
     (:currency (first units))
     (apply + (map :amount units)))))

(defn multiply
  "Multiply the monetary units or constants together. We try to be flexible and
  allow nil or empty inputs. The result of providing nil or empty inputs will be
  nil."
  [& units-or-constants]
  {:pre [(validate ::multiply-inputs units-or-constants)]}
  (let [constants (remove
                   #(or (nil? %) (:currency %))
                   units-or-constants)]
    (when-let [units (not-empty (filter :currency units-or-constants))]
      (when-let [combined (->> units-or-constants
                               (map #(or (:amount %) %))
                               (remove nil?)
                               not-empty)]
        (monetary-unit
         (:currency (first units))
         (apply * combined))))))

;;; Specs

(s/def ::currency #{:cad :usd})

;; Must be a number, and must not exceed maximum value for a long
(s/def ::amount (s/and number? long))

(s/def ::number number?)
(s/def :unq/monetary-unit
  (s/keys :req-un [::currency ::amount]))

(s/def ::unit-or-constant
  (s/or :number ::number
        :monetary-unit ::monetary-unit))

(s/def ::rounding-mode #{:half-up :ceil :floor})

(s/def ::monetary-unit :unq/monetary-unit)

(s/def ::same-currency
  #(boolean
    (let [currencies (->> % (map :currency) (remove nil?) seq)]
      (if currencies
        (->> currencies
             (map :currency)
             (apply =))
        true))))

(s/def ::multiply-inputs
  (s/nilable
   (s/or
    :empty empty?
    :multiply-inputs (s/and
                      (s/every (s/nilable ::unit-or-constant))
                      ::same-currency))))

(s/def ::add-input
  (s/nilable ::monetary-unit))

(s/def ::add-inputs
  (s/nilable
   (s/or
    :empty empty?
    :add-inputs (s/and
                 (s/every ::add-input)
                 ::same-currency))))

;;; Private

(defn- validate
  [spec form]
  (when-not (s/valid? spec form)
    (throw (ex-info "Spec validation failed."
                    {:spec spec
                     :form form
                     :explain (s/explain-str spec form)})))
  true)
