(ns tag.deps
  (:require [clojure.java.io :as io]
            [clojure.string :as s]
            [tag.shell :as sh]))

(defn- clojure-tree []
  (let [result (sh/sh "clojure" "-Stree")]
    (when (zero? (:exit result))
      (:out result))))

(defn- parse-deps-tree [tree-output]
  (when tree-output
    (let [lines (s/split-lines tree-output)
          parse-line (fn [line]
                       (let [indent (count (re-find #"^ +" line))
                             level (if (pos? indent) (/ indent 2) 0)
                             dep-str (s/trim line)
                             [coord version] (if (s/includes? dep-str " ")
                                               (s/split dep-str #" " 2)
                                               [dep-str nil])
                             version (when version (s/replace version #"[()]" ""))]
                         {:coordinate coord
                          :version version
                          :level level}))]
      (remove #(s/blank? (:coordinate %)) (map parse-line lines)))))

(defn- build-deps-tree [parsed-deps]
  (let [;; direct dependencies (level 0)
        direct-deps (filter #(zero? (:level %)) parsed-deps)

        ;; create path map from dependency to its parents
        deps-hierarchy (reduce (fn [acc [parent & children]]
                                 (if (seq children)
                                   (reduce (fn [a child]
                                             (update-in a [(:coordinate child) :via]
                                                        (fnil conj #{}) (:coordinate parent)))
                                           acc children)
                                   acc))
                               {}
                               (partition-by :level parsed-deps))

        ;; build the dependency map with basic details
        build-dep-map (fn [deps]
                        (reduce (fn [m {:keys [coordinate version level]}]
                                  (let [dep-info {:version version}
                                        dep-info (if-let [via (get-in deps-hierarchy [coordinate :via])]
                                                   (assoc dep-info :via (vec via))
                                                   dep-info)]
                                    (assoc m coordinate dep-info)))
                                {} deps))]

    {:direct-deps (build-dep-map direct-deps)
     :all-deps (build-dep-map parsed-deps)}))

(defn find-deps
  "collect dependencies information.
   returns a map with :direct-deps and :all-deps keys."
  []
  (let [tree (clojure-tree)]
    (if tree
      (->> tree
           parse-deps-tree
           build-deps-tree)
      {:direct-deps {}, :all-deps {}})))
