;; Copyright 2011-2015 Michael S. Klishin, Alex Petrov, and the ClojureWerkz Team
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns clojurewerkz.elastisch.rest.document
  "Key operations on documents: indexing, search, deletion, etc"
  (:refer-clojure :exclude [get replace count sort])
  (:require [clojurewerkz.elastisch.rest :as rest]
            [cheshire.core :as json]
            [clojure.string :as string]
            [clojure.set :refer :all]
            [clojurewerkz.elastisch.rest.utils :refer [join-names]]
            [clojurewerkz.elastisch.arguments :as ar]
            [clojurewerkz.elastisch.rest.response :refer [not-found? hits-from]])
  (:import clojurewerkz.elastisch.rest.Connection))

;;
;; API
;;

(defn create
  "Adds document to the search index. Document id will be generated automatically.

   Options:

     * :id (string): unique document id. If not provided, it will be generated by ElasticSearch
     * :timestamp (string): document timestamp either as millis since the epoch,
                                          or, in the configured date format
     * :ttl (long): document TTL in milliseconds. Must be > 0
     * :refresh (boolean, default: false): should a refresh be executed post this index operation?
     * :version (long): document version
     * :version-type (string, default: \"internal\"): \"internal\" or \"external\"
     * :content-type (string): document content type
     * :routing (string): controls the shard routing of the request. Using this value to hash the shard
                          and not the id
     * :percolate (string): the percolate query to use to reduce the percolated queries that are going to run against this doc.
                           Can be set to \"*\" which means \"all queries\"
     * :parent (string): parent document id

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])

   (doc/create conn \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28})

   (doc/create conn \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28} :id \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  ([^Connection conn index mapping-type document & args]
     (rest/post conn (rest/mapping-type-url conn
                                            index mapping-type)
                {:body document :query-params (ar/->opts args)})))

(defn put
  "Creates or updates a document in the search index, using the provided document id"
  ([^Connection conn index mapping-type id document]
     (rest/put conn (rest/record-url conn
                                     index mapping-type id)
               {:body document}))
  ([^Connection conn index mapping-type id document & args]
     (rest/put conn (rest/record-url conn
                                     index mapping-type id)
               {:body document :query-params (ar/->opts args)})))

(defn upsert
  "Updates an existing document in the search index with given partial document,
  the provided document will be inserted if the document does not already exist"
  ([^Connection conn index mapping-type id partial-doc]
    (rest/post conn (rest/record-update-url conn
                                            index mapping-type id) {:body {:doc partial-doc
                                                                           :doc_as_upsert true}}))
  ([^Connection conn index mapping-type id partial-doc & args]
    (rest/post conn (rest/record-update-url conn
                                            index mapping-type id)
               {:body {:doc partial-doc
                       :doc_as_upsert true}
                :query-params (ar/->opts args)})))

(defn update-with-partial-doc
  "Updates an existing document in the search index with given partial document"
  ([^Connection conn index mapping-type id partial-doc]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id) {:body {:doc partial-doc}}))
  ([^Connection conn index mapping-type id partial-doc & args]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body {:doc partial-doc} :query-params (ar/->opts args)})))

(defn update-with-script
  "Updates a document using a script.

  Examples:

  (require '[clojurewerkz.elastisch.rest.document :as doc])

  (doc/update-with-script conn \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\"
                                  \"ctx._source.age = ctx._source.age += 1\" {} :lang \"groovy\")
  "
  ([^Connection conn index mapping-type id script]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body {:script script}}))
  ([^Connection conn index mapping-type id script params]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body {:script script :params params}}))
  ([^Connection conn index mapping-type id script params & args]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                (let [optional-params (ar/->opts args)]
                  {:body (merge {:script script :params params} optional-params)}))))


(defn get
  "Fetches and returns a document by id or nil if it does not exist.

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])

   (doc/get conn \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\")"
  [^Connection conn index mapping-type id & args]
  (let [result (rest/get conn (rest/record-url conn
                                               index mapping-type id)
                         {:query-params (ar/->opts args)})]
    (if (not-found? result)
      nil
      result)))

(defn delete
  "Deletes document from the index."
  ([^Connection conn index mapping-type id]
     (rest/delete conn (rest/record-url conn
                                        index mapping-type id)))
  ([^Connection conn index mapping-type id & args]
     (rest/delete conn(rest/record-url conn
                                       index mapping-type id)
                  {:query-params (ar/->opts args)})))

(defn present?
  "Returns true if a document with the given id is present in the provided index
   with the given mapping type."
  [^Connection conn index mapping-type id]
  (not (nil? (get conn index mapping-type id))))

(defn multi-get
  "Multi get returns only documents that are found (exist).

   Queries can passed as a collection of maps with three keys: :_index,
   :_type and :_id:

   (doc/multi-get conn [{:_index index-name :_type mapping-type :_id \"1\"}
                        {:_index index-name :_type mapping-type :_id \"2\"}])


   2-argument version accepts an index name that eliminates the need to include
   :_index in every query map:

   (doc/multi-get conn index-name [{:_type mapping-type :_id \"1\"}
                                   {:_type mapping-type :_id \"2\"}])

   3-argument version also accepts a mapping type that eliminates the need to include
   :_type in every query map:

   (doc/multi-get conn index-name mapping-type [{:_id \"1\"}
                                                {:_id \"2\"}])"
  ([^Connection conn query]
     (let [results (rest/post conn (rest/index-mget-url conn)
                              {:body {:docs query}})]
       (filter :found (:docs results))))
  ([^Connection conn index query]
     (let [results (rest/post conn (rest/index-mget-url conn
                                                        index)
                              {:body {:docs query}})]
       (filter :found (:docs results))))
  ([^Connection conn index mapping-type query]
     (let [results (rest/post conn (rest/index-mget-url conn
                                                        index mapping-type)
                              {:body {:docs query}})]
       (filter :found (:docs results)))))

(defn search
  "Performs a search query across one or more indexes and one or more mapping types.

   Passing index name as \"_all\" means searching across all indexes.

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/search conn \"people\" \"person\" :query (q/prefix :username \"appl\"))"
  [^Connection conn index mapping-type & args]
  (let [opts (ar/->opts args)
        qk   [:search_type :scroll :routing :preference :ignore_unavailable]
        qp   (select-keys opts qk)
        body (apply dissoc (concat [opts] qk))]
    (rest/post conn (rest/search-url conn
                                     (join-names index)
                                     (join-names mapping-type))
               {:body body
                :query-params qp})))

(defn search-all-types
  "Performs a search query across one or more indexes and all mapping types."
  [^Connection conn index & args]
  (let [opts (ar/->opts args)
        qk   [:search_type :scroll :routing :preference :ignore_unavailable]
        qp   (select-keys opts qk)
        body (apply dissoc (concat [opts] qk))]
    (rest/post conn (rest/search-url conn
                                     (join-names index))
               {:body body
                :query-params qp})))

(defn search-all-indexes-and-types
  "Performs a search query across all indexes and all mapping types.
   This may put very high load on your ElasticSearch cluster so use
   this function with care."
  [^Connection conn & args]
  (let [opts (ar/->opts args)
        qk   [:search_type :scroll :routing :preference]
        qp   (select-keys opts qk)
        body (apply dissoc (concat [opts] qk))]
    (rest/post conn (rest/search-url conn)
               {:body body
                :query-params qp})))

(defn scroll
  "Performs a scroll query, fetching the next page of results from a
   query given a scroll id"
  [^Connection conn scroll-id & args]
  (let [opts (ar/->opts args)
        qk   [:search_type :scroll :routing :preference]
        qp   (select-keys opts qk)
        body scroll-id]
    (rest/post-string conn (rest/scroll-url conn)
                      {:body body
                       :query-params qp})))

(defn scroll-seq
  "Returns a lazy sequence of all documents for a given scroll query"
  ([^Connection conn prev-resp {:keys [search_type]}]
   (let [hits (hits-from prev-resp)
         scroll-id (:_scroll_id prev-resp)]
     (if (or (seq hits) (= search_type "scan"))
       (concat hits (lazy-seq (scroll-seq conn (scroll conn scroll-id :scroll "1m"))))
       hits)))
  ([^Connection conn prev-resp]
   (scroll-seq conn prev-resp nil)))

(defn replace
  "Replaces document with given id with a new one"
  [^Connection conn index mapping-type id document]
  (delete conn index mapping-type id)
  (put conn index mapping-type id document))


(defn count
  "Performs a count query.

   Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])
   (require '[clojurewerkz.elastisch.query :as q])

   (doc/count conn \"people\" \"person\")
   (doc/count conn \"people\" \"person\" (q/prefix :username \"appl\"))"
  ([^Connection conn index mapping-type]
     (rest/get conn (rest/count-url conn
                                    (join-names index) (join-names mapping-type))))
  ([^Connection conn index mapping-type query]
     (rest/post conn (rest/count-url conn
                                     (join-names index) (join-names mapping-type))
                {:body {:query query}}))
  ([^Connection conn index mapping-type query & args]
     (rest/post conn (rest/count-url conn
                                     (join-names index) (join-names mapping-type))
                {:query-params (select-keys (ar/->opts args) [:df :analyzer :default_operator :ignore_unavailable])
                 :body {:query query}})))

(def ^{:doc "Optional parameters that all query-based delete functions share"
       :const true}
  optional-delete-query-parameters [:df :analyzer :default_operator :consistency])

;;NB! requires delete-by-query plugin
(defn delete-by-query
  "Performs a delete-by-query operation."
  ([^Connection conn index mapping-type query]
     (rest/delete conn
                  (rest/delete-by-query-url
                    conn
                    (join-names index)
                    (join-names mapping-type))
                  {:body {:query query}}))
  ([^Connection conn index mapping-type query & args]
     (rest/delete conn
                  (rest/delete-by-query-url
                    conn
                    (join-names index)
                    (join-names mapping-type))
                  {:query-params (select-keys (ar/->opts args)
                                              (conj optional-delete-query-parameters
                                                    :ignore_unavailable))
                   :body {:query query}})))

(defn delete-by-query-across-all-types
  "Performs a delete-by-query operation across all mapping types."
  ([^Connection conn index query]
     (rest/delete conn
                  (rest/delete-by-query-url conn (join-names index))
                  {:body {:query query}}))
  ([^Connection conn index query & args]
     (rest/delete conn
                  (rest/delete-by-query-url conn (join-names index))
                  {:query-params (select-keys (ar/->opts args)
                                              (conj optional-delete-query-parameters
                                                    :ignore_unavailable))
                   :body {:query query}})))

(defn delete-by-query-across-all-indexes-and-types
  "Performs a delete-by-query operation across all indexes and mapping types.
   This may put very high load on your ElasticSearch cluster so use this function with care."
  ([^Connection conn query]
     (rest/delete conn (rest/delete-by-query-url conn) {:body {:query query}}))
  ([^Connection conn query & args]
     (rest/delete conn (rest/delete-by-query-url conn)
                  {:query-params (select-keys (ar/->opts args) optional-delete-query-parameters)
                   :body {:query query}})))


(defn more-like-this
  "Performs a More Like This (MLT) query."
  [^Connection conn index mapping-type & args]
  (rest/get conn
            (rest/more-like-this-url conn index mapping-type)
            {:body (json/encode {:query {:mlt (ar/->opts args)}})}))

(defn validate-query
  "Validates a query without actually executing it. Has the same API as clojurewerkz.elastisch.rest.document/search
   but does not take the mapping type parameter."
  [^Connection conn index query & args]
  (rest/get conn (rest/query-validation-url conn
                                            index)
            {:body (json/encode {:query query})
             :query-params (ar/->opts args)}))


(defn analyze
  "Examples:

   (require '[clojurewerkz.elastisch.rest.document :as doc])

   (doc/analyze conn \"foo bar baz\")
   (doc/analyze conn \"foo bar baz\" :index \"some-index-name\")
   (doc/analyze conn \"foo bar baz\" :analyzer \"whitespace\")
   (doc/analyze conn \"foo bar baz\" :index \"some-index-name\" :field \"some-field-name\")"
  ([^Connection conn text & args]
     (let [opts (ar/->opts args)]
       (rest/get conn (rest/analyze-url conn
                                        (:index opts))
                 {:query-params (assoc opts :text text)}))))
