;; Copyright 2011-2016 Michael S. Klishin, Alex Petrov, and the ClojureWerkz Team
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns clojurewerkz.elastisch.rest.document
  "Key operations on documents: indexing, search, deletion, etc"
  (:refer-clojure :exclude [get replace count sort])
  (:require [clojurewerkz.elastisch.rest :as rest]
            [cheshire.core :as json]
            [clojure.string :as string]
            [clojure.set :refer :all]
            [clojurewerkz.elastisch.rest.utils :refer [join-names]]
            [clojurewerkz.elastisch.rest.response :refer [not-found? hits-from]])
  (:import clojurewerkz.elastisch.rest.Connection))

;;
;; API
;;

(defn create
  "Adds document to the search index. Document id will be generated automatically.

  Options:

    * `:id` (string): unique document id. If not provided, it will be generated by Elasticsearch
    * `:timestamp` (string): document timestamp either as millis since the epoch,
                                         or, in the configured date format
    * `:ttl` (long): document TTL in milliseconds. Must be > 0
    * `:refresh` (boolean, default: `false`): should a refresh be executed post this index operation?
    * `:version` (long): document version
    * `:version-type` (string, default: `\"internal\"`): `\"internal\"` or `\"external\"`
    * `:content-type` (string): document content type
    * `:routing` (string): controls the shard routing of the request. Using this value to hash the shard
                         and not the id
    * `:percolate` (string): the percolate query to use to reduce the percolated queries that are going to run against this doc.
                          Can be set to `\"*\"` which means \"all queries\"
    * `:parent` (string): parent document id

  Examples:

  ```clojure
  (require '[clojurewerkz.elastisch.rest.document :as doc])

  (doc/create conn \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28})

  (doc/create conn \"people\" \"person\" {:first-name \"John\" :last-name \"Appleseed\" :age 28} {:id \"1825c5432775b8d1a477acfae57e91ac8c767aed\"})
  ```"
  ([^Connection conn index mapping-type document] (create conn index mapping-type document nil))
  ([^Connection conn index mapping-type document opts]
     (rest/post conn (rest/mapping-type-url conn
                                            index mapping-type)
                {:body document :query-params opts})))

(defn put
  "Creates or updates a document in the search index, using the provided document id"
  ([^Connection conn index mapping-type id document]
     (rest/put conn (rest/record-url conn
                                     index mapping-type id)
               {:body document}))
  ([^Connection conn index mapping-type id document opts]
     (rest/put conn (rest/record-url conn
                                     index mapping-type id)
               {:body document :query-params opts})))

(defn upsert
  "Updates an existing document in the search index with given partial document,
  the provided document will be inserted if the document does not already exist"
  ([^Connection conn index mapping-type id partial-doc]
    (rest/post conn (rest/record-update-url conn
                                            index mapping-type id) {:body {:doc partial-doc
                                                                           :doc_as_upsert true}}))
  ([^Connection conn index mapping-type id partial-doc opts]
    (rest/post conn (rest/record-update-url conn
                                            index mapping-type id)
               {:body {:doc partial-doc
                       :doc_as_upsert true}
                :query-params opts})))

(defn update-with-partial-doc
  "Updates an existing document in the search index with given partial document"
  ([^Connection conn index mapping-type id partial-doc]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id) {:body {:doc partial-doc}}))
  ([^Connection conn index mapping-type id partial-doc opts]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body {:doc partial-doc} :query-params opts})))

(defn update-with-script
  "Updates a document using a script.

  Examples:

  ```clojure
  (require '[clojurewerkz.elastisch.rest.document :as doc])

  (doc/update-with-script conn \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\"
                               \"ctx._source.age = ctx._source.age += 1\" {} {:lang \"groovy\"})
  ```"
  ([^Connection conn index mapping-type id script]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body {:script script}}))
  ([^Connection conn index mapping-type id script params]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body {:script script :params params}}))
  ([^Connection conn index mapping-type id script params opts]
     (rest/post conn (rest/record-update-url conn
                                             index mapping-type id)
                {:body (merge {:script script :params params}
                              opts)})))


(defn get
  "Fetches and returns a document by id or `nil` if it does not exist.

  Examples:

  ```clojure
  (require '[clojurewerkz.elastisch.rest.document :as doc])

  (doc/get conn \"people\" \"person\" \"1825c5432775b8d1a477acfae57e91ac8c767aed\")
  ```"
  ([^Connection conn index mapping-type id] (get conn index mapping-type id nil))
  ([^Connection conn index mapping-type id opts]
   (let [result (rest/get conn (rest/record-url conn
                                                index mapping-type id)
                          {:query-params opts})]
     (if (not-found? result)
       nil
       result))))

(defn delete
  "Deletes document from the index."
  ([^Connection conn index mapping-type id]
     (rest/delete conn (rest/record-url conn
                                        index mapping-type id)))
  ([^Connection conn index mapping-type id opts]
     (rest/delete conn(rest/record-url conn
                                       index mapping-type id)
                  {:query-params opts})))

(defn present?
  "Returns true if a document with the given id is present in the provided index
  with the given mapping type."
  [^Connection conn index mapping-type id]
  (not (nil? (get conn index mapping-type id))))

(defn multi-get
  "Multi get returns only documents that are found (exist).

  Queries can passed as a collection of maps with three keys: `:_index`,
  `:_type` and `:_id`:

  ```clojure
  (doc/multi-get conn [{:_index index-name :_type mapping-type :_id \"1\"}
                       {:_index index-name :_type mapping-type :_id \"2\"}])
  ```

  2-argument version accepts an index name that eliminates the need to include
  `:_index` in every query map:

  ```clojure
  (doc/multi-get conn index-name [{:_type mapping-type :_id \"1\"}
                                  {:_type mapping-type :_id \"2\"}])
  ```

  3-argument version also accepts a mapping type that eliminates the need to include
  `:_type` in every query map:

  ```clojure
  (doc/multi-get conn index-name mapping-type [{:_id \"1\"}
                                               {:_id \"2\"}])
  ```"
  ([^Connection conn query]
     (let [results (rest/post conn (rest/index-mget-url conn)
                              {:body {:docs query}})]
       (filter :found (:docs results))))
  ([^Connection conn index query]
     (let [results (rest/post conn (rest/index-mget-url conn
                                                        index)
                              {:body {:docs query}})]
       (filter :found (:docs results))))
  ([^Connection conn index mapping-type query]
     (let [results (rest/post conn (rest/index-mget-url conn
                                                        index mapping-type)
                              {:body {:docs query}})]
       (filter :found (:docs results)))))

(defn search
  "Performs a search query across one or more indexes and one or more mapping types.

  Passing index name as `\"_all\"` means searching across all indexes, multiple
  indexes or types can be passed in as a seq of strings.

  Examples:

  ```clojure
  (require '[clojurewerkz.elastisch.rest.document :as doc])
  (require '[clojurewerkz.elastisch.query :as q])

  (doc/search conn \"people\" \"person\" :query (q/prefix {:username \"appl\"}))
  ```"
  ([^Connection conn index mapping-type] (search conn index mapping-type nil))
  ([^Connection conn index mapping-type opts]
   (let [qk [:search_type :scroll :routing :preference :ignore_unavailable]
         qp (select-keys opts qk)
         body (apply dissoc (concat [opts] qk))]
     (rest/post conn (rest/search-url conn
                                      (join-names index)
                                      (join-names mapping-type))
                {:body body
                 :query-params qp}))))

(defn search-all-types
  "Performs a search query across one or more indexes and all mapping types.

  Multiple indexes can be passed in as a seq of strings."
  ([^Connection conn index] (search-all-types conn index nil))
  ([^Connection conn index opts]
   (let [qk   [:search_type :scroll :routing :preference :ignore_unavailable]
         qp   (select-keys opts qk)
         body (apply dissoc (concat [opts] qk))]
     (rest/post conn (rest/search-url conn
                                      (join-names index))
                {:body body
                 :query-params qp}))))

(defn search-all-indexes-and-types
  "Performs a search query across all indexes and all mapping types.
  This may put very high load on your Elasticsearch cluster so use
  this function with care."
  ([^Connection conn] (search-all-types conn nil))
  ([^Connection conn opts]
   (let [qk   [:search_type :scroll :routing :preference]
         qp   (select-keys opts qk)
         body (apply dissoc (concat [opts] qk))]
     (rest/post conn (rest/search-url conn)
                {:body body
                 :query-params qp}))))

(defn scroll
  "Performs a scroll query, fetching the next page of results from a
  query given a scroll id"
  ([^Connection conn scroll-id] (scroll conn scroll-id nil))
  ([^Connection conn scroll-id opts]
   (let [qk [:search_type :scroll :routing :preference]
         qp   (select-keys opts qk)
         body scroll-id]
     (rest/post-string conn (rest/scroll-url conn)
                       {:body body
                        :query-params qp}))))

(defn scroll-seq
  "Returns a lazy sequence of all documents for a given scroll query"
  ([^Connection conn prev-resp {scroll-ttl :scroll
                                :or {scroll-ttl "1m"}
                                :keys [search_type]}]
   (let [hits (hits-from prev-resp)
         scroll-id (:_scroll_id prev-resp)]
     (if (or (seq hits) (= search_type "scan"))
       (->> (scroll conn scroll-id {:scroll scroll-ttl})
            (#(scroll-seq conn % {:scroll scroll-ttl}))
            (lazy-seq)
            (concat hits))
       hits)))
  ([^Connection conn prev-resp]
   (scroll-seq conn prev-resp nil)))

(defn clear-scroll
  "Clear the scroll."
  [^Connection conn scroll-id]
  (rest/delete conn
               (rest/url-with-path conn "_search" "scroll")
               {:body {:scroll_id [scroll-id]}}))

(defn replace
  "Replaces document with given id with a new one"
  [^Connection conn index mapping-type id document]
  (delete conn index mapping-type id)
  (put conn index mapping-type id document))


(defn count
  "Performs a count query over one or more indexes and types.

  Multiple indexes and types can be specified using a seq of strings, otherwise
  specifying a string is enough.

  ```clojure
  Examples:

  (require '[clojurewerkz.elastisch.rest.document :as doc])
  (require '[clojurewerkz.elastisch.query :as q])

  (doc/count conn \"people\" \"person\")
  (doc/count conn \"people\" \"person\" (q/prefix {:username \"appl\"}))
  ```"
  ([^Connection conn index mapping-type]
     (rest/get conn (rest/count-url conn
                                    (join-names index) (join-names mapping-type))))
  ([^Connection conn index mapping-type query]
     (rest/post conn (rest/count-url conn
                                     (join-names index) (join-names mapping-type))
                {:body {:query query}}))
  ([^Connection conn index mapping-type query opts]
     (rest/post conn (rest/count-url conn
                                     (join-names index) (join-names mapping-type))
                {:query-params (select-keys opts [:df :analyzer :default_operator :ignore_unavailable])
                 :body {:query query}})))

(def ^{:doc "Optional parameters that all query-based delete functions share"
       :doc/format :markdown
       :const true}
  optional-delete-query-parameters [:df :analyzer :default_operator :consistency])

;;NB! requires delete-by-query plugin
(defn delete-by-query
  "Performs a delete-by-query operation over one or more indexes and types.

  Multiple indexes and types can be specified by passing in a seq of strings,
  otherwise specifying a string suffices."
  ([^Connection conn index mapping-type query]
     (rest/delete conn
                  (rest/delete-by-query-url
                    conn
                    (join-names index)
                    (join-names mapping-type))
                  {:body {:query query}}))
  ([^Connection conn index mapping-type query opts]
     (rest/delete conn
                  (rest/delete-by-query-url
                    conn
                    (join-names index)
                    (join-names mapping-type))
                  {:query-params (select-keys opts
                                              (conj optional-delete-query-parameters
                                                    :ignore_unavailable))
                   :body {:query query}})))

(defn delete-by-query-across-all-types
  "Performs a delete-by-query operation over one or more indexes, across all
  mapping types.

  Multiple indexes can be specified using a seq of strings, otherwise a single
  string suffices."
  ([^Connection conn index query]
     (rest/delete conn
                  (rest/delete-by-query-url conn (join-names index))
                  {:body {:query query}}))
  ([^Connection conn index query opts]
     (rest/delete conn
                  (rest/delete-by-query-url conn (join-names index))
                  {:query-params (select-keys opts
                                              (conj optional-delete-query-parameters
                                                    :ignore_unavailable))
                   :body {:query query}})))

(defn delete-by-query-across-all-indexes-and-types
  "Performs a delete-by-query operation across all indexes and mapping types.
  This may put very high load on your Elasticsearch cluster so use this function with care."
  ([^Connection conn query]
     (rest/delete conn (rest/delete-by-query-url conn) {:body {:query query}}))
  ([^Connection conn query opts]
     (rest/delete conn (rest/delete-by-query-url conn)
                  {:query-params (select-keys opts optional-delete-query-parameters)
                   :body {:query query}})))


(defn more-like-this
  "Performs a More Like This (MLT) query."
  ([^Connection conn index mapping-type] (more-like-this conn index mapping-type nil))
  ([^Connection conn index mapping-type opts]
   (rest/get conn
             (rest/more-like-this-url conn index mapping-type)
             {:body (json/encode {:query {:mlt opts}})})))

(defn validate-query
  "Validates a query without actually executing it. Has the same API as [[search]]
  but does not take the `mapping-type` parameter."
  ([^Connection conn index query] (validate-query conn index query nil))
  ([^Connection conn index query opts]
   (rest/get conn (rest/query-validation-url conn
                                             index)
             {:body (json/encode {:query query})
              :query-params opts})))


(defn analyze
  "Examples:

  ```clojure
  (require '[clojurewerkz.elastisch.rest.document :as doc])

  (doc/analyze conn \"foo bar baz\")
  (doc/analyze conn \"foo bar baz\" {:index \"some-index-name\"})
  (doc/analyze conn \"foo bar baz\" {:analyzer \"whitespace\"})
  (doc/analyze conn \"foo bar baz\" {:index \"some-index-name\" :field \"some-field-name\"})
  ```"
  ([^Connection conn text] (analyze conn text nil))
  ([^Connection conn text opts]
     (rest/get conn (rest/analyze-url conn
                                      (:index opts))
               {:query-params (assoc opts :text text)})))
