(ns tieapp
  (:use compojure.core)
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [tiesql.common :refer :all]
            [tiesql.tools.http-server :as s]
            [tiesql.tools.http-routes :as tr]
            [tiesql.tools.ds :as ds]
            [tiesql.tools.http-service :as hs]
            [tiesql.tools.c3p0 :as tcj]))


#_(defn init-system [e-file-name]
    (let [{:keys [file-name init-name port]} (hs/default-config-map e-file-name)
          ds (ds/create (tcj/new-c3p0datasource))
          tie (hs/read-file file-name ds init-name)
          tie-routes (tr/tie-routes tie ds)
          app-routes (tr/app-routes tie-routes)
          ]))


(defrecord DB [name]
  component/Lifecycle
  (start [component]
    (log/info "Starting database connection ...")
    (let [ds-v (ds/create (tcj/new-c3p0datasource))]
      (assoc component :ds ds-v)))
  (stop [component]
    (log/info "Starting database connection ...")
    (if (:ds component)
      (do
        (ds/close (:ds component))
        (dissoc component :ds))
      component)))


(defn new-db []
  (map->DB {:name ""}))


(defrecord Tie [file-name init-name db]
  component/Lifecycle
  (start [component]
    (log/info "Starting tie file ... ")
    (->> (hs/read-file file-name (:ds db) init-name)
         (assoc component :tms)))
  (stop [component]
    (dissoc component :tms nil)))


(defn new-tie [file-name init-name]
  (map->Tie (hash-map :file-name file-name
                      :init-name init-name)))


(defrecord TieRoutes [tie db]
  component/Lifecycle
  (start [component]
    (log/info "Opening tie-routes ...  ")
    (assoc component :tie-routes (tr/tie-routes (:tms tie) (:ds db))))
  (stop [component]
    (log/info "Closing tit routes ")
    (dissoc component :tie-routes)))


(defn new-tie-routes []
  (map->TieRoutes {}))

(defrecord AppRoutes [config tie-routes db]
  component/Lifecycle
  (start [component]
    (log/info "Opening app routes...... ")
    (->> (tr/app-routes (:tie-routes tie-routes))
         (assoc component :routes)))
  (stop [component]
    (log/info "Stoping app routes.......")
    (dissoc component :routes)))


(defn new-routes [config-map]
  (map->AppRoutes {:config config-map}))


(defn build-system [tie-app-file]
  (let [config-options (hs/default-config-map tie-app-file)
        {:keys [host port tiesql-file tiesql-init]} config-options]
    (-> (component/system-map
          :db (new-db)
          :tie (new-tie tiesql-file tiesql-init)
          :tie-routes (new-tie-routes)
          :routes (new-routes {})
          :server (s/new-server port))
        (component/system-using
          {:tie        {:db :db}
           :tie-routes {:db  :db
                        :tie :tie}
           :routes     {:db         :db
                        :tie-routes :tie-routes}
           :server     {:routes :routes}}))))

;(def ^:dynamic tms-file-name "tie.edn.sql")


(defn app-start [tie-app-file]
  (->> (build-system tie-app-file)
       (component/start)))