(ns tiesql.tools.sql-executor
  (:require [clojure.set]
            [clojure.core.async :as async :refer [<! >! <!! chan alt! go go-loop onto-chan sliding-buffer]]
            [clojure.java.jdbc :as jdbc]
            [clj-common :as cc]
            [tiesql.core-util :as cu]
            [tiesql.common :refer :all]))


(defn apply-handler
  [handler m]
  (-> (handler m)
      (async/<!!)))


(defn apply-handler-parallel
  [handler m-coll]
  (->> m-coll
       (map #(handler %))
       (doall)
       (async/merge)
       (async/take (count m-coll))
       (async/into [])
       (async/<!!)))


(defn jdbc-handler-single
  [ds tm]
  (let [dml-type (dml-type-key tm)
        sql (sql-key tm)
        result (result-key tm)]
    ;todo Need to move this log from here
    (debug (select-keys tm [sql-key dml-type]))
    (condp = dml-type
      dml-type-select-key
      (if (contains? result result-array-key)
        (jdbc/query ds sql :as-arrays? true :identifiers clojure.string/lower-case)
        (jdbc/query ds sql :as-arrays? false :identifiers clojure.string/lower-case))
      dml-type-insert-key
      (jdbc/execute! ds sql :multi? true)
      (jdbc/execute! ds sql))))


(defn warp-map-output
  [handler]
  (fn [m]
    (try
      (let [^:double st (System/nanoTime)
            r (handler m)
            total (/ (-' (System/nanoTime) st) 1000000.0)
            r (if (seq? r) (into [] r) r)]
        (if (cc/failed? r)
          r
          (assoc m output-key r
                   exec-time-key total
                   :start-time st)))
      ;;Need to catch execption as it is run inside async/thread
      (catch Exception e
        (-> (cc/fail {query-exception-key (.getMessage e)})
            (merge m))))))


(defn warp-async-go
  [handler]
  (fn [m]
    (async/go
      (let [t-v (or (timeout-key m) 2000)
            exec-ch (async/thread (handler m))
            [v rch] (async/alts! [exec-ch (async/timeout t-v)])]
        (if (= rch exec-ch)
          v
          ;; Need to assoc exception here as it returns from here
          (-> {query-exception-key "SQL Execution time out"
               timeout-key         t-v}
              (cc/fail)
              (merge m)))))))



(defn default-jdbc-handler [ds]
  (-> (partial jdbc-handler-single ds)
      (warp-map-output)
      (warp-async-go)))



(def Parallel :parallel)
(def Transaction :transaction)
(def Serial :serial)
(def SerialNotFailed :serial-not-failed)


(defn do-execute [type ds m-coll]
  (condp = type
    Parallel
    (-> (default-jdbc-handler ds)
        (apply-handler-parallel m-coll))
    Serial
    (-> (map #(apply-handler (default-jdbc-handler ds) %))
        (transduce conj m-coll))
    SerialNotFailed
    (-> (map #(apply-handler (default-jdbc-handler ds) %))
        (cc/comp-xf-until)
        (transduce conj m-coll))
    commit-all-key
    (do-execute SerialNotFailed ds m-coll)
    ;; default serial
    (do-execute Serial ds m-coll)))


(defmulti executor (fn [type _ _] type))


(defmethod executor
  :default
  [type ds _]
  (fn [tm-coll]
    (do-execute type ds tm-coll)))


(defn is-rollback?
  [commit-type read-only result-coll]
  (cond
    (= true read-only) true
    (= commit-type commit-none-key) true
    (and (= commit-type commit-all-key)
         (cc/failed? result-coll)) true
    :else false))


(defn has-transaction?
  [ds]
  (and (jdbc/db-find-connection ds)
       (< 0 (jdbc/get-level ds))))


(defn do-transaction-execute
  [tx-map ds m-coll]
  (let [isolation (or (:isolation tx-map) :serializable)
        read-only? (cu/read-only? tx-map)
        commit-type (cu/commit-type m-coll)]
    (jdbc/with-db-transaction
      [t-conn ds
       :isolation isolation
       :read-only? read-only?]
      (let [result-coll (do-execute commit-type t-conn m-coll)]
        (when (is-rollback? commit-type read-only? result-coll)
          (jdbc/db-set-rollback-only! t-conn))
        result-coll))))


(defmethod executor
  Transaction
  [_ ds tms]
  (let [tx-map (apply hash-map (get-in tms [global-key tx-prop]))]
    (fn [m-coll]
      (if (has-transaction? ds)
        (let [commit-type (cu/commit-type m-coll)]
          (do-execute commit-type ds m-coll))
        (do-transaction-execute tx-map ds m-coll)))))





