(ns self-kafka.consumer
  (:require [self-kafka.core :refer [to-clojure as-properties]]
            [manifold.stream :as s])
  (:import [kafka.consumer ConsumerConfig Consumer KafkaStream]
           [kafka.javaapi.consumer ConsumerConnector]))

(defn consumer
  "Uses information in Zookeeper to connect to Kafka. More info on settings
   is available here: https://kafka.apache.org/08/configuration.html
   Recommended for using with with-resource:
   (with-resource [c (consumer m)]
     shutdown
     (take 5 (messages c \"test\")))
   Keys:
   zookeeper.connect             : host:port for Zookeeper. e.g: 127.0.0.1:2181
   group.id                      : consumer group. e.g. group1
   auto.offset.reset             : what to do if an offset is out of range, e.g. smallest, largest
   auto.commit.interval.ms       : the frequency that the consumed offsets are committed to zookeeper.
   auto.commit.enable            : if set to true, the consumer periodically commits to zookeeper the latest consumed offset of each partition"
  [m]
  (let [config (ConsumerConfig. (as-properties m))]
    (Consumer/createJavaConsumerConnector config)))

(defn shutdown
  "Closes the connection to Zookeeper and stops consuming messages."
  [^ConsumerConnector consumer]
  (.shutdown consumer))

(defn messages
  "Creates a sequence of KafkaMessage messages from the given topic. Consumes
  messages from a single stream."
  [^ConsumerConnector consumer topic & {:keys [threads]
                                        :or   {threads 1}}]
  (let [[_topic [stream & _]]
        (first (.createMessageStreams consumer {topic (int threads)}))]
    (.iterator ^KafkaStream stream)))

(defn consumer-stream
  "takes a kafka consumer config map, a topic and and options map and returns a manifold stream which represents the
  topic.  When the stream is closed, the Kafka Consumer will be shut down."
  [consumer-config topic & {:keys [threads]
                            :or   {threads 1}}]
  (let [consumer (consumer consumer-config)
        stream (->> (messages consumer topic :threads threads)
                    s/->source
                    (s/map to-clojure))]
    (s/on-drained stream (fn [] (shutdown consumer)))
    stream))
