(ns material-ui.core
  "DOM element constructors for React. Mirrors om.dom namespace"
  (:refer-clojure :exclude [map meta time])
        
                                              
  (:require
   [clojure.string :as str]
   om.dom
         cljs.core)
       
  (:import
   [cljs.tagged_literals JSValue]))

     
(defn kebab-case
  "Converts CamelCase / camelCase to kebab-case"
  [s]
  (str/join "-" (map str/lower-case (re-seq #"\w[a-z]+" s))))

     
(def material-tags
  '[AppBar
  AppCanvas
  Circle
  Checkbox
  DatePicker
  DialogWindow
  Dialog
  DropDownIcon
  DropDownMenu
  EnhancedButton
  FlatButton
  FloatingActionButton
  FocusRipple
  IconButton
  Icon
  InkBar
  Input
  LeftNav
  MenuItem
  Menu
  Overlay
  Paper
  RadioButton
  RaisedButton
  Slider
  SlideIn
  Snackbar
  Tab
  TabTemplate
  Tabs
  TableHeader
  TableRowsItem
  TableRows
  Toggle
  ToolbarGroup
  Toolbar
  Tooltip
  TouchRipple])

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Private

     
(defn clj->js [v]
  (JSValue. v))

     
(defn literal?
  "Returns true if form is a literal value (number, string, map, etc),
  otherwise false."
  [form]
  (not (or (symbol? form)
           (list? form))))

(defn camel-case
  "Converts kebab-case to camelCase"
  [s]
  (str/replace s #"-(\w)" (comp str/upper-case second)))

(defn- opt-key-case
  "Converts attributes that are kebab-case and should be camelCase"
  [attr]
  (if (or (< (count attr) 5)
          (case (subs attr 0 5) ("data-" "aria-") true false))
    attr
    (camel-case attr)))

(defn- opt-key-alias
  "Converts aliased attributes"
  [opt]
  (case opt
    :class :className
    :for :htmlFor
    opt))

(defn format-opt-key
  "Returns potentially formatted name for DOM element attribute.
   Converts kebab-case to camelCase."
  [opt-key]
  (-> opt-key
      opt-key-alias
      name
      opt-key-case
      keyword))

(declare format-opts)

(defn format-opt-val
  "Returns potentially modified value for DOM element attribute.
   Recursively formats map values (ie :style attribute)"
  [opt-val]
  (cond
   (map? opt-val)
   (format-opts opt-val)

        
   (not (literal? opt-val))
        
   `(format-opts ~opt-val)

   :else
   opt-val))

(defn format-opts
  "Returns JavaScript object for React DOM attributes from opts map"
  [opts]
  (if (map? opts)
    (->> opts
         (clojure.core/map
          (fn [[k v]] [(format-opt-key k) (format-opt-val v)]))
         (into {})
         clj->js)
    opts))

(defn ^boolean possible-coll? [form]
  (or (coll? form)
      (symbol? form)
      (list? form)))

      
                                 
                                                   
                                     
      

(defn js-opts? [x]
       
  (instance? JSValue x)
        
                                            )

(defn element-args
  "Returns a vector of [opts children] for from first and second
  argument given to DOM function"
  [opts children]
  (cond
   (nil? opts) [nil children]
   (map? opts) [(format-opts opts) children]
   (js-opts? opts) [opts children]
   :else [nil (cons opts children)]))

      
                                  
                                                     
                                                 

     
(defn ^:private gen-bootstrap-inline-fn [tag]
  `(defmacro ~(symbol (kebab-case (str tag)))
     [opts# & children#]
     (let [ctor# '~(symbol "js" (str "MaterialUI." (name tag)))]
       (if (literal? opts#)
         (let [[opts# children#] (element-args opts# children#)]
           (cond
             (every? (complement possible-coll?) children#)
            `(~ctor# ~opts# ~@children#)

            (and (= (count children#) 1) (vector? (first children#)))
            `(~ctor# ~opts# ~@(-> children# first flatten))

            :else
            `(apply ~ctor# ~opts# (flatten (vector ~@children#)))))
         `(mui/element ~ctor# ~opts# (vector ~@children#))))))

(defmacro ^:private gen-bootstrap-inline-fns []
  `(do ~@(clojure.core/map gen-bootstrap-inline-fn material-tags)))

     
(gen-bootstrap-inline-fns)

;;;;;;;;;;;; This file autogenerated from src/material_ui/core.cljx
