(ns tiesql.jdbc
  (:require
    [clojure.tools.logging :as log]
    [clojure.java.jdbc :as jdbc]
    [clj-common :as cc]
    [tiesql.common :refer :all]
    [tiesql.core :as tie]
    [tiesql.core-util :as cu]
    [tiesql.impl.param-impl :as p]
    [tiesql.tools.file-reader :as fr]
    [tiesql.tools.sql-executor :as ce]))


(defn read-file
  ([file-name] (read-file file-name (tie/new-process-context-impl)))
  ([file-name pc]
   (-> (fr/read-file file-name (tie/comp-child-key pc))
       (assoc-in [global-key file-name-key] file-name)
       (assoc-in [global-key process-context-key] (tie/dissoc-k-schema pc)))))


(defn select-name
  [tms name-seq]
  (tie/select-name tms name-seq))


(defn get-dml
  [tms]
  (let [p (comp (filter cu/has-dml-type?)
                (map sql-key)
                (filter (fn [v] (if (< 1 (count v))
                                  true false)))
                (map first)
                (map #(clojure.string/replace % #":\w+" "?")))]
    (into [] p (vals tms))))

;(first (vals {:1 2}))



(defn pull-sequence!
  [tms ds & {:keys [name params step]
             :or   {params {}
                    step   {}}}]
  (-> (merge-with merge (cu/assoc-single-key) (get-in tms [global-key process-context-key]) step)
      (tie/assoc-executor (ce/executor ce/Parallel ds tms))
      (tie/comp-child-key true)
      (tie/fetch-data tms name params)
      (cu/map-single-key-result)))


(defn warp-pull-sequence!
  [tms ds step]
  (fn [name param]
    (pull-sequence! tms ds
                    :name name
                    :params param
                    :step step)))


(defn get-process
  [tms ds step-m e-type]
  (let [gen-fn (or (get step-m :generator)
                   (warp-pull-sequence! tms ds step-m)
                   )]
    (-> (merge-with merge (get-in tms [global-key process-context-key]) step-m)
        (tie/assoc-executor (ce/executor e-type ds tms))
        (p/assoc-params-ref-gen-key gen-fn)
        (tie/comp-child-key true))))


(defn pull-one!
  "Read or query value from database. It will return as model map "
  [tms ds & {:keys [name params step]
             :or   {params {}
                    step   {}}}]
  (if (not (keyword? name))
    (cc/fail "Name must be a keyword")
    (-> (get-process tms ds step ce/Parallel)
        (tie/fetch-data tms name params)
        (cu/map-result))))


(defn pull-batch!
  "Read or query value from database. It will return as model map "
  [tms ds & {:keys [name params step]
             :or   {params {}
                    step   {}}}]
  (-> (get-process tms ds step ce/Parallel)
      (tie/fetch-data tms name params)))


(defn push-one!
  "Write single map in databse. It will run only one DB O/P"
  [tms ds & {:keys [name params step]}]
  (if (not (keyword? name))
    (cc/fail "Name must be a keyword")
    (let [p (-> (get-process tms ds step ce/Transaction)
                (dissoc column-key result-key))]
      (cc/try-> tms
                (select-name name)
                (tie/run-process p params)
                (cu/map-result)))))


(defn push-batch!
  "Create, update or delete value in database. DB O/P will be run within transaction. "
  [tms ds & {:keys [name params step]
             :or   {step {}}}]
  (let [p (-> (get-process tms ds step ce/Transaction)
              (dissoc column-key result-key))]
    (cc/try-> tms
              (select-name name)
              (tie/run-model-process p params))))


(defn validate-dml!
  [ds str-coll]
  (jdbc/with-db-connection
    [conn ds]
    (doseq [str str-coll]
      (jdbc/prepare-statement (:connection conn) str)))
  (log/info (format "checking %d dml statement is done " (count str-coll))))



(defn db-do
  "call db do commands "
  [tms ds name-coll]
  (try
    (let [tm-coll (vals (tie/select-name tms name-coll))]
      (doseq [m tm-coll]
        (when-let [sql (get-in m [sql-key])]
          (debug "db do with " sql)
          (jdbc/db-do-commands ds false sql))))
    (catch Exception e
      (log/error e)
      (log/error (.getMessage e)))))
