(ns tiesql.impl.validation-impl
  (:require [schema.core :as s]
            [schema.macros :as sm]
            [schema.utils :as su]
            [clj-common :as cc]
            [tiesql.common :refer :all]
            [tiesql.impl.util :as pu]
            [tiesql.compiler.schema :as sch]))



(defn resolve-type [w]
  (let [t (type w)]
    (if (= clojure.lang.Symbol t)
      (resolve w)
      t)))


(def resolve-type? (s/pred resolve-type 'resolve-type))



(defn new-vali-type [name index]
  {k-validate (fn [w]
                (->
                  [(s/one s/Keyword "Source Data Model")
                   (s/one s/Keyword "Type of validation ")
                   (s/one resolve-type? "Clojure or Java type")
                   (s/one s/Str "fail message")]
                  (s/validate w)
                  ))
   k-emission (fn [ks] (do
                         (-> ks
                             (assoc 2 (resolve-type (nth ks 2)))
                             (update-in [0] cc/as-lower-case-keyword))))
   k-name     name
   k-order    index
   k-process  (fn [result]
                (let [[p-value _ v-type e-message] result]
                  (if (= v-type (type p-value))
                    result
                    (cc/fail {:msg   e-message
                              :value p-value
                              :type  (str (type p-value))}))))
   })




(defn new-vali-contain [kname index]
  (let [v [(s/one s/Keyword "Source Data Model")
           (s/one s/Keyword "Type of validation ")
           (s/one resolve-type? "Clojure or Java type")
           (s/one s/Str "fail message")]]
    {k-validate (fn [w] (s/validate v w))
     k-emission (fn [ks] (-> ks
                             (assoc 2 (resolve-type (nth ks 2)))
                             (update-in [0] cc/as-lower-case-keyword)))
     k-name     kname
     k-order    index
     k-process  (fn [result]
                  (let [[p-value _ v-type e-message] result
                        r (mapv #(= v-type (type %)) p-value)
                        r (every? true? r)]
                    (if r
                      result
                      (cc/fail e-message))))
     }))



(defn new-vali-range [name index]
  (let [v [(s/one s/Keyword "Source Data Model")
           (s/one s/Keyword "Type of validation ")
           (s/one s/Int "Min range value")
           (s/one s/Int "Max range value")
           (s/one s/Str "fail message")]]
    {k-validate (fn [w] (s/validate v w))
     k-emission (fn [w] (-> w
                            (update-in [0] cc/as-lower-case-keyword)))
     k-name     name
     k-order    index
     k-process  (fn [result]
                  (let [[p-value _ min max e-message] result]
                    (if (and (>= p-value min)
                             (<= p-value max))
                      result
                      (cc/fail e-message))))
     }))



(defn name-equal?
  [[_ t]]
  t)



(defn get-all-vali-key
  [m]
  (let [sql-key (sql-key m)
        validation-key (validation-key m)
        input-key (cc/as-sequential (input-key m))
        sql-p (into #{} (rest sql-key))
        v-map (group-by first validation-key)]
    (for [p input-key
          p1 (keys (select-keys p sql-p))
          v1 (p1 v-map)]
      (assoc v1 0 (p1 p)))))


(defn validation-process
  [child-proc-map m]
  (->> (get-all-vali-key m)
       (sort-by #(k-order ((name-equal? %) child-proc-map)))
       (reduce (fn [acc ks]
                 (let [cp (get-in child-proc-map [(name-equal? ks) k-process])
                       w (cp ks)]
                   (if (cc/failed? w)
                     (reduced w)
                     acc))
                 ) m)))


(defn validation-compile
  [childs vali-seq sql-params-seq]
  (let [sql-params-seq (rest sql-params-seq)]
    (->> vali-seq
         (cc/distinct-with-range 2)
         (filter (fn [[f]]
                   (contains? (into #{} sql-params-seq) f)
                   ))
         (into [])
         (pu/proc-compile name-equal? childs))))


(defn validation-schema [kname child-proc-map acc]
  (let [v (s/pred (partial pu/proc-schema-validate name-equal? child-proc-map) 'k-schema-validate)]
    (-> acc
        (sch/assoc-schema-key (s/optional-key kname) v)
        (sch/assoc-extend-schema-key (s/optional-key kname) v))))


(defn new-validation-key-process
  [u-type proc-map]
  (let [kname (k-name proc-map)
        child-proc-map (childs proc-map)]
    (if u-type
      (-> proc-map
          (dissoc childs)
          (assoc k-process (map (partial validation-process child-proc-map))))
      (-> proc-map
          (dissoc childs)
          (assoc k-validate (partial validation-schema kname child-proc-map))
          (assoc k-emission (partial validation-compile child-proc-map))))))


(defn validation-key-child
  []
  {vali-type    (new-vali-type vali-type 0)
   vali-range   (new-vali-range vali-range 3)
   vali-contain (new-vali-contain vali-contain 1)})


(defn new-validation-key
  [c index]
  {k-order        index
   k-name         validation-key
   k-process-type input-key
   childs         c})


(defn comp-validation-child
  [p-context atype]
  (if-let [v (validation-key p-context)]
    (->> (new-validation-key-process atype v)
         (assoc p-context validation-key))
    p-context))

