(ns tiesql.system
  (:use compojure.core)
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [tiesql.common :refer :all]
            [tiesql.tools.ds :as ds]
            [tiesql.tools.c3p0 :as tcj]
            [tiesql.tools.http-routes :as tr]
            [tiesql.tools.jdbc-http-adapter :as hs]))


#_(defn init-system [e-file-name]
    (let [{:keys [file-name init-name port]} (hs/default-config-map e-file-name)
          ds (ds/create (tcj/new-c3p0datasource))
          tie (hs/read-file file-name ds init-name)
          tie-routes (tr/tie-routes tie ds)
          app-routes (tr/default-app-routes tie-routes)
          ]))

(defrecord Config [name]
  component/Lifecycle
  (start [component]
    (log/info "Read config file" name)
    (assoc component :config (hs/default-config-map name)))
  (stop [component]
    (dissoc component :config)))


(def file-name "tiesql.edn")

(defn new-config
  ([] (new-config file-name))
  ([f]
   (map->Config {:name f})))


(defrecord DB [config]
  component/Lifecycle
  (start [component]
    (log/info "Starting database connection ...")
    (let [ds-v (ds/create (tcj/new-c3p0datasource))]
      (assoc component :ds ds-v)))
  (stop [component]
    (log/info "Starting database connection ...")
    (if (:ds component)
      (do
        (ds/close (:ds component))
        (dissoc component :ds))
      component)))


(defn new-db []
  (map->DB {}))


(defrecord TieRoutes [config db]
  component/Lifecycle
  (start [component]
    (log/info "Opening app routes...... ")
    (let [{:keys [tiesql-file tiesql-init]} (:config config)]
      (->> (hs/read-file tiesql-file (:ds db) tiesql-init)
           (tr/tie-routes (:ds db))
           (assoc component :routes))))
  (stop [component]
    (log/info "Stoping app routes.......")
    (dissoc component :routes)))


(defn new-routes []
  (map->TieRoutes {}))


(defn tie-system
  [file-name]
  (-> (component/system-map
        :config (new-config file-name)
        :db (new-db)
        :routes (new-routes))
      (component/system-using
        {:db     {:config :config}
         :routes {:config :config
                  :db     :db}})))


(defn tie-routes
  [sys]
  (get-in sys [:routes :routes]))


(defn tie-app-routes
  [sys tie-path]
  (-> (tie-routes sys)
      (tr/default-app-routes tie-path )))
