(ns tieapp
  (:use compojure.core)
  (:require [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [immutant.web :as im]
            [tiesql.common :refer :all]
            [tiesql.tools.ds :as ds]
            [tiesql.tools.c3p0 :as tcj]
            [tiesql.tools.http-routes :as tr]
            [tiesql.tools.http-service :as hs]))


#_(defn init-system [e-file-name]
    (let [{:keys [file-name init-name port]} (hs/default-config-map e-file-name)
          ds (ds/create (tcj/new-c3p0datasource))
          tie (hs/read-file file-name ds init-name)
          tie-routes (tr/tie-routes tie ds)
          app-routes (tr/app-routes tie-routes)
          ]))


(defrecord Server [config routes]
  component/Lifecycle
  (start [component]
    (log/info "Starting server  ")
    (let [{:keys [port]} (:config config)
          server (im/run (:routes routes) {:port port})]
      (log/info "Starting server done  with port....... " port)
      (assoc component :server server)))
  (stop [component]
    (log/info "Stoping server.........")
    (let [server (:server component)]
      (if server
        (do
          (im/stop server)
          (assoc component :server nil))
        component))))


(defn new-server []
  (map->Server {}))


(defrecord Config [name]
  component/Lifecycle
  (start [component]
    (log/info "Read config file" name)
    (assoc component :config (hs/default-config-map name)))
  (stop [component]
    (dissoc component :config)))


(def file-name "tie.app.edn")

(defn new-config
  ([] (new-config file-name))
  ([f]
   (map->Config {:name f})))


(defrecord DB [config]
  component/Lifecycle
  (start [component]
    (log/info "Starting database connection ...")
    (let [ds-v (ds/create (tcj/new-c3p0datasource))]
      (assoc component :ds ds-v)))
  (stop [component]
    (log/info "Starting database connection ...")
    (if (:ds component)
      (do
        (ds/close (:ds component))
        (dissoc component :ds))
      component)))


(defn new-db []
  (map->DB {}))


(defrecord AppRoutes [config db]
  component/Lifecycle
  (start [component]
    (log/info "Opening app routes...... ")
    (let [{:keys [tiesql-file tiesql-init]} (:config config)]
      (->> (hs/read-file tiesql-file (:ds db) tiesql-init)
           (tr/tie-routes (:ds db))
           (tr/app-routes)
           (assoc component :routes))))
  (stop [component]
    (log/info "Stoping app routes.......")
    (dissoc component :routes)))


(defn new-routes []
  (map->AppRoutes {}))


(defn build-system
  ([] (build-system nil))
  ([tie-app-file]
   (-> (component/system-map
         :config (new-config tie-app-file)
         :db (new-db)
         :routes (new-routes)
         :server (new-server))
       (component/system-using
         {:db     {:config :config}
          :routes {:config :config
                   :db     :db}
          :server {:config :config
                   :routes :routes}}))))


(defn app-start [tie-app-file]
  (->> (build-system tie-app-file)
       (component/start)))