(ns leiningen.resolve-java-sources-and-javadocs
  (:require
   [leiningen.core.main]
   [cemerick.pomegranate.aether])
  (:import
   (java.io File)))

(defn safe-sort [coll]
  (try
    (->> coll
         (sort (fn [x y]
                 (try
                   (compare x y)
                   (catch Exception e
                     (leiningen.core.main/warn (pr-str [::could-not-sort x y]))
                     (when (System/getProperty "leiningen.resolve-java-sources-and-javadocs.throw")
                       (throw e))
                     0)))))
    (catch Exception e
      (leiningen.core.main/warn (pr-str [::could-not-sort coll]))
      (when (System/getProperty "leiningen.resolve-java-sources-and-javadocs.throw")
        (throw e))
      coll)))

(defn ensure-no-lists [x]
  {:pre [(vector? x)]}
  (->> x (mapv (fn [y]
                 (let [v (cond-> y
                           (sequential? y) vec)]
                   (cond-> v
                     (vector? v) ensure-no-lists))))))

(defn serialize [x]
  {:pre  [(not (string? x))]
   :post [(not (string? %))]}
  (->> x
       (mapv (fn outer [[k v]]
               [(ensure-no-lists k)
                (some->> v
                         (mapv (fn inner [[kk vv]]
                                 [(ensure-no-lists kk) (some->> vv
                                                                (map ensure-no-lists)
                                                                safe-sort
                                                                vec)]))
                         (safe-sort)
                         vec)]))
       safe-sort
       vec))

(defn deserialize [x]
  {:pre  [(not (string? x))]
   :post [(not (string? %))]}
  (->> x
       (map (fn [[k v]]
              [k (if (and v (empty? v))
                   []
                   (some->> v
                            (map (fn [[kk vv]]
                                   [kk (some->> vv
                                                set)]))
                            (into {})))]))
       (into {})))

(defn add [{:keys                 [repositories managed-dependencies]
            {:keys [classifiers]} :resolve-java-sources-and-javadocs
            :or                   {classifiers #{"javadoc" "sources"}}
            :as                   project}]
  (let [classifiers (set classifiers)
        repositories (->> repositories
                          (remove (fn [x]
                                    (.contains (pr-str x) "password")))
                          (into {}))
        cache-filename (-> "user.home" System/getProperty (File. ".lein-source-and-javadocs-cache") str)
        initial-cache-value (or (when (-> cache-filename java.io.File. .exists)
                                  (->> cache-filename slurp read-string deserialize))
                                {})
        cache-atom (atom initial-cache-value)]
    (update project
            :dependencies
            (fn [deps]
              (let [resolve! (memoize (fn resolve! [x]
                                        (let [v (or (get @cache-atom x)
                                                    (try
                                                      (let [v (cemerick.pomegranate.aether/resolve-dependencies :coordinates x
                                                                                                                :repositories repositories)
                                                            [x] x]
                                                        (when (and (find v x)
                                                                   (-> x (get 3) classifiers))
                                                          (leiningen.core.main/info (str ::found " " (pr-str x))))
                                                        ;; ensure the cache gets set to something:
                                                        (doto v assert))
                                                      (catch Exception e
                                                        (if (#{(Class/forName "org.eclipse.aether.resolution.DependencyResolutionException")
                                                               (Class/forName "org.eclipse.aether.transfer.ArtifactNotFoundException")
                                                               (Class/forName "org.eclipse.aether.resolution.ArtifactResolutionException")}
                                                             (class e))
                                                          []
                                                          (do
                                                            (-> e .printStackTrace)
                                                            nil)))))]
                                          (when v
                                            (swap! cache-atom assoc x v))
                                          v)))
                    flatten-deps (fn flatten-deps [xs]
                                   (->> xs
                                        (mapcat (fn [[k v]]
                                                  (apply list k v)))))
                    add (->> deps
                             (mapcat (fn [[dep version & args :as original]]
                                       (let [version (or version
                                                         (->> managed-dependencies
                                                              (filter (fn [[a]]
                                                                        (= dep a)))
                                                              first
                                                              second))]
                                         (if-not version ;; handles managed-dependencies
                                           [original]
                                           (let [transitive (->> (resolve! [(assoc-in original [1] version)])
                                                                 flatten-deps)]
                                             (->> transitive
                                                  (mapcat (fn [[dep version :as original]]
                                                            (assert version (pr-str original))
                                                            (->> classifiers
                                                                 (mapv (partial vector dep version :classifier))
                                                                 (into [original])
                                                                 (distinct)
                                                                 (remove (comp nil? second))
                                                                 (filter (fn [x]
                                                                           (->> (resolve! [x])
                                                                                flatten-deps
                                                                                (filter #{x})
                                                                                first
                                                                                some?))))))
                                                  (distinct)
                                                  (vec)))))))
                             (distinct)
                             (filter (fn [[_ _ _ x]]
                                       (classifiers x)))
                             (filter (fn remove-non-matching-versions [[s-or-j-name s-or-j-version :as s-or-j]]
                                       (let [[_ matching-version :as matching] (->> deps
                                                                                    (filter (fn [[matching-name]]
                                                                                              (= matching-name
                                                                                                 s-or-j-name)))
                                                                                    first)]
                                         (if matching
                                           (= s-or-j-version matching-version)
                                           s-or-j))))
                             (group-by (fn [[dep version _ classifier]]
                                         [dep classifier]))
                             (vals)
                             ;; avoid :pedantic faults by picking one source:
                             (map (fn [equivalent-deps]
                                    (let [pred (fn [xs [inner-dep inner-version]]
                                                 {:pre [inner-dep inner-version]}
                                                 (->> xs
                                                      (some (fn [[dep version]]
                                                              {:pre [dep]}
                                                              (and (= dep inner-dep)
                                                                   (= version inner-version))))))]
                                      (or (->> equivalent-deps
                                               (filter (partial pred deps))
                                               (first))
                                          (->> equivalent-deps
                                               (filter (partial pred managed-dependencies))
                                               (first))
                                          (->> equivalent-deps
                                               (sort-by second)
                                               (last))))))
                             (mapv (fn [[d :as x]]
                                     (conj x :exclusions '[[*]]))))]
                (when-not (= initial-cache-value @cache-atom)
                  (->> @cache-atom
                       serialize
                       pr-str
                       (spit cache-filename)))
                ;; order can be sensitive
                (into add deps))))))

(defn resolve-java-sources-and-javadocs
  [project & args]
  (add project))
