(ns leiningen.resolve-java-sources-and-javadocs
  (:require
   [leiningen.core.main]
   [cemerick.pomegranate.aether])
  (:import
   (java.io File)))

;; XXX things to be fetched (javadoc/sources) should be configurable

(defn add [{:keys [repositories managed-dependencies] :as project}]
  (let [repositories (->> repositories
                          (remove (fn [x]
                                    (.contains (pr-str x) "password")))
                          (into {}))
        cache-filename (-> "user.dir" System/getProperty (File. ".lein-source-and-javadocs-cache") str)
        ;; XXX ensure it has a canonical sort order. serde
        initial-cache-value (or (when (-> cache-filename java.io.File. .exists)
                                  (-> cache-filename slurp read-string))
                                {})
        cache-atom (atom initial-cache-value)]
    (update project
            :dependencies
            (fn [deps]
              (let [resolve! (memoize (fn resolve! [x]
                                        (let [v (or (get @cache-atom x)
                                                    (try
                                                      (let [v (cemerick.pomegranate.aether/resolve-dependencies :coordinates x
                                                                                                                :repositories repositories)
                                                            [x] x]
                                                        (when (and (find v x)
                                                                   (-> x (get 3) #{"sources" "javadoc"}))
                                                          (leiningen.core.main/info (str ::found " " (pr-str x))))
                                                        ;; ensure the cache gets set to something:
                                                        (doto v assert))
                                                      (catch Exception e
                                                        (if (#{(Class/forName "org.eclipse.aether.resolution.DependencyResolutionException")
                                                               (Class/forName "org.eclipse.aether.transfer.ArtifactNotFoundException")
                                                               (Class/forName "org.eclipse.aether.resolution.ArtifactResolutionException")}
                                                             (class e))
                                                          []
                                                          (do
                                                            (-> e .printStackTrace)
                                                            nil)))))]
                                          (when v
                                            (swap! cache-atom assoc x v))
                                          v)))
                    flatten-deps (fn flatten-deps [xs]
                                   (->> xs
                                        (mapcat (fn [[k v]]
                                                  (apply list k v)))))
                    add (->> deps
                             (mapcat (fn [[dep version & args :as original]]
                                       (if ('#{org.clojure/clojure} dep)
                                         [original]
                                         (let [version (or version
                                                           (->> managed-dependencies
                                                                (filter (fn [[a]]
                                                                          (= dep a)))
                                                                first
                                                                second))]
                                           (if-not version ;; some minor case related to managed-dependencies
                                             [original]
                                             (let [transitive (->> (resolve! [(assoc-in original [1] version)])
                                                                   flatten-deps)]
                                               (->> transitive
                                                    (mapcat (fn [[dep version :as original]]
                                                              (assert version (pr-str original))
                                                              (->> [original
                                                                    [dep version :classifier "sources"]
                                                                    #_ [dep version :classifier "javadoc"]]
                                                                   (distinct)
                                                                   (remove (comp nil? second))
                                                                   (filter (fn [x]
                                                                             (->> (resolve! [x])
                                                                                  flatten-deps
                                                                                  (filter #{x})
                                                                                  first
                                                                                  some?))))))
                                                    (distinct)
                                                    (vec))))))))
                             (distinct)
                             (filter (fn [[_ _ _ x]]
                                       (#{"sources" #_ "javadoc"} x)))
                             #_ (remove (fn [[n _ _ c]]
                                          (or
                                           ;; these tend to be problematic, triggering various compilation errors.
                                           (#{"sources"} c))))
                             (filterv (fn remove-non-matching-versions [[s-or-j-name s-or-j-version :as s-or-j]]
                                        (let [[_ matching-version :as matching] (->> deps
                                                                                     (filter (fn [[matching-name]]
                                                                                               (= matching-name
                                                                                                  s-or-j-name)))
                                                                                     first)]
                                          (if matching
                                            (= s-or-j-version matching-version)
                                            s-or-j))))
                             (group-by (fn [[dep version _ classifier]]
                                         [dep classifier]))
                             (vals)
                             ;; avoid :pedantic faults by picking one source:
                             (map (fn [equivalent-deps]
                                    (->> equivalent-deps
                                         (sort-by second)
                                         ;; XXX prefer, if present, one that matches an entry in `:dependencies` (or :m-deps)
                                         (last))))
                             (mapv (fn [[d :as x]]
                                     (conj x :exclusions ['*]))))]
                (when-not (= initial-cache-value @cache-atom)
                  (spit cache-filename (pr-str @cache-atom)))
                ;; order can be sensitive
                (into add deps))))))

(defn resolve-java-sources-and-javadocs
  [project & args]
  (add project))
