(ns cisco.tools.namespace.parallel-refresh
  (:refer-clojure :exclude [time])
  (:require
   [clojure.java.classpath :as classpath]
   [clojure.java.io :as io]
   [clojure.spec.alpha :as spec]
   [clojure.string :as string]
   [clojure.tools.namespace.dependency]
   [clojure.tools.namespace.dir :as dir]
   [clojure.tools.namespace.find :as find]
   [clojure.tools.namespace.parse :as parse]
   [clojure.tools.namespace.reload :as reload]
   [clojure.tools.namespace.repl :as repl]
   [clojure.tools.namespace.track :as track]
   [clojure.tools.reader.reader-types :refer [indexing-push-back-reader push-back-reader]]
   [loom.alg]
   [loom.graph]
   [nedap.speced.def :as speced]
   [nedap.utils.collections.eager :refer [partitioning-pmap]])
  (:import
   (java.io File)
   (java.util.concurrent LinkedBlockingQueue ThreadPoolExecutor TimeUnit)
   (java.util.concurrent.locks ReentrantLock)))

;; XXX I think tool.n has a bug where it uses topo-sort, only to ruin it. investigate
;; while we're here, impl some ideas from https://github.com/reducecombine/rehardened
;; XXX rename to rehardened.tools.namespace (ns) + tools.namespace.rehardened (proj/repo)

(defn islands [graph]
  (loom.alg/connected-components (loom.graph/digraph graph)))

(speced/defn read-ns-decl
  "Reads ns declaration in file with line/column metadata"
  [^string? filename]
  (when-not (-> filename File. .isDirectory)
    (try
      (with-open [reader (-> filename io/reader push-back-reader indexing-push-back-reader)]
        (parse/read-ns-decl reader))
      (catch Exception e
        (if (-> e ex-data :type #{:reader-exception})
          nil
          (throw e))))))

(spec/def ::ns-form (spec/and sequential?
                              (comp #{'ns `ns} first)))

(speced/defn ^::resource-path ns-decl->resource-path [^::ns-form ns-decl, extension]
  (-> ns-decl
      second
      str
      munge
      (string/replace "." "/")
      (str extension)))

(spec/def ::resource-path (spec/and string?
                                    (complement string/blank?)
                                    (complement #{\. \! \? \-})
                                    (fn [x]
                                      (re-find #"\.clj([cs])?$" x))))

(speced/defn resource-path->filenames [^::resource-path resource-path]
  (->> (-> (Thread/currentThread)
           (.getContextClassLoader)
           (.getResources resource-path))
       (enumeration-seq)
       (distinct) ;; just in case
       (mapv str)))

;; XXX memoize jars
(speced/defn ^{::speced/spec (spec/coll-of ::ns-form
                                           :min-count 1)}
  find-ns-decls []
  (->> (classpath/system-classpath)
       find/find-ns-decls
       (filter identity)))

(spec/def ::classpath-ns-forms (spec/coll-of ::ns-form
                                             :kind set?
                                             :min-count 1))

;; NOTE: this used to be limited to the `refresh-dirs`, but that proved to be unsafe
;; e.g. many tests namespaces deemed as leaves would depend on e.g. matcher-combinators, which would then be
;; concurrently `require`d.
(speced/defn ^::classpath-ns-forms classpath-ns-forms
  "Returns all the ns-forms contained or required in the entire classpath."
  []
  (->> (find-ns-decls)
       (set)))

(defn files-and-deps [ns-forms read-opts]
  (->> ns-forms
       (reduce (speced/fn [m ^::ns-form decl]
                 (let [deps (parse/deps-from-ns-decl decl)
                       name (parse/name-from-ns-decl decl)]
                   (-> m
                       (assoc-in [:depmap name] (into (empty deps)
                                                      ;; remove self-requires (occasionally found in the wild)
                                                      (remove #{name}
                                                              deps))))))
               {})))

(defn project-graph [^::classpath-ns-forms base]
  (let [{:keys [depmap]} (files-and-deps base nil)]
    (#'clojure.tools.namespace.track/update-deps (clojure.tools.namespace.dependency/graph)
                                                 depmap)))

(def print-lock (Object.))

(defn debug [& xs]
  (when (#{"true" "1" "yes" "y" "t" "debug"}
         (some-> "cisco.tools.namespace.parallel-refresh.debug"
                 System/getProperty
                 string/lower-case))
    (locking print-lock
      (apply println xs))))

;; XXX simplify
(defn positions [pred coll]
  (keep-indexed (fn [idx x]
                  (when (pred x)
                    idx))
                coll))

(defn elem-index [item xs]
  (->> xs (positions #{item}) first))

(defn sort-list-by-list [crit xs]
  (->> xs
       (sort (fn [a b]
               (< (or (elem-index a crit) Long/MAX_VALUE)
                  (or (elem-index b crit) Long/MAX_VALUE))))))

(defn executor []
  (let [c (-> (Runtime/getRuntime) .availableProcessors)]
    ;; important - given its a fixed size pool, the queue must be unbounded, so that no work gets rejected
    (ThreadPoolExecutor. c c 60 TimeUnit/SECONDS (LinkedBlockingQueue.))))

(def ^:dynamic ^ThreadPoolExecutor *executor* nil)

(spec/def ::error-result
  (spec/tuple simple-symbol? (partial instance? Throwable)))

(spec/def ::result
  (spec/or ::ok           #{::ok}
           ::error-result ::error-result))

(defn in-bg [promises-atom f]
  {:pre [*executor*]}
  (let [p (promise)]
    (swap! promises-atom conj p)
    (send-via *executor* (agent nil) (fn [_]
                                       (try
                                         (speced/let [^::result v (f)]
                                           (deliver p v))
                                         (catch Throwable e
                                           (deliver p ['unknown-ns e])))))))

(defmacro time
  "Prints over `#'print-lock`."
  [expr]
  `(let [start# (. System (nanoTime))
         ret# ~expr]
     (locking print-lock
       (prn (str "Elapsed time: " (/ (double (- (. System (nanoTime)) start#)) 1000000.0) " msecs")))
     ret#))

(spec/def ::workload (spec/coll-of symbol?))

(speced/defn process-leaf [idx, ^simple-symbol? leave, promises, leaves-promises]
  (let [leave-promise (get leaves-promises idx)]
    (assert leave-promise)
    (swap! promises conj leave-promise)
    (in-bg promises (fn []
                      (debug "Reloading leaf namespace in"
                             (-> (Thread/currentThread)
                                 .getName)
                             "-"
                             leave)
                      (deliver leave-promise
                               (try
                                 (require :reload leave)
                                 ::ok
                                 (catch Throwable e
                                   (locking print-lock
                                     (-> e .printStackTrace))
                                   [leave e])))
                      ::ok))))

(spec/def ::project-namespaces (spec/coll-of simple-symbol? :kind set?))

(speced/defn ^::project-namespaces project-namespaces
  "Returns all the namespaces contained or required in the current project.

  Excludes third-party dependencies."
  []
  (->> (#'dir/find-files repl/refresh-dirs find/clj)
       (partitioning-pmap (speced/fn [^File file]
                            (let [decl (-> file str read-ns-decl)
                                  n (some-> decl parse/name-from-ns-decl)]
                              [n])))
       (apply concat)
       (distinct)
       (filter identity)
       (set)))

(speced/defn process-island [^deref promises
                             ^set? workload-set
                             ^::project-namespaces the-project-namespaces
                             ^set? leaves
                             ^::workload workload
                             island]
  (in-bg promises (fn []
                    (let [corpus (some->> island
                                          (filter workload-set)
                                          (seq) ;; shortcircuit a later costlier filtering
                                          (filter the-project-namespaces)
                                          (seq)
                                          (group-by (partial contains? leaves)))]
                      (if-not corpus
                        ::ok
                        (let [{island-nonleaves false, island-leaves true} corpus]
                          (when (seq island-nonleaves)
                            (debug "Reloading"
                                   (count island-nonleaves)
                                   "non-leaf namespaces in"
                                   (-> (Thread/currentThread) .getName (str ":\n  "))
                                   (str "["
                                        (->> island-nonleaves
                                             (string/join "\n    "))
                                        "]")))
                          (speced/let [will-req (some->> island-nonleaves
                                                         (sort-list-by-list workload)
                                                         (seq))
                                       ^::result succ (try
                                                        (some->> will-req
                                                                 (apply require :reload))
                                                        ::ok
                                                        (catch Throwable e
                                                          (locking print-lock
                                                            (-> e .printStackTrace))
                                                          [(first will-req) e]))]
                            (if-not (#{::ok} succ)
                              succ
                              (let [leaves-promises (mapv (fn [_]
                                                            (promise))
                                                          island-leaves)]
                                (doseq [p leaves-promises]
                                  (swap! promises conj p))
                                (->> island-leaves
                                     (sort-list-by-list workload)
                                     (map-indexed (fn [i l]
                                                    (process-leaf i
                                                                  l
                                                                  promises
                                                                  leaves-promises)))
                                     (doall))
                                succ)))))))))

(speced/defn perform-refresh [^::workload workload]
  (let [the-classpath-ns-forms (classpath-ns-forms)
        the-classpath-ns-names (->> the-classpath-ns-forms
                                    (map parse/name-from-ns-decl)
                                    set)
        the-project-namespaces (project-namespaces)
        workload-set (set workload)
        {:keys [dependencies dependents]} (project-graph the-classpath-ns-forms)
        leaves (->> dependencies
                    (keys)
                    (remove (partial find dependents))
                    (set))
        leaves-3rd-party-dependencies (->> leaves
                                           (filter the-project-namespaces)
                                           (mapcat (speced/fn [^simple-symbol? n]
                                                     (speced/let [^::speced/nilable ^set? d (dependencies n)]
                                                       (if-not d
                                                         []
                                                         (remove the-project-namespaces d)))))
                                           (set)
                                           ;; filter out e.g. leiningen.* namespaces
                                           (filter the-classpath-ns-names)
                                           (seq))
        promises (atom [])]

    ;; These need to be `require`d before anything else, and sequentially, for safety.
    ;; There's no associated `println` because normally these will be actually loaded just once
    ;; during the entire JVM lifecycle:
    (some->> leaves-3rd-party-dependencies (apply require))

    (binding [*executor* (executor)]
      (->> dependencies
           (islands)
           (filterv (partial some workload-set))
           (run! (partial process-island
                          promises
                          workload-set
                          the-project-namespaces
                          leaves
                          workload)))
      (Thread/sleep 150) ;; ensure there are promises
      (while (or (-> *executor* .getActiveCount pos?)
                 (some (fn [p]
                         (= ::timeout
                            (deref p 50 ::timeout)))
                       @promises))
        (Thread/sleep 50))
      (-> *executor* .shutdown)
      (->> @promises
           (map deref)
           (remove #{::ok})
           first))))

;; XXX move to different ns
(defn track-reload-one
  [tracker]
  (let [{unload ::track/unload, load ::track/load} tracker]
    (cond
      (seq unload)
      (let [n (first unload)]
        (reload/remove-lib n)
        (update-in tracker [::track/unload] rest))

      (seq load)
      (try
        (speced/let [^::speced/nilable ^::error-result v (perform-refresh load)
                     [maybe-failed-ns ex] v]
          (cond-> tracker
            true            (update-in [::track/load] (constantly ()))
            maybe-failed-ns (assoc ::reload/error-ns maybe-failed-ns)
            ex              (assoc ::reload/error ex)))
        (catch Throwable t
          (assoc tracker
                 ::reload/error    t
                 ::reload/error-ns (first load)
                 ::track/unload    load)))

      :else
      tracker)))

(defn track-reload
  [tracker]
  (loop [tracker (dissoc tracker ::reload/error ::reload/error-ns)]
    (let [{error ::reload/error, unload ::track/unload, load ::track/load} tracker]
      (if (and (not error)
               (or (seq load) (seq unload)))
        (recur (track-reload-one tracker))
        tracker))))

(defn referred
  "Plays better with vars coming from Potemkin.

  For an explanation see https://github.com/jonase/eastwood/issues/307#issuecomment-764304379"
  [ns]
  (reduce (fn [m [sym var-ref]]
            (let [ns-sym (-> var-ref symbol namespace symbol)
                  var-sym (-> var-ref symbol name symbol)]
              (assoc-in m [ns-sym var-sym] sym)))
          {}
          (ns-refers ns)))

(defn do-refresh [scan-opts after-sym]
  (when after-sym
    (assert (symbol? after-sym) ":after value must be a symbol")
    (assert (namespace after-sym)
            ":after value must be a namespace-qualified symbol"))
  (let [current-ns-name (ns-name *ns*)
        current-ns-refers (referred *ns*)
        current-ns-aliases (#'repl/aliased *ns*)]
    (alter-var-root #'repl/refresh-tracker dir/scan-dirs repl/refresh-dirs scan-opts)
    (alter-var-root #'repl/refresh-tracker #'repl/remove-disabled)
    ;; avoid a deadlock by using two steps:
    (let [v (track-reload repl/refresh-tracker)]
      (alter-var-root #'repl/refresh-tracker (constantly v)))
    (in-ns current-ns-name)
    (let [result (#'repl/print-and-return repl/refresh-tracker)]
      (if (= :ok result)
        (if after-sym
          (if-let [after (ns-resolve *ns* after-sym)]
            (after)
            (throw (Exception.
                    (str "Cannot resolve :after symbol " after-sym))))
          result)
        ;; There was an error, recover as much as we can:
        (do (when-not (or (false? (::repl/unload (meta *ns*)))
                          (false? (::repl/load (meta *ns*))))
              (#'repl/recover-ns current-ns-refers current-ns-aliases))
            ;; Return the Exception to the REPL:
            result)))))

(def ^ReentrantLock refresh-lock
  "Prevents concurrent invocations from e.g. an IDE + a terminal."
  (ReentrantLock.))

(defn refresh [& options]
  (try
    (if-not (-> refresh-lock (.tryLock 60 TimeUnit/SECONDS))
      (locking print-lock
        (println "Couldn't refresh because some other thread is already performing one."))
      (let [{:keys [after]} options]
        (do-refresh {:platform find/clj} after)))
    (finally
      (try
        (-> refresh-lock .unlock)
        (catch IllegalMonitorStateException _)))))

(defn gc-later
  "An optional helper, not directly used by this lib.

  You can set it as an `:after` option, which can improve performance in some cases."
  []
  (future
    (System/gc)))
