(ns cisco.tools.namespace.parallel-refresh
  (:refer-clojure :exclude [time])
  (:require
   [clojure.java.classpath :as classpath]
   [clojure.java.io :as io]
   [clojure.spec.alpha :as spec]
   [clojure.string :as string]
   [clojure.tools.namespace.dependency]
   [clojure.tools.namespace.dir :as dir]
   [clojure.tools.namespace.find :as find]
   [clojure.tools.namespace.parse :as parse]
   [clojure.tools.namespace.reload :as reload]
   [clojure.tools.namespace.repl :as repl]
   [clojure.tools.namespace.track :as track]
   [formatting-stack.linters.one-resource-per-ns]
   [formatting-stack.util :refer [read-ns-decl]]
   [loom.alg]
   [loom.graph]
   [nedap.speced.def :as speced]
   [nedap.utils.collections.eager :refer [partitioning-pmap]])
  (:import
   (java.io File)
   (java.util.concurrent LinkedBlockingQueue ThreadPoolExecutor TimeUnit)))

;; XXX I think tool.n has a bug where it uses topo-sort, only to ruin it. investigate
;; while we're here, impl some ideas from https://github.com/reducecombine/rehardened
;; XXX rename to rehardened.tools.namespace (ns) + tools.namespace.rehardened (proj/repo)

(defn islands [graph]
  (loom.alg/connected-components (loom.graph/digraph graph)))

;; XXX vendor
(defn ns-sym->filename* [ns-sym]
  (some-> (list 'ns ns-sym)
          (formatting-stack.linters.one-resource-per-ns/ns-decl->resource-path ".clj")
          formatting-stack.linters.one-resource-per-ns/resource-path->filenames
          (first)
          (clojure.string/replace "file:" "")))

;; This is safe to memoize - there aren't that many different files acoss workloads during a given JVM's lifetime
(def ns-sym->filename (memoize ns-sym->filename*))

(defn find-files [dirs platform]
  (->> dirs
       (map io/file)
       (map (speced/fn [^File f]
              (-> f .getCanonicalFile)))
       (filter (speced/fn [^File f]
                 (-> f .exists)))
       (mapcat (speced/fn [^File f]
                 (-> f (find/find-sources-in-dir platform))))
       (map (speced/fn [^File f]
              (-> f .getCanonicalFile)))))

(speced/defn ^{::speced/spec (spec/coll-of any? :min-count 1)} classpath-directories
  "A replacement for `#'classpath/classpath-directories` with which external tooling cannot interfere.

  See: https://github.com/clojure-emacs/cider-nrepl/pull/668"
  []
  (->> (classpath/system-classpath)
       (filter (speced/fn [^File f]
                 (-> f .isDirectory)))))

(speced/defn ^set? project-namespaces
  "Returns all the namespaces contained or required in the current project.

  Excludes third-party dependencies."
  []
  (->> (find-files repl/refresh-dirs ;; important - filtering by `repl/refresh-dirs` prevents checkouts from fattening islands
                   find/clj)
       (partitioning-pmap (speced/fn [^File file]
                            (let [decl (-> file str read-ns-decl)
                                  n (some-> decl parse/name-from-ns-decl)]
                              [n])))
       (apply concat)
       (distinct)
       (filter identity)
       (set)))

(defn project-graph [base]
  (let [all (->> base
                 (partitioning-pmap (comp ns-sym->filename symbol str))
                 (remove (fn [s]
                           (or (nil? s)
                               (string/starts-with? s "jar:")))))
        {:keys [depmap]} (#'clojure.tools.namespace.file/files-and-deps all nil)]
    (#'clojure.tools.namespace.track/update-deps (clojure.tools.namespace.dependency/graph)
                                                 depmap)))

(def print-lock (Object.))

(defn debug [& xs]
  (when (contains? #{nil "" "false"}
                   (System/getProperty "cisco.tools.namespace.parallel-refresh.debug"))
    (locking print-lock
      (apply println xs))))

;; XXX simplify
(defn positions [pred coll]
  (keep-indexed (fn [idx x]
                  (when (pred x)
                    idx))
                coll))

(defn elem-index [item xs]
  (->> xs (positions #{item}) first))

(defn sort-list-by-list [crit xs]
  (->> xs
       (sort (fn [a b]
               (< (or (elem-index a crit) Long/MAX_VALUE)
                  (or (elem-index b crit) Long/MAX_VALUE))))))

(defn executor []
  (let [c (-> (Runtime/getRuntime) .availableProcessors)]
    ;; important - given its a fixed size pool, the queue must be unbounded, so that no work gets rejected
    (ThreadPoolExecutor. c c 60 TimeUnit/SECONDS (LinkedBlockingQueue.))))

(def ^:dynamic ^ThreadPoolExecutor *executor* nil)

(spec/def ::error-result
  (spec/tuple simple-symbol? (partial instance? Throwable)))

(spec/def ::result
  (spec/or ::ok           #{::ok}
           ::error-result ::error-result))

(defn in-bg [promises-atom f]
  {:pre [*executor*]}
  (let [p (promise)]
    (swap! promises-atom conj p)
    (send-via *executor* (agent nil) (fn [_]
                                       (speced/let [^::result v (f)]
                                         (deliver p v))))))

(defmacro time
  "Prints over `#'print-lock`."
  [expr]
  `(let [start# (. System (nanoTime))
         ret# ~expr]
     (locking print-lock
       (prn (str "Elapsed time: " (/ (double (- (. System (nanoTime)) start#)) 1000000.0) " msecs")))
     ret#))

(spec/def ::workload (spec/coll-of symbol?))

(defn process-leaf [idx leave promises leaves-promises]
  (let [leave-promise (get leaves-promises idx)]
    (assert leave-promise)
    (swap! promises conj leave-promise)
    (in-bg promises (fn []
                      (debug "Reloading leaf namespace in"
                             (-> (Thread/currentThread)
                                 .getName)
                             "-"
                             leave)
                      (deliver leave-promise
                               (try
                                 (require :reload leave)
                                 ::ok
                                 (catch Throwable e
                                   (-> e .printStackTrace)
                                   [leave e])))
                      ::ok))))

(speced/defn process-island [^deref promises
                             ^set? workload-set
                             ^set? the-project-namespaces
                             ^set? leaves
                             ^::workload workload
                             island]
  (in-bg promises (fn []
                    (let [corpus (some->> island
                                          (filter workload-set)
                                          (seq) ;; shortcircuit a later costlier filtering
                                          (filter the-project-namespaces)
                                          (seq)
                                          (group-by (partial contains? leaves)))]
                      (if-not corpus
                        ::ok
                        (let [{island-nonleaves false, island-leaves true} corpus]
                          (when (seq island-nonleaves)
                            (debug "Reloading"
                                   (count island-nonleaves)
                                   "non-leaf namespaces in"
                                   (-> (Thread/currentThread) .getName (str ":\n  "))
                                   (str "["
                                        (->> island-nonleaves
                                             (string/join "\n    "))
                                        "]")))
                          (speced/let [will-req (some->> island-nonleaves
                                                         (sort-list-by-list workload)
                                                         (seq))
                                       ^::result succ (try
                                                        (some->> will-req
                                                                 (apply require :reload))
                                                        ::ok
                                                        (catch Exception e
                                                          (-> e .printStackTrace)
                                                          [(first will-req) e]))]
                            (if-not (#{::ok} succ)
                              succ
                              (let [leaves-promises (mapv (fn [_]
                                                            (promise))
                                                          island-leaves)]
                                (doseq [p leaves-promises]
                                  (swap! promises conj p))
                                (->> island-leaves
                                     (sort-list-by-list workload)
                                     (map-indexed (fn [i l]
                                                    (process-leaf i
                                                                  l
                                                                  promises
                                                                  leaves-promises)))
                                     (doall))
                                succ)))))))))

(speced/defn perform-refresh [^::workload workload]
  (let [the-project-namespaces (project-namespaces)
        workload-set (set workload)
        {:keys [dependencies dependents]} (project-graph the-project-namespaces)
        leaves (->> dependencies
                    (keys)
                    (remove (partial find dependents))
                    (set))
        promises (atom [])]
    (binding [*executor* (executor)]
      (->> dependencies
           (map (fn [[k v]]
                  [k
                   ;; important - remove irrelevant libspecs for a fine-grained `island`-ing
                   (filterv the-project-namespaces v)]))
           (into {})
           (islands)
           (filterv (partial some workload-set))
           (run! (partial process-island
                          promises
                          workload-set
                          the-project-namespaces
                          leaves
                          workload)))
      (Thread/sleep 150) ;; ensure there are promises
      (while (or (-> *executor* .getActiveCount pos?)
                 (some (fn [p]
                         (= ::timeout
                            (deref p 50 ::timeout)))
                       @promises))
        (Thread/sleep 50))
      (-> *executor* .shutdown)
      (->> @promises
           (map deref)
           (remove #{::ok})
           first))))

;; XXX move to different ns
(defn track-reload-one
  [tracker]
  (let [{unload ::track/unload, load ::track/load} tracker]
    (cond
      (seq unload)
      (let [n (first unload)]
        (reload/remove-lib n)
        (update-in tracker [::track/unload] rest))

      (seq load)
      (try
        (speced/let [^::speced/nilable ^::error-result v (perform-refresh load)
                     [maybe-failed-ns ex] v]
          (cond-> tracker
            true            (update-in [::track/load] (constantly ()))
            maybe-failed-ns (assoc ::reload/error-ns maybe-failed-ns)
            ex              (assoc ::reload/error ex)))
        (catch Throwable t
          (assoc tracker
                 ::reload/error    t
                 ::reload/error-ns (first load)
                 ::track/unload    load)))

      :else
      tracker)))

(defn track-reload
  [tracker]
  (loop [tracker (dissoc tracker ::reload/error ::reload/error-ns)]
    (let [{error ::reload/error, unload ::track/unload, load ::track/load} tracker]
      (if (and (not error)
               (or (seq load) (seq unload)))
        (recur (track-reload-one tracker))
        tracker))))

(defn do-refresh [scan-opts after-sym]
  (when after-sym
    (assert (symbol? after-sym) ":after value must be a symbol")
    (assert (namespace after-sym)
            ":after value must be a namespace-qualified symbol"))
  (let [current-ns-name (ns-name *ns*)
        current-ns-refers (#'repl/referred *ns*)
        current-ns-aliases (#'repl/aliased *ns*)]
    (alter-var-root #'repl/refresh-tracker dir/scan-dirs repl/refresh-dirs scan-opts)
    (alter-var-root #'repl/refresh-tracker #'repl/remove-disabled)
    (#'repl/print-pending-reloads repl/refresh-tracker)
    ;; avoid a deadlock by using two steps:
    (let [v (track-reload repl/refresh-tracker)]
      (alter-var-root #'repl/refresh-tracker (constantly v)))
    (in-ns current-ns-name)
    (let [result (#'repl/print-and-return repl/refresh-tracker)]
      (if (= :ok result)
        (if after-sym
          (if-let [after (ns-resolve *ns* after-sym)]
            (after)
            (throw (Exception.
                    (str "Cannot resolve :after symbol " after-sym))))
          result)
        ;; There was an error, recover as much as we can:
        (do (when-not (or (false? (::repl/unload (meta *ns*)))
                          (false? (::repl/load (meta *ns*))))
              (#'repl/recover-ns current-ns-refers current-ns-aliases))
            ;; Return the Exception to the REPL:
            result)))))

(def refresh-lock
  "Prevents concurrent invocations from e.g. an IDE + a terminal."
  (Object.))

(defn refresh
  [& options]
  (locking refresh-lock
    (let [{:keys [after]} options]
      (do-refresh {:platform find/clj} after))))

(defn gc-later
  "An optional helper, not directly used by this lib.

  You can set it as an `:after` option, which can improve performance in some cases."
  []
  (future
    (System/gc)))
