(ns flanders.spec
  (:require #?(:clj  [clojure.core.match :refer [match]]
               :cljs [cljs.core.match :refer-macros [match]])
            [clojure.future :refer :all] ;; Remove after CLJ 1.9
            [clojure.spec :as s]
            #?(:clj  [flanders.types]
               :cljs [flanders.types
                      :refer [AnythingType BooleanType EitherType InstType
                              IntegerType KeywordType MapEntry MapType
                              NumberType SequenceOfType SetOfType StringType]]))
  #?(:clj (:import [flanders.types
                    AnythingType BooleanType EitherType InstType IntegerType
                    KeywordType MapEntry MapType NumberType SequenceOfType
                    SetOfType StringType])))

(defprotocol SpecedNode
  (->spec' [node ns f]))

(defn speced-node? [node]
  (satisfies? SpecedNode node))

(defn ns-str? [s]
  (and (string? s)
       (re-matches #"\w+(\.?\w+)*" s)))

(defn ->spec [node ns]
  (->spec' node ns ->spec))

(s/fdef ->spec
        :args (s/cat :node speced-node? :ns ns-str?)
        :ret any?)

(defn- key? [spec]
  (and (set? spec)
       (= 1 (count spec))
       (keyword? (first spec))))

(extend-protocol SpecedNode

  ;; Branches

  EitherType
  (->spec' [{:keys [choices]} ns f]
    (letfn [(f' [ent] (f ent ns))]
      (let [specs (map f' choices)
            keys (for [i (range (count specs))]
                   (keyword (str "choice" i)))]
        (eval `(s/or ~@(interleave keys specs))))))

  MapEntry
  (->spec' [{:keys [key type]} ns f]
    (let [key-spec (f key ns)]
      (when (key? key-spec)
        (let [key-name (-> key-spec first name)
              type-ns (str ns "." key-name)
              type-spec (f type type-ns)
              result-kw (keyword ns key-name)]
          (eval `(s/def ~result-kw ~type-spec))
          result-kw))))

  MapType
  (->spec' [{:keys [entries]} ns f]
    (letfn [(f' [ent] (f ent ns))]
      (let [[req-ents opt-ents] (split-with :required? entries)
            req-specs (->> (mapv f' req-ents)
                           (filterv keyword?))
            opt-specs (->> (mapv f' opt-ents)
                           (filterv keyword?))]
        (eval `(s/keys :req-un ~req-specs
                       :opt-un ~req-specs)))))

  SequenceOfType
  (->spec' [{:keys [type]} ns f]
    (eval `(s/coll-of ~(f type ns))))

  SetOfType
  (->spec' [{:keys [type]} ns f]
    (eval `(s/coll-of ~(f type ns) :kind set?)))

  ;; Leaves

  AnythingType
  (->spec' [{:keys [spec]} _ _]
    (or spec
        any?))

  BooleanType
  (->spec' [{:keys [open? spec default]} _ _]
    (match [(some? spec) open? default]
           [true _   _  ] spec
           [_   true _  ] boolean?
           [_   _    nil] boolean?
           :else          #{default}))

  InstType
  (->spec' [{:keys [spec]} _ _]
    (or spec
        inst?))

  IntegerType
  (->spec' [{:keys [open? spec values]} _ _]
    (match [(some? spec) open? values]
           [true _    _  ] spec
           [_    true _  ] integer?
           [_    _    nil] integer?
           :else           (set values)))

  KeywordType
  (->spec' [{:keys [open? spec values]} _ _]
    (match [(some? spec) open? values]
           [true _     _  ] spec
           [_    true  _  ] keyword?
           [_    _     nil] keyword?
           :else            (set values)))

  NumberType
  (->spec' [{:keys [open? spec values]} _ _]
    (match [(some? spec) open? (seq values)]
           [true _    _  ] spec
           [_    true _  ] number?
           [_    _    nil] number?
           :else           (set values)))

  StringType
  (->spec' [{:keys [open? spec values]} _ _]
    (match [(some? spec) open? (seq values)]
           [true  _   _  ] spec
           [_    true _  ] string?
           [_    _    nil] string?
           :else           (set values))))
