(ns ctim.schemas.vulnerability
  (:require #?(:clj  [flanders.core :as f :refer [def-entity-type def-map-type def-eq]]
               :cljs [flanders.core :as f :refer-macros [def-entity-type def-map-type def-eq]])
            #?(:clj [ctim.lib.generators :as gen])
            [ctim.schemas.common :as c]
            [ctim.schemas.vocabularies :as v]))

(def type-identifier "vulnerability")

(def-eq VulnerabilityTypeIdentifier type-identifier
  :description "The fixed value \"vulnerability\"")

(def vulnerability-desc
  (str "a mistake in software that can be directly "
       "used by a hacker to gain access to a system or network"))

(def vulnerability-desc-link
  "[Vulnerability](http://docs.oasis-open.org/cti/stix/v2.0/cs01/part2-stix-objects/stix-v2.0-cs01-part2-stix-objects.html#_Toc496714334)")

(defn valid-score?
  "check that a score is above 0 and below 10"
  [score]
  (< 0 score 10))

(def Score
  (f/num
   :description "a Score number from 0 to 10"
   :spec valid-score?
   :float? true
   #?(:clj :gen)
   #?(:clj gen/score)))

(def cvss-v3-vector-string-exp
  (re-pattern "^CVSS:3.0/((AV:[NALP]|AC:[LH]|PR:[UNLH]|UI:[NR]|S:[UC]|[CIA]:[NLH]|E:[XUPFH]|RL:[XOTWU]|RC:[XURC]|[CIA]R:[XLMH]|MAV:[XNALP]|MAC:[XLH]|MPR:[XUNLH]|MUI:[XNR]|MS:[XUC]|M[CIA]:[XNLH])/)*(AV:[NALP]|AC:[LH]|PR:[UNLH]|UI:[NR]|S:[UC]|[CIA]:[NLH]|E:[XUPFH]|RL:[XOTWU]|RC:[XURC]|[CIA]R:[XLMH]|MAV:[XNALP]|MAC:[XLH]|MPR:[XUNLH]|MUI:[XNR]|MS:[XUC]|M[CIA]:[XNLH])$"))


(comment
  ;; base, temporal, environmental vectors
  ;; for use in building the regexp expressions in the vulnerability
  :reference "https://www.first.org/cvss/v2/guide#2-4-Base-Temporal-Environmental-Vectors"
  "Base   AV:[L,A,N]/AC:[H,M,L]/Au:[M,S,N]/C:[N,P,C]/I:[N,P,C]/A:[N,P,C]
   Temporal   E:[U,POC,F,H,ND]/RL:[OF,TF,W,U,ND]/RC:[UC,UR,C,ND]
   Environmental   CDP:[N,L,LM,MH,H,ND]/TD:[N,L,M,H,ND]/CR:[L,M,H,ND]/IR:[L,M,H,ND]/AR:[L,M,H,ND]")


;; example: (AV:N/AC:M/Au:N/C:C/I:C/A:C)
(def cvss-v2-vector-string-exp
  #"^\(AV:[LAN]/AC:[HML]/Au:[NSM]/C:[NPC]/I:[NPC]/A:[NPC]\)$")

;; example: (E:U/RL:OF/RC:C)
(def cvss-v2-temporal-vector-string-exp
  #"^\(E:(U|POC|F|H|ND)/RL:(OF|TF|W|U|ND)/RC:(UC|UR|C|ND)\)$")

;; example: (CDP:N/TD:L/CR:L/IR:ND/AR:ND)
(def cvss-v2-environmental-vector-string-exp
  #"^\(CDP:(N|L|LM|MH|H|ND)/TD:(N|L|M|H|ND)/CR:(L|M|H|ND)/IR:(L|M|H|ND)/AR:(L|M|H|ND)\)$")

(defn cvss-v2-vector-string?
  "validate a v2 vector string using a regexp"
  [s]
  (re-matches cvss-v2-vector-string-exp s))

(defn cvss-v2-temporal-vector-string?
  "validate a v2 temporal vector string using a regexp"
  [s]
  (re-matches cvss-v2-temporal-vector-string-exp s))

(defn cvss-v2-environmental-vector-string?
  "validate a v2 environmental vector string using a regexp"
  [s]
  (re-matches cvss-v2-environmental-vector-string-exp s))

(defn cvss-v3-vector-string?
  "validate a v3 vector string using a regexp"
  [s]
  (re-matches cvss-v3-vector-string-exp s))

(def CVSSv2VectorString
  (f/str
   :description
   (str "a text representation of a set of CVSSv2 metrics."
        "It is commonly used to record or transfer "
        "CVSSv2 metric information in a concise form")
   :spec cvss-v2-vector-string?
   #?(:clj :gen)
   #?(:clj gen/cvss-v2-vector-string)))

(def CVSSv2TemporalVectorString
  (f/str
   :description
   (str "A text representation of a set of CVSSv2 temporal metrics."
        "Temporal metrics allow analysists to calculate threat severity "
        "based on temporal factors such as reliability of vulnerability "
        "reports, availability of mitigations, and the ease or difficulty "
        "of conducting the exploit. "
        "It is commonly used to record or transfer "
        "CVSSv2 metric information in a concise form")
   :spec cvss-v2-temporal-vector-string?
   #?(:clj :gen)
   #?(:clj gen/cvss-v2-temporal-vector-string)))

(def CVSSv2EnvironmentalVectorString
  (f/str
   :description
   (str "A text representation of a set of CVSSv2 environmental metrics. "
        "Environmental metrics allow analysists to calculate threat "
        "scores in relation to environmental security requirements, "
        "collateral damage potential, and target availability. "
        "It is commonly used to record or transfer "
        "CVSSv2 metric information in a concise form")
   :spec cvss-v2-environmental-vector-string?
   #?(:clj :gen)
   #?(:clj gen/cvss-v2-environmental-vector-string)))


(def CVSSv3VectorString
  (f/str
   :description
   (str "a text representation of a set of CVSSv3 metrics."
        "It is commonly used to record or transfer "
        "CVSSv3 metric information in a concise form")
   :spec cvss-v3-vector-string?
   #?(:clj :gen)
   #?(:clj gen/cvss-v3-vector-string)))

(def-map-type CVSSv2
  (concat
   (f/required-entries
    (f/entry :vector_string CVSSv2VectorString)
    (f/entry :base_score Score)
    (f/entry :base_severity v/HighMedLow))
   (f/optional-entries
    (f/entry :access_vector v/CVSSv2AccessVector)
    (f/entry :access_complexity v/CVSSv2AccessComplexity)
    (f/entry :authentication v/CVSSv2Authentication)
    (f/entry :confidentiality_impact v/CVSSv2ConfidentialityImpact)
    (f/entry :integrity_impact v/CVSSv2IntegrityImpact)
    (f/entry :availability_impact v/CVSSv2AvailabilityImpact)

    ;; CVSS v2.0 Temporal Factors
    (f/entry :exploitability v/CVSSv2Exploitability)
    (f/entry :remediation_level v/CVSSv2RemediationLevel)
    (f/entry :report_confidence v/CVSSv2ReportConfidence)
    (f/entry :temporal_vector_string CVSSv2TemporalVectorString)

    ;; CVSS v2.0 Environmental Factors
    (f/entry :collateral_damage_potential v/CVSSv2CollateralDamagePotential)
    (f/entry :target_distribution v/CVSSv2TargetDistribution)
    (f/entry :confidentiality_requirement v/CVSSv2SecurityRequirement)
    (f/entry :integrity_requirement v/CVSSv2SecurityRequirement)
    (f/entry :availability_requirement v/CVSSv2SecurityRequirement)
    (f/entry :environmental_vector_string CVSSv2EnvironmentalVectorString)

    ;; CVSS v2.0 metadata flags added in NVD 0.1_beta - 2017-07-05
    (f/entry :obtain_all_privilege f/any-bool)
    (f/entry :obtain_user_privilege f/any-bool)
    (f/entry :obtain_other_privilege f/any-bool)
    (f/entry :user_interaction_required f/any-bool))))

(def-map-type CVSSv3
  (concat
   (f/required-entries
    (f/entry :vector_string CVSSv3VectorString)
    (f/entry :base_score Score)
    (f/entry :base_severity v/CVSSv3Severity))
   (f/optional-entries
    (f/entry :attack_vector v/CVSSv3AttackVector
             :description (str "Reflects the context by which "
                               "vulnerability exploitation is possible"))
    (f/entry :modified_attack_vector v/CVSSv3ModifiedAttackVector
             :description "modified attack vector")

    (f/entry :attack_complexity v/CVSSv3AttackComplexity
             :description (str "describes the conditions beyond the attacker's "
                               "control that must exist in order to exploit the "
                               "vulnerability"))
    (f/entry :modified_attack_complexity v/CVSSv3ModifiedAttackComplexity
             :description "modified attack complexity")

    (f/entry :privileges_required v/CVSSv3PrivilegesRequired
             :description (str "describes the level of privileges an attacker "
                               "must possess before successfully exploiting the vulnerability"))
    (f/entry :modified_privileges_required v/CVSSv3ModifiedPrivilegesRequired
             :description "modified privileges required")

    (f/entry :user_interaction v/CVSSv3UserInteraction
             :description (str "captures the requirement for a user, "
                               "other than the attacker, to participate in "
                               "the successful compromise of the vulnerable component"))
    (f/entry :modified_user_interaction v/CVSSv3ModifiedUserInteraction
             :description "modified user interaction")

    (f/entry :scope v/CVSSv3Scope
             :description (str "the ability for a vulnerability in one "
                               "software component to impact resources beyond "
                               "its means, or privileges"))
    (f/entry :modified_scope v/CVSSv3ModifiedScope
             :description "modified scope")

    (f/entry :confidentiality_impact v/CVSSv3ConfidentialityImpact
             :description (str "measures the impact to the confidentiality of"
                               "the information resources managed by a software "
                               "component due to a successfully exploited vulnerability"))

    (f/entry :modified_confidentiality_impact v/CVSSv3ModifiedConfidentialityImpact
             :description "modified confidentiality impact")

    (f/entry :integrity_impact v/CVSSv3IntegrityImpact
             :description (str "measures the impact to integrity "
                               "of a successfully exploited vulnerability"))
    (f/entry :modified_integrity_impact v/CVSSv3ModifiedIntegrityImpact
             :description (str "modified integrity impact"))

    (f/entry :availability_impact v/CVSSv3AvailabilityImpact
             :description (str "measures the impact to the availability "
                               "of the impacted component resulting from a successfully"
                               "exploited vulnerability"))
    (f/entry :modified_availability_impact v/CVSSv3ModifiedAvailabilityImpact
             :description "modified availability impact")

    (f/entry :exploit_code_maturity v/CVSSv3ExploitCodeMaturity
             :description "measures the likelihood of the vulnerability being attacked")

    (f/entry :remediation_level v/CVSSv3RemediationLevel
             :description (str "Remediation Level of a vulnerability "
                               "is an important factor for prioritization"))
    (f/entry :report_confidence v/CVSSv3ReportConfidence
             :description (str "measures the degree of confidence in the existence "
                               "of the vulnerability and the credibility of the "
                               "known technical details"))
    (f/entry :temporal_score Score
             :description (str "Round up(CVSSv3BaseScore × "
                               "CVSSv3ExploitCodeMaturity × "
                               "CVSSv3RemediationLevel × "
                               "CVSSv3ReportConfidence)"))
    (f/entry :temporal_severity Score
             :description "temporal severity")

    ;; requirements
    (f/entry :availability_requirement v/CVSSv3SecurityRequirements)
    (f/entry :confidentiality_requirement v/CVSSv3SecurityRequirements)
    (f/entry :integrity_requirement v/CVSSv3SecurityRequirements)

    ;; optional scores
    (f/entry :environmental_score Score)
    (f/entry :environmental_severity v/CVSSv3Severity))))

(def-map-type VulnerabilityImpact
  (f/optional-entries
   (f/entry :cvss_v3 CVSSv3)
   (f/entry :cvss_v2 CVSSv2)
   (f/entry :impact_score Score)
   (f/entry :exploitability_score Score)))

;; CVE Metadata (ID Assigner), description, references, v3 impact data published_date, lastmodified_date

(def-map-type CVEDataMeta
  (f/optional-entries
   (f/entry :id c/ShortString)
   (f/entry :assigner c/ShortString)))

(def-map-type CVE
  (f/required-entries
   (f/entry :cve_data_meta CVEDataMeta)))

(def-entity-type Vulnerability
  {:description vulnerability-desc
   :reference vulnerability-desc-link}
  c/base-entity-entries
  c/describable-entity-entries
  c/sourcable-object-entries
  (f/required-entries
   (f/entry :description c/Markdown
            :description
            (str "A description that provides more details and context about the Vulnerability,"
                 "potentially including its purpose and its key characteristics."))
   (f/entry :type VulnerabilityTypeIdentifier
            :description (str "The fixed value " type-identifier)))
  (f/optional-entries
   (f/entry :cve CVE)
   (f/entry :impact VulnerabilityImpact)
   (f/entry :published_date c/Time)
   (f/entry :last_modified_date c/Time)))

(def-entity-type NewVulnerability
  "For submitting a new vulnerability"
  (:entries Vulnerability)
  c/base-new-entity-entries
  (f/optional-entries
   (f/entry :type VulnerabilityTypeIdentifier)))

(def-entity-type StoredVulnerability
  "A vulnerability as stored in the data store"
  (:entries Vulnerability)
  c/base-stored-entity-entries)

(def VulnerabilityRef
  (c/ref-for-type type-identifier))
