(ns thi.ng.geom.attribs
  #?(:cljs (:require-macros [thi.ng.math.macros :as mm]))
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.utils :as gu]
   [thi.ng.geom.vector :as v :refer [vec2 vec3]]
   [thi.ng.geom.matrix :refer [M44]]
   [thi.ng.math.core :as m]
   #?(:clj [thi.ng.math.macros :as mm])))

(defn face-attribs
  "Vertex attribute generator using given seq of attribs. The seq
  should contain at least the same number of elements as there are
  faces to be generated. Each item itself is a vector of attrib
  values (in vertex order) to be assigned to each vertex. Returns
  generator fn."
  [fattribs] (fn [fid vid _ _] (-> fattribs (nth fid) (nth vid))))

(defn const-face-attribs
  "Similar to face-attribs fn, but for attributes which are constant
  for all vertices of a single face. Takes a seq of attrib values and
  returns generator fn."
  [fattribs] (fn [fid _ _ _] (nth fattribs fid)))
(defn generate-face-attribs
  "Takes a vector of face vertices, face id, a map of vertex attribute
  generator fns and an options arg passed to the attribute generator
  fns. Returns 2-elem vector of [verts vert-attribs]. The generator
  fns themselves take these 4 args and should return the attribute for
  a single vertex: face-id, vertex-id, vertex, opts (a map)."
  [verts face-id attribs opts]
  [verts
   (when (seq attribs)
     (reduce-kv
      (fn [acc k f] (assoc acc k (map-indexed #(f face-id % %2 opts) verts)))
      {} attribs))])

(defn supplied-attrib
  "Higher order helper, returns attribute generator fn for types which
  emit pre-computed values as part of their `as-mesh` impl. Takes
  attrib key and for each vertex looks up value in opts map (which is
  supplied by supporting types, e.g. sphere). If called via 2 args,
  the looked up values will also be passed to transformation fn (2nd
  arg)."
  ([attrib] (fn [_ id _ opts] (-> opts (get attrib) (nth id))))
  ([attrib tx] (fn [_ id _ opts] (-> opts (get attrib) (nth id) tx))))
(defn uv-rect-for-size
  ([w] (uv-rect-for-size w w))
  ([w h] (uv-rect-for-size w w 0.0 0.0 1.0 1.0))
  ([w h x y uw vh]
   (let [u  (* 0.5 (/ uw w))
         v  (* 0.5 (/ vh h))
         iu (- uw u)
         iv (- vh v)]
     (mapv #(m/+ % x y) [(vec2 u v) (vec2 iu v) (vec2 iu iv) (vec2 u iv)]))))

(defn uv-cube-map-h
  ([h] (uv-cube-map-h h false))
  ([h pow2?]
   (let [w  (* h 6)
         tw (if pow2? (m/ceil-pow2 w) w)
         fw (/ (/ w tw) 6.0)]
     (mapv #(uv-rect-for-size h h (* % fw) 0.0 fw 1.0) (range 6)))))

(defn uv-cube-map-v
  ([h] (uv-cube-map-v h false))
  ([w pow2?]
   (let [h  (* w 6)
         th (if pow2? (m/ceil-pow2 h) h)
         fh (/ (/ h th) 6.0)]
     (mapv #(uv-rect-for-size w w 0.0 (* % fh) 1.0 fh) (range 6)))))

(def uv-default-rect [(vec2) (vec2 1.0 0.0) (vec2 1.0) (vec2 0.0 1.0)])
(def uv-faces (face-attribs (repeat uv-default-rect)))
(defn uv-tube
  "Generates tubular UV coordinates, using data provided in
  options map (:u :v :du :dv keys)"
  [_ vid _ {:keys [u v du dv]}]
  (case (int vid)
    0 (v/vec2 u v)
    1 (v/vec2 (+ u du) v)
    2 (v/vec2 (+ u du) (+ v dv))
    (v/vec2 u (+ v dv))))
(defn uv-flat-disc
  "Generates UV coordinates for a circle/trianglefan, using data
  provided in options map (:theta & :r keys). The first vertex of each
  triangle is assumed to be in the center of the circle, the other 2
  vertices are located at the circles perimeter."
  [_ vid _ {:keys [theta r] :as opts}]
  (case (int vid)
    0 (vec2 0.5)
    1 (vec2 (mm/madd (Math/cos theta) r 0.5)
            (mm/madd (Math/sin theta) r 0.5))
    (let [theta (+ theta (get opts :delta))]
      (vec2 (mm/madd (Math/cos theta) r 0.5)
            (mm/madd (Math/sin theta) r 0.5)))))

(defn uv-polygon-disc
  "HOF UV generator for polygons/polyhedras. Takes polygon resolution
  and computes N UV coords, returns generator fn"
  [res]
  (->> res
       m/norm-range
       (mapv #(m/+ (g/as-cartesian (vec2 0.5 (* % m/TWO_PI))) 0.5))
       repeat
       face-attribs))
