(ns thi.ng.trio.inference
  (:require
   [thi.ng.trio.core :as api]
   [thi.ng.trio.query :as q]
   [thi.ng.trio.vocabs.rdf :refer [rdf]]
   [thi.ng.trio.vocabs.rdfs :refer [rdfs]]
   [thi.ng.trio.vocabs.owl :refer [owl]]
   [clojure.set :as set]
   #+clj [taoensso.timbre :refer [debug info warn error]]))

(def prop-rules
  {:prp-dom   {:q [{:where [['?p (:domain rdfs) '?c]
                            ['?x '?p '?y]]}]
               :c [['?x (:type rdf) '?c]]}

   :prp-rng   {:q [{:where [['?p (:range rdfs) '?c]
                            ['?x '?p '?y]]}]
               :c [['?y (:type rdf) '?c]]}

   :prp-fp    {:q [{:where [['?p (:type rdf) (:FunctionalProperty owl)]
                            ['?x '?p '?y1]
                            ['?x '?p '?y2]]}]
               :c [['?y1 (:sameAs owl) '?y2]]}

   :prp-ifp   {:q [{:where [['?p (:type rdf) (:InverseFunctionalProperty owl)]
                            ['?x1 '?p '?y]
                            ['?x2 '?p '?y]]}]
               :c [['?x1 (:sameAs owl) '?x2]]}

   :prp-irp   {:q [{:where [['?p (:type rdf) (:IrreflexiveProperty owl)]
                            ['?x '?p '?x]]}]
               :c false}

   :prp-symp  {:q [{:where [['?p (:type rdf) (:SymmetricProperty owl)]
                            ['?x '?p '?y]]}]
               :c [['?y '?p '?x]]}

   :prp-asyp  {:q [{:where [['?p (:type rdf) (:AsymmetricProperty owl)]
                            ['?x '?p '?y]
                            ['?y '?p '?x]]}]
               :c false}

   :prp-trp   {:q [{:where [['?p (:type rdf) (:TransitiveProperty owl)]
                            ['?x '?p '?y]
                            ['?y '?p '?z]]}]
               :c [['?x '?p '?z]]}

   :prp-spo1  {:q [{:where [['?p1 (:subPropertyOf rdfs) '?p2]
                            ['?x '?p1 '?y]]}]
               :c [['?x '?p2 '?y]]}

   ;; :prp-spo2  nil ;; TODO requires property chains

   :prp-eqp1  {:q [{:where [['?p1 (:equivalentProperty owl) '?p2]
                            ['?x '?p1 '?y]]}]
               :c [['?x '?p2 '?y]]}

   :prp-eqp2  {:q [{:where [['?p1 (:equivalentProperty owl) '?p2]
                            ['?x '?p2 '?y]]}]
               :c [['?x '?p1 '?y]]}

   :prp-pdw   {:q [{:where [['?p1 (:propertyDisjointWith owl) '?p2]
                            ['?x '?p1 '?y]
                            ['?x '?p2 '?y]]}]
               :c false}

   ;; :prp-adp   nil ;; TODO required property chains

   :prp-inv1  {:q [{:where [['?p1 (:inverseOf owl) '?p2]
                            ['?x '?p1 '?y]]}]
               :c [['?y '?p2 '?x]]}

   :prp-inv2  {:q [{:where [['?p1 (:inverseOf owl) '?p2]
                            ['?x '?p2 '?y]]}]
               :c [['?y '?p1 '?x]]}

   ;; :prp-key   nil ;; TODO required property chains

   :prp-npa1  {:q [{:where [['?x (:sourceIndividual owl) '?i1]
                            ['?x (:assertionProperty owl) '?p]
                            ['?x (:targetIndividual owl) '?i2]
                            ['?i1 '?p '?i2]]}]
               :c false}

   :prp-npa2  {:q [{:where [['?x (:sourceIndividual owl) '?i]
                            ['?x (:assertionProperty owl) '?p]
                            ['?x (:targetValue owl) '?lt]
                            ['?i '?p '?lt]]}]
               :c false}
   })

(def class-rules
  {:cls-thing     {:q true
                   :c [[(:Thing owl) (:type rdf) (:Class owl)]]}

   :cls-nothing1  {:q true
                   :c [[(:Nothing owl) (:type rdf) (:Class owl)]]}

   :cls-nothing2  {:q [{:where [['?x (:type rdf) (:Nothing owl)]]}]
                   :c false}

   ;; :cls-int1      nil ;; TODO

   ;; :cls-int2      nil ;; TODO

   ;; :cls-uni       nil ;; TODO

   :cls-com       {:q [{:where [['?c1 (:complementOf owl) '?c2]
                                ['?x (:type rdf) '?c1]]}]
                   :c [['?x (:type rdf) '?c2]]}

   :cls-svf1      {:q [{:where [['?x (:someValuesFrom owl) '?y]
                                ['?x (:onProperty owl) '?p]
                                ['?u '?p '?v]
                                ['?v (:type rdf) '?y]]}]
                   :c [['?u (:type rdf) '?x]]}

   :cls-svf2      {:q [{:where [['?x (:someValuesFrom owl) (:Thing owl)]
                                ['?x (:onProperty owl) '?p]
                                ['?u '?p '?v]]}]
                   :c [['?u (:type rdf) '?x]]}

   :cls-avf       {:q [{:where [['?x (:allValuesFrom owl) '?y]
                                ['?x (:onProperty owl) '?p]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?v]]}]
                   :c [['?v (:type rdf) '?y]]}

   :cls-hv1       {:q [{:where [['?x (:hasValue owl) '?y]
                                ['?x (:onProperty owl) '?p]
                                ['?u (:type rdf) '?x]]}]
                   :c [['?u '?p '?y]]}

   :cls-hv2       {:q [{:where [['?x (:hasValue owl) '?y]
                                ['?x (:onProperty owl) '?p]
                                ['?u '?p '?y]]}]
                   :c [['?u (:type rdf) '?x]]}

   :cls-maxc1     {:q [{:where [['?x (:maxCardinality owl) 0]
                                ['?x (:onProperty owl) '?p]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?y]]}]
                   :c false}

   :cls-maxc2     {:q [{:where [['?x (:maxCardinality owl) 1]
                                ['?x (:onProperty owl) '?p]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?y1]
                                ['?u '?p '?y2]]}]
                   :c [['?y1 (:sameAs owl) '?y2]]}

   :cls-maxqc1    {:q [{:where [['?x (:maxQualifiedCardinality owl) 0]
                                ['?x (:onProperty owl) '?p]
                                ['?x (:onClass owl) '?c]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?y]
                                ['?y (:type rdf) '?c]]}]
                   :c false}

   :cls-maxqc2    {:q [{:where [['?x (:maxQualifiedCardinality owl) 0]
                                ['?x (:onProperty owl) '?p]
                                ['?x (:onClass owl) (:Thing owl)]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?y]]}]
                   :c false}

   :cls-maxqc3    {:q [{:where [['?x (:maxQualifiedCardinality owl) 1]
                                ['?x (:onProperty owl) '?p]
                                ['?x (:onClass owl) '?c]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?y1]
                                ['?y1 (:type rdf) '?c]
                                ['?u '?p '?y2]
                                ['?y2 (:type rdf) '?c]]}]
                   :c [['?y1 (:sameAs owl) '?y2]]}

   :cls-maxqc4    {:q [{:where [['?x (:maxQualifiedCardinality owl) 1]
                                ['?x (:onProperty owl) '?p]
                                ['?x (:onClass owl) (:Thing owl)]
                                ['?u (:type rdf) '?x]
                                ['?u '?p '?y1]
                                ['?u '?p '?y2]]}]
                   :c [['?y1 (:sameAs owl) '?y2]]}

   ;; :cls-oo        nil ;; TODO
   })

(def class-semantics
  {:cax-sco {:q [{:where [['?c1 (:subClassOf rdfs) '?c2]
                          ['?x (:type rdf) '?c1]]}]
             :c [['?x (:type rdf) '?c2]]}

   :cax-eqc1 {:q [{:where [['?c1 (:equivalentClass owl) '?c2]
                           ['?x (:type rdf) '?c1]]}]
              :c [['?x (:type rdf) '?c2]]}

   :cax-eqc2 {:q [{:where [['?c1 (:equivalentClass owl) '?c2]
                           ['?x (:type rdf) '?c2]]}]
              :c [['?x (:type rdf) '?c1]]}

   :cax-dw   {:q [{:where [['?c1 (:disjointWith owl) '?c2]
                           ['?x (:type rdf) '?c1]
                           ['?x (:type rdf) '?c2]]}]
              :c false}

   ;; :cax-adc  nil ;; TODO
   })

(def schema-rules
  {:scm-cls  {:q [{:where [['?c (:type rdf) (:Class owl)]]}]
              :c [['?c (:subClassOf rdfs) '?c]
                  ['?c (:equivalentClass owl) '?c]
                  ['?c (:subClassOf rdfs) (:Thing owl)]
                  [(:Nothing owl) (:subClassOf rdfs) '?c]]}

   :scm-sco  {:q [{:where [['?c1 (:subClassOf rdfs) '?c2]
                           ['?c2 (:subClassOf rdfs) '?c3]]}]
              :c [['?c1 (:subClassOf rdfs) '?c3]]}

   :scm-eqc1 {:q [{:where [['?c1 (:equivalentClass owl) '?c2]]}]
              :c [['?c1 (:subClassOf rdfs) '?c2]
                  ['?c2 (:subClassOf rdfs) '?c1]]}

   :scm-eqc2 {:q [{:where [['?c1 (:subClassOf rdfs) '?c2]
                           ['?c2 (:subClassOf rdfs) '?c1]]}]
              :c [['?c1 (:equivalentClass owl) '?c2]]}

   :scm-op   {:q [{:where [['?p (:type rdf) (:ObjectProperty owl)]]}]
              :c [['?p (:subPropertyOf rdfs) '?p]
                  ['?p (:equivalentProperty owl) '?p]]}

   :scm-dp   {:q [{:where [['?p (:type rdf) (:DatatypeProperty owl)]]}]
              :c [['?p (:subPropertyOf rdfs) '?p]
                  ['?p (:equivalentProperty owl) '?p]]}

   :scm-spo  {:q [{:where [['?p1 (:subPropertyOf rdfs) '?p2]
                           ['?p2 (:subPropertyOf rdfs) '?p3]]}]
              :c [['?p1 (:subPropertyOf rdfs) '?p3]]}

   :scm-eqp1 {:q [{:where [['?p1 (:equivalentProperty owl) '?p2]]}]
              :c [['?p1 (:subPropertyOf rdfs) '?p2]
                  ['?p2 (:subPropertyOf rdfs) '?p1]]}

   :scm-eqp2 {:q [{:where [['?p1 (:subPropertyOf rdfs) '?p2]
                           ['?p2 (:subPropertyOf rdfs) '?p1]]}]
              :c [['?p1 (:equivalentProperty owl) '?p2]]}

   :scm-dom1 {:q [{:where [['?p (:domain rdfs) '?c1]
                           ['?c1 (:subClassOf rdfs) '?c2]]}]
              :c [['?p (:domain rdfs) '?c2]]}

   :scm-dom2 {:q [{:where [['?p2 (:domain rdfs) '?c]
                           ['?p1 (:subPropertyOf rdfs) '?p2]]}]
              :c [['?p1 (:domain rdfs) '?c]]}

   :scm-rng1 {:q [{:where [['?p (:range rdfs) '?c1]
                           ['?c1 (:subClassOf rdfs) '?c2]]}]
              :c [['?p (:range rdfs) '?c2]]}

   :scm-rng2 {:q [{:where [['?p2 (:range rdfs) '?c]
                           ['?p1 (:subPropertyOf rdfs) '?p2]]}]
              :c [['?p1 (:range rdfs) '?c]]}

   :scm-hv   {:q [{:where [['?c1 (:hasValue owl) '?i]
                           ['?c1 (:onProperty owl) '?p1]
                           ['?c2 (:hasValue owl) '?i]
                           ['?c2 (:onProperty owl) '?p2]
                           ['?p1 (:subPropertyOf rdfs) '?p2]]}]
              :c [['?c1 (:subClassOf rdfs) '?c2]]}

   :scm-svf1 {:q [{:where [['?c1 (:someValuesFrom owl) '?y1]
                           ['?c1 (:onProperty owl) '?p]
                           ['?c2 (:someValuesFrom owl) '?y2]
                           ['?c2 (:onProperty owl) '?p]
                           ['?y1 (:subClassOf rdfs) '?y2]]}]
              :c [['?c1 (:subClassOf rdfs) '?c2]]}

   :scm-svf2 {:q [{:where [['?c1 (:someValuesFrom owl) '?y]
                           ['?c1 (:onProperty owl) '?p1]
                           ['?c2 (:someValuesFrom owl) '?y]
                           ['?c2 (:onProperty owl) '?p2]
                           ['?p1 (:subPropertyOf rdfs) '?p2]]}]
              :c [['?c1 (:subClassOf rdfs) '?c2]]}

   :scm-avf1 {:q [{:where [['?c1 (:allValuesFrom owl) '?y1]
                           ['?c1 (:onProperty owl) '?p]
                           ['?c2 (:allValuesFrom owl) '?y2]
                           ['?c2 (:onProperty owl) '?p]
                           ['?y1 (:subClassOf rdfs) '?y2]]}]
              :c [['?c1 (:subClassOf rdfs) '?c2]]}

   :scm-avf2 {:q [{:where [['?c1 (:allValuesFrom owl) '?y]
                           ['?c1 (:onProperty owl) '?p1]
                           ['?c2 (:allValuesFrom owl) '?y]
                           ['?c2 (:onProperty owl) '?p2]
                           ['?p1 (:subPropertyOf rdfs) '?p2]]}]
              :c [['?c2 (:subClassOf rdfs) '?c1]]}

   ;; :scm-int  nil ;; TODO

   ;; :scm-uni  nil ;; TODO
   })


(defn pattern-equiv
  [[s p o] [s' p' o']]
  (and
   (if (q/qvar? s) (q/qvar? s') (= s s'))
   (if (q/qvar? p) (q/qvar? p') (= p p'))
   (if (q/qvar? o) (q/qvar? o') (= o o'))))

(defn subsumes?
  [[s p o] [s' p' o']]
  (and
   (or (q/qvar? s) (= s s'))
   (or (q/qvar? p) (= p p'))
   (or (q/qvar? o) (= o o'))))

(defn grounded?
  [[s p o]] (not (or (q/qvar? s) (q/qvar? p) (q/qvar? o))))

(defn normalize
  [t] (mapv #(if (q/qvar? %) nil %) t))

(defn infer-rule
  [ds id {:keys [q c]} inf]
  (let [rule {:construct c :query q}]
    (loop [ds ds, inf inf]
      (let [inf' (->> inf
                      (set/difference (q/query (assoc rule :from ds)))
                      (filter #(nil? (seq (apply api/select ds %)))))]
        #+clj (debug :inf inf')
        (if (seq inf')
          (recur
           (if id
             (api/add-triples ds id inf')
             (api/add-triples ds inf'))
           (if (< (count inf) (count inf'))
             (into inf' inf)
             (into inf inf')))
          [ds inf])))))

(defn infer-rules
  "Takes a PModel or PDataset, a map of rule specs and applies
  infer-rule to all rules over max-passes. Stops as soon as no new
  triples can be inferred for any rule. Accepts an optional graph name
  `g` as target for inferred triples within a PDataSet. Returns 2-elem
  vector of [updated-model inf-map] where inf-map is a map of inferred
  triples with rule IDs as their keys."
  ([ds rules max-passes]
     (infer-rules ds nil rules max-passes))
  ([ds g rules max-passes]
     (loop [state [ds {} false] i 1]
       (if (> i max-passes)
         (butlast state)
         (let [state (reduce
                      (fn [[ds inf :as state] [id rule]]
                        #+clj (debug :rule id)
                        (let [[ds inf'] (infer-rule ds g rule #{})]
                          (if (seq inf')
                            [ds (assoc inf id (into (or (inf id) #{}) inf')) true]
                            state)))
                      (assoc state 2 false) rules)]
           #+clj (debug :pass i :state (peek state))
           (if (peek state)
             (recur state (inc i))
             (butlast state)))))))
