(ns thi.ng.trio.core
  (:refer-clojure :exclude [object? indexed?])
  (:require
   [thi.ng.common.data.core :as d]
   [thi.ng.common.data.unionfind :as u]
   [thi.ng.common.error :as err]
   [clojure.set :as set]
   #+clj [clojure.pprint]
   #+clj [clojure.core.protocols :as cp]))

(defprotocol PTripleSeq
 (triple-seq [_]))
(defprotocol PModelConvert
 (as-model [_]))
(defprotocol PModel
  (subjects [_])
  (predicates [_])
  (objects [_])
  (subject? [_ x])
  (predicate? [_ x])
  (object? [_ x])
  (indexed? [_ x])
  (model-size [_]))
(defprotocol PModelUpdate
  (add-triple [_ s] [_ g s])
  (add-triples [_ triples] [_ g triples])
  (remove-triple [_ s] [_ g s])
  (remove-triples [_ triples] [_ g triples])
  (update-triple [_ s s'] [_ g s s'])
  (remove-subject [_ s] [_ g s]))
(defprotocol PModelWatch
  (add-pre-commit-hook [_ id f])
  (add-post-commit-hook [_ id f])
  (remove-pre-commit-hook [_ id])
  (remove-post-commit-hook [_ id]))
(defprotocol PModelSelect
  (select [_] [_ s p o] [_ g s p o])
  (select-with-alts [_ s p o] [_ g s p o]))
(defprotocol PDataset
  (remove-model [_ id])
  (update-model [_ id m])
  (get-model [_ id]))
(defprotocol PAliasModelSupport
  (rewrite-alias [_ a b]))

(defn- remove-from-index
  [idx i1 i2 i3]
  (let [kv (idx i1)
        v (disj (kv i2) i3)
        kv (if (seq v) kv (dissoc kv i2))]
    (if (seq kv)
      (if (seq v)
        (assoc-in idx [i1 i2] v)
        (assoc idx i1 kv))
      (dissoc idx i1))))

(defn- rewrite-alias*
  [store pred id p q]
  (if (pred store p)
    (let [xs (apply select store (assoc [nil nil nil] id p))
          store (remove-triples store xs)]
      (add-triples store (map #(assoc % id q) xs)))
    store))

(defn rewrite-alias-naive
  [store p q]
  (-> store
      (rewrite-alias* subject?   0 p q)
      (rewrite-alias* predicate? 1 p q)
      (rewrite-alias* object?    2 p q)))

(defn trace [prefix x] (prn prefix x) x)

(declare swizzle swizzle-assoc)

(deftype Triple
  #+clj  [s p o ^:unsynchronized-mutable __hash]
  #+cljs [s p o ^:mutable __hash]
#+clj clojure.lang.ILookup
#+clj (valAt
       [_ k] (swizzle _ k nil))
#+clj (valAt
       [_ k nf] (swizzle _ k nf))

#+clj java.util.concurrent.Callable
#+clj (call
       [_] (.invoke ^clojure.lang.IFn _))
#+clj java.lang.Runnable
#+clj (run
        [_] (.invoke ^clojure.lang.IFn _))

#+clj clojure.lang.IFn
#+clj (invoke
       [_ k] (swizzle _ k nil))
#+clj (invoke
       [_ k nf] (swizzle _ k nf))
#+clj (applyTo
       [_ args]
       (condp = (count args)
         1 (swizzle _ (first args) nil)
         2 (swizzle _ (first args) (second args))
         (err/arity-error! (count args))))

#+clj clojure.lang.IPersistentVector
#+clj clojure.lang.Associative
#+clj (count
       [_] 3)
#+clj (length
       [_] 3)
#+clj (containsKey
       [_ k] (not (nil? (#{0 1 2 :s :p :o} k))))
#+clj (entryAt
       [_ k] (clojure.lang.MapEntry. k (case (int k) 0 s, 1 p, 2 o, (err/key-error! k))))
#+clj (assoc
          [_ k v] (swizzle-assoc _ k v))
#+clj (assocN
       [_ k v]
       (case (int k)
         0 (Triple. v p o nil)
         1 (Triple. s v o nil)
         2 (Triple. s p v nil)
         (err/key-error! k)))

#+clj java.util.Collection
#+clj (isEmpty
       [_] false)
#+clj (iterator
       [_] (.iterator ^java.util.Collection (list s p o)))
#+clj (toArray
       [_] (object-array _))
#+clj (size
       [_] 3)

#+clj clojure.lang.Sequential
#+clj clojure.lang.Seqable
#+clj (seq [_] (seq [s p o]))
#+clj (cons
       [_ x] [s p o x])
#+clj (peek
       [_] o)
#+clj (pop
       [_] [s p])
#+clj (rseq
       [_] (seq [o p s]))
#+clj (nth
       [_ k] (case (int k) 0 s, 1 p, 2 o, (err/key-error! k)))
#+clj (nth
       [_ k nf] (case (int k) 0 s, 1 p, 2 o, nf))
#+clj (equiv
       [_ x]
       (if (instance? Triple x)
           (and (clojure.lang.Util/equiv s (.-s ^Triple x))
                (clojure.lang.Util/equiv p (.-p ^Triple x))
                (clojure.lang.Util/equiv o (.-o ^Triple x)))
           (and (instance? java.util.Collection x)
                (== 3 (count x))
                (clojure.lang.Util/equiv s (nth x 0))
                (clojure.lang.Util/equiv p (nth x 1))
                (clojure.lang.Util/equiv o (nth x 2)))))
#+clj (equals
       [_ x]
       (if (instance? Triple x)
           (and (clojure.lang.Util/equals s (.-s ^Triple x))
                (clojure.lang.Util/equals p (.-p ^Triple x))
                (clojure.lang.Util/equals o (.-o ^Triple x)))
           (and (instance? java.util.Collection x)
                (== 3 (count x))
                (clojure.lang.Util/equals s (nth x 0))
                (clojure.lang.Util/equals p (nth x 1))
                (clojure.lang.Util/equals o (nth x 2)))))
#+clj (hashCode
       [_]
       (-> 31
           (unchecked-add-int (hash s))
           (unchecked-multiply-int 31)
           (unchecked-add-int (hash p))
           (unchecked-multiply-int 31)
           (unchecked-add-int (hash o))))

#+clj clojure.lang.IHashEq
#+clj (hasheq
       [_]
       (or __hash (set! __hash
                        (mix-collection-hash
                         (-> 31
                             (unchecked-add-int (hash s))
                             (unchecked-multiply-int 31)
                             (unchecked-add-int (hash p))
                             (unchecked-multiply-int 31)
                             (unchecked-add-int (hash o)))
                         3))))

#+clj Comparable
#+clj (compareTo
       [_ x]
       (if (instance? Triple x)
         (let [c (compare s (.-s ^Triple x))]
           (if (== 0 c)
             (let [c (compare p (.-p ^Triple x))]
               (if (== 0 c)
                 (compare o (.-o ^Triple x))
                 c))
             c))
         (let [c (count x)]
           (if (== 3 c) (compare x _) (- 3 c)))))

#+clj cp/InternalReduce
#+clj (internal-reduce
       [_ f start]
       (let [acc (f start s)]
         (if (reduced? acc)
           @acc
           (let [acc (f acc p)]
             (if (reduced? acc)
               @acc
               (let [acc (f acc o)]
                 (if (reduced? acc)
                   @acc
                   acc)))))))

#+clj cp/CollReduce
#+clj (coll-reduce
       [_ f]
       (let [acc (f s p)]
         (if (reduced? acc)
           @acc
           (let [acc (f acc o)]
             (if (reduced? acc)
               @acc
               acc)))))
#+clj (coll-reduce
       [_ f start]
       (let [acc (f start s)]
         (if (reduced? acc)
           @acc
           (let [acc (f acc p)]
             (if (reduced? acc)
               @acc
               (let [acc (f acc o)]
                 (if (reduced? acc)
                   @acc
                   acc)))))))

#+clj Object
#+clj (toString
       [_]
       (.toString
        (doto (StringBuilder. "[")
          (.append (pr-str s))
          (.append " ")
          (.append (pr-str p))
          (.append " ")
          (.append (pr-str o))
          (.append "]"))))
#+cljs ICloneable
#+cljs (-clone
        [_] (Triple. s p o __hash))

#+cljs ILookup
#+cljs (-lookup
        [_ k] (swizzle _ k nil))
#+cljs (-lookup
        [_ k nf] (swizzle _ k nf))

#+cljs IFn
#+cljs (-invoke
        [_ k] (swizzle _ k nil))
#+cljs (-invoke
        [_ k nf] (swizzle _ k nf))

#+cljs ICounted
#+cljs (-count
        [_] 3)

#+cljs IAssociative
#+cljs (-contains-key?
        [_ k] (not (nil? (#{0 1 2 :s :p :o} k))))
#+cljs (-assoc
        [_ k v] (swizzle-assoc _ k v))

#+cljs IVector
#+cljs (-assoc-n
        [_ k v]
        (case (int k)
          0 (Triple. v p o nil)
          1 (Triple. s v o nil)
          2 (Triple. s p v nil)
          (err/key-error! k)))

#+cljs ISequential
#+cljs ISeq
#+cljs (-first
        [_] s)
#+cljs (-rest
        [_] (cons p (cons o nil)))

#+cljs INext
#+cljs (-next
        [_] (cons p (cons o nil)))

#+cljs ISeqable
#+cljs (-seq
        [_] _)

#+cljs IReversible
#+cljs (-rseq
        [_] (Triple. o p s nil))

#+cljs IIndexed
#+cljs (-nth
        [_ k] (case (int k) 0 s, 1 p, 2 o, (err/key-error! k)))
#+cljs (-nth
        [_ k nf] (case (int k) 0 s, 1 p, 2 o, nf))

#+cljs ICollection
#+cljs (-conj
        [_ x] [s p o x])

#+cljs IStack
#+cljs (-peek
        [_] o)
#+cljs (-pop
        [_] [s p])

#+cljs IComparable
#+cljs (-compare
        [_ x]
        (if (instance? Triple x)
          (let [c (compare s (.-s ^Triple x))]
            (if (== 0 c)
              (let [c (compare p (.-p ^Triple x))]
                (if (== 0 c)
                  (compare o (.-o ^Triple x))
                  c))
              c))
          (let [c (count x)]
            (if (== 3 c) (compare x _) (- 3 c)))))

#+cljs IHash
#+cljs (-hash
        [_]
        (or __hash
            (set! (.-__hash _)
                  (mix-collection-hash
                   (-> 31 (+ (hash s))
                       (bit-or 0)
                       (imul 31) (+ (hash p))
                       (bit-or 0)
                       (imul 31) (+ (hash o))
                       (bit-or 0))
                   3))))

#+cljs IEquiv
#+cljs (-equiv
        [_ x]
        (if (instance? Triple x)
          (and (= s (.-s ^Triple x)) (= p (.-p ^Triple x)) (= o (.-o ^Triple x)))
          (and (sequential? x) (== 3 (count x))
               (= s (nth x 0)) (= p (nth x 1)) (= o (nth x 2)))))

#+cljs IReduce
#+cljs (-reduce
        [coll f]
        (let [acc (f s p)]
          (if (reduced? acc)
            @acc
            (let [acc (f acc o)]
              (if (reduced? acc)
                @acc
                acc)))))
#+cljs (-reduce
        [coll f start]
        (let [acc (f start s)]
          (if (reduced? acc)
            @acc
            (let [acc (f acc p)]
              (if (reduced? acc)
                @acc
                (let [acc (f acc o)]
                  (if (reduced? acc)
                    @acc
                    acc)))))))

#+cljs Object
#+cljs (toString
        [_] (str "[" (pr-str s) " " (pr-str p) " " (pr-str o) "]"))
)
#+clj (defmethod clojure.pprint/simple-dispatch Triple
  [^Triple o] ((get-method clojure.pprint/simple-dispatch clojure.lang.IPersistentVector) o))
#+clj (defmethod print-method Triple [^Triple o ^java.io.Writer w] (.write w (.toString o)))
(defn- lookup3
  [^Triple _ k nf]
  (case k
    \s (.-s _)
    \p (.-p _)
    \o (.-o _)
    (or nf (err/key-error! k))))

(defn- swizzle
  [^Triple _ k default]
  (if (number? k)
    (case (int k)
      0 (.-s _)
      1 (.-p _)
      2 (.-o _)
      (or default (err/key-error! k)))
    (case k
      :s (.-s _)
      :p (.-p _)
      :o (.-o _)
      (let [n (name k) c (count n)]
        (case c
          2 [(lookup3 _ (nth n 0) default)
             (lookup3 _ (nth n 1) default)]
          3 (Triple.
             (lookup3 _ (nth n 0) default)
             (lookup3 _ (nth n 1) default)
             (lookup3 _ (nth n 2) default)
             nil)
          (or default (err/key-error! k)))))))

(defn- swizzle-assoc*
  [_ keymap k v]
  (let [n (name k)
        c (count n)]
    (if (and (<= c (count keymap)) (== c (count v) (count (into #{} n))))
      (loop [acc (vec _), i 0, n n]
        (if n
          (recur (assoc acc (keymap (first n)) (v i)) (inc i) (next n))
          (Triple. (acc 0) (acc 1) (acc 2) nil)))
      (err/key-error! k))))

(defn- swizzle-assoc
  [^Triple _ k v]
  (case k
    :s (Triple. v (.-p _) (.-o _) nil)
    :p (Triple. (.-s _) v (.-o _) nil)
    :o (Triple. (.-s _) (.-p _) v nil)
    0 (Triple. v (.-p _) (.-o _) nil)
    1 (Triple. (.-s _) v (.-o _) nil)
    2 (Triple. (.-s _) (.-p _) v nil)
    (swizzle-assoc* _ {\s 0 \p 1 \o 2} k v)))
(defn triple
  ([s p o] (Triple. s p o nil))
  ([t] (if (instance? Triple t) t (Triple. (first t) (nth t 1) (nth t 2) nil))))

(defn- select-with-alts-1
  [coll idx]
  (->> (set/intersection coll (set (keys idx)))
       (mapcat #(->> % idx vals (apply concat)))))

(defn- select-with-alts-2
  [outer inner idx]
  (mapcat
   (fn [o]
     (let [out (idx o)]
       (->> (set (keys out))
            (set/intersection inner)
            (mapcat out))))
   outer))

(defn- select-with-alts-3
  [outer inner preds idx lookup]
  (reduce
   (fn [acc o]
     (let [out (idx o)]
       (->> (set (keys out))
            (set/intersection preds)
            (reduce
             (fn [acc p]
               (if-let [t (some #(if (inner (lookup %)) %) (out p))]
                 (conj acc t)
                 acc))
             acc))))
   [] outer))

(defrecord PlainMemoryStore [spo pos osp size]
  PModelUpdate
  (add-triple
    [_ [s p o :as t]]
    (if (-> spo (get s nil) (get p nil) (get t nil)) _
        (let [s (get (find spo s) 0 s)
              p (get (find pos p) 0 p)
              o (get (find osp o) 0 o)
              t (Triple. s p o nil)]
          (PlainMemoryStore.
           (update-in spo [s p] d/set-conj t)
           (update-in pos [p o] d/set-conj t)
           (update-in osp [o s] d/set-conj t)
           (inc size)))))
  (add-triples [_ triples]
    (loop [changed? false, spo spo, pos pos, osp osp, size size, xs triples]
      (if xs
        (let [[s p o :as t] (first xs)]
          (if (-> spo (get s nil) (get p nil) (get t nil))
            (recur changed? spo pos osp size (next xs))
            (let [s (get (find spo s) 0 s)
                  p (get (find pos p) 0 p)
                  o (get (find osp o) 0 o)
                  t (Triple. s p o nil)]
              (recur
               true
               (update-in spo [s p] d/set-conj t)
               (update-in pos [p o] d/set-conj t)
               (update-in osp [o s] d/set-conj t)
               (inc size)
               (next xs)))))
        (if changed?
          (PlainMemoryStore. spo pos osp size)
          _))))
  (remove-triple [_ [s p o :as t]]
    (if (-> spo (get s nil) (get p nil) (get t nil))
      (PlainMemoryStore.
       (remove-from-index spo s p t)
       (remove-from-index pos p o t)
       (remove-from-index osp o s t)
       (dec size))
      _))
  (remove-triples [_ triples]
    (loop [changed? false, spo spo, pos pos, osp osp, size size, xs triples]
      (if xs
        (let [[s p o :as t] (first xs)]
          (if (-> spo (get s nil) (get p nil) (get t nil))
            (recur
             true
             (remove-from-index spo s p t)
             (remove-from-index pos p o t)
             (remove-from-index osp o s t)
             (dec size)
             (next xs))
            (recur changed? spo pos osp size (next xs))))
        (if changed?
          (PlainMemoryStore. spo pos osp size)
          _))))
  (update-triple [_ s1 s2]
    (add-triple (remove-triple _ s1) s2))
  (remove-subject [_ s]
    (remove-triples _ (select _ s nil nil)))

  PModel
  (subject? [_ x]
    (if (spo x) x))
  (predicate? [_ x]
    (if (pos x) x))
  (object? [_ x]
    (if (osp x) x))
  (indexed? [_ x]
    (if (or (spo x) (pos x) (osp x)) x))
  (subjects [_] (keys spo))
  (predicates [_] (keys pos))
  (objects [_] (keys osp))
  (model-size [_] size)

  PModelSelect
  (select [_]
    (select _ nil nil nil))
  (select
    [_ s p o]
    (if s
      (if p
        (if o
          ;; s p o
          (let [t (triple s p o)]
            (if (-> spo (get s nil) (get p nil) (get t nil)) [t]))
          ;; s p nil
          (-> spo (get s nil) (get p nil)))
        ;; s nil o / s nil nil
        (if o
          (-> osp (get o nil) (get s nil))
          (->> (spo s) vals (apply concat))))
      (if p
        (if o
          ;; nil p o
          (-> pos (get p nil) (get o nil))
          ;; nil p nil
          (->> (pos p) vals (apply concat)))
        (if o
          ;; nil nil o
          (->> (osp o) vals (apply concat))
          ;; nil nil nil
          (->> spo vals (mapcat vals) (apply concat))))))

  (select-with-alts
    [_ s p o]
    (let [s (if (set? s) (if-not (empty? s) s) (if s #{s}))
          p (if (set? p) (if-not (empty? p) p) (if p #{p}))
          o (if (set? o) (if-not (empty? o) o) (if o #{o}))]
      (if s
        (if p
          (if o
            (select-with-alts-3 s o p spo peek)
            (select-with-alts-2 s p spo))
          (if o
            (select-with-alts-2 o s osp)
            (select-with-alts-1 s spo)))
        (if p
          (if o
            (select-with-alts-2 p o pos)
            (select-with-alts-1 p pos))
          (if o
            (select-with-alts-1 o osp)
            (->> spo vals (mapcat vals) (apply concat)))))))

  PAliasModelSupport
  (rewrite-alias
    [_ a b] (rewrite-alias-naive _ a b)))
(defprotocol PNode
  (get-children [_])
  (add-child [_ c])
  (remove-child [_ c])
  (is-leaf? [_]))

(defprotocol PGraph
  (add-node [_ n] [_ n parents])
  (get-node [_ t])
  (get-node-for-id [_ id])
  (get-ids [_])
  (get-nodes [_]))

(deftype TripleNode [triple ^:unsynchronized-mutable children]
  PNode
  (get-children
    [_] children)
  (add-child
    [_ c] (set! children (conj (or children #{}) c)) _)
  (remove-child
    [_ c] (set! children (disj children c)) _)
  (is-leaf?
    [_] (and (first triple) (nth triple 1) (nth triple 2)))

  Object
  (toString [_] (str (pr-str triple) " " (pr-str children))))

(def ^:private root [nil nil nil])

(defn tg-select-with-alts
  [_ s p o]
  (if (and (seq s) (seq p) (seq o))
    (mapcat
     (fn [[s p o]] (select _ s p o))
     (d/cartesian-product s p o))))

(declare index-branch)

(deftype TripleGraph [nodes ids next-id size]
  Object
  (toString
    [_] (str ":nodes " (pr-str nodes)
             " :ids " (pr-str ids)
             " :next " next-id))

  PGraph
  (add-node
    [_ n] (add-node _ n nil))
  (add-node
    [_ n parents]
    (let [g (TripleGraph.
             (assoc nodes next-id n)
             (assoc ids (.-triple  ^TripleNode n) next-id)
             (inc next-id)
             (if (is-leaf? n) (inc size) size))]
      (when (seq parents)
        (doseq [^TripleNode p parents]
          (add-child p next-id)))
      g))
  (get-node
    [_ t] (nodes (ids t)))
  (get-node-for-id
    [_ id] (nodes id))

  PModel
  (subject?
    [_ x] ((subjects _) x))
  (predicate?
    [_ x] ((predicates _) x))
  (object?
    [_ x] ((objects _) x))
  (indexed?
    [_ x] (or (subject? _ x) (predicate? _ x) (object? _ x)))
  (subjects
    [_] (->> (nodes 0)
             (get-children)
             (map #(first (.-triple ^TripleNode (nodes %))))
             (filter identity)
             (set)))
  (predicates
    [_] (->> (nodes 0)
             (get-children)
             (map #(nth (.-triple ^TripleNode (nodes %)) 1))
             (filter identity)
             (set)))
  (objects
    [_] (->> (nodes 0)
             (get-children)
             (map #(nth (.-triple ^TripleNode (nodes %)) 2))
             (filter identity)
             (set)))
  (model-size
    [_] size)

  PModelUpdate
  (add-triple
    [_ [s p o :as t]]
    (if-not (ids t)
      (let [id (.-next-id _)
            ^TripleGraph g (add-node _ (TripleNode. (Triple. s p o nil) nil) nil)]
        (-> (index-branch g id [[s p nil] [s nil nil] root])
            (index-branch id [[nil p o] [nil p nil] root])
            (index-branch id [[s nil o] [nil nil o] root])))
      _))
  (add-triples
    [_ triples] (reduce add-triple _ triples))

  PModelSelect
  (select
    [_] (select _ nil nil nil))
  (select
    [_ s p o]
    (let [ids (if (or s p o)
                (if-let [id (ids (triple s p o))] [id])
                (->> (nodes 0)
                     (get-children)
                     (filter #(first (.-triple ^TripleNode (nodes %))))))]
      (if (seq ids)
        (loop [acc (transient [])
               ids (into #+clj clojure.lang.PersistentQueue/EMPTY
                         #+cljs cljs.core.PersistentQueue.EMPTY ids)]
          (if (seq ids)
            (let [^TripleNode n (nodes (peek ids))
                  c (get-children n)]
              (if c
                (recur acc (into (pop ids) c))
                (recur (conj! acc (.-triple n)) (pop ids))))
            (persistent! acc))))))
  (select-with-alts
    [_ s p o]
    (let [s (if (set? s) (if-not (empty? s) s) (if s #{s}))
          p (if (set? p) (if-not (empty? p) p) (if p #{p}))
          o (if (set? o) (if-not (empty? o) o) (if o #{o}))]
      (if s
        (if p
          (if o
            (tg-select-with-alts
             _
             (set/intersection s (subjects _))
             (set/intersection p (predicates _))
             (set/intersection o (objects _)))
            (tg-select-with-alts
             _ (set/intersection s (subjects _)) (set/intersection p (predicates _)) #{nil}))
          (if o
            (tg-select-with-alts
             _ (set/intersection s (subjects _)) #{nil} (set/intersection o (objects _)))
            (tg-select-with-alts
             _ (set/intersection s (subjects _)) #{nil} #{nil})))
        (if p
          (if o
            (tg-select-with-alts
             _ #{nil} (set/intersection p (predicates _)) (set/intersection o (objects _)))
            (tg-select-with-alts
             _ #{nil} (set/intersection p (predicates _)) #{nil}))
          (if o
            (tg-select-with-alts
             _ #{nil} #{nil} (set/intersection o (objects _)))
            (select _ nil nil nil)))))))

(defn- index-branch
  [g id patterns]
  (loop [^TripleGraph g g, id id, ps patterns]
    (if ps
      (let [id' ((.-ids g) (first ps))]
        (if id'
          (do
            (add-child ^TripleNode ((.-nodes g) id') id)
            g)
          (recur
           (add-node g (TripleNode. (triple (first ps)) #{id}) nil)
           (.-next-id g)
           (next ps))))
      g)))

#+clj
(defn triple-graph->dot
  ([g]
     (str "digraph g {\n"
          "node[color=black,style=filled,fontname=Inconsolata,fontcolor=white,fontsize=9];\n"
          "edge[fontname=Inconsolata,fontsize=9];\n"
          (triple-graph->dot g 0 "")
          "}"))
  ([g id dot]
     (let [n (get-node-for-id g id)
           t (.-triple ^TripleNode n)
           d (format
              "%d[label=\"%d: %s\",color=%s];\n"
              id id (pr-str t)
              (if (is-leaf? n) "red" "grey"))]
       (reduce
        (fn [dot id'] (triple-graph->dot g id' (str dot (format "%d -> %d;\n" id id'))))
        (str dot d)
        (get-children n)))))

(defrecord AliasMemoryStore [store aliases]
  PModelUpdate
  (add-triple
    [_ [s p o]]
    (let [t [(or (u/canonical aliases s) s)
             (or (u/canonical aliases p) p)
             (or (u/canonical aliases o) o)]]
      (AliasMemoryStore. (add-triple store t) aliases)))
  (add-triples
    [_ triples]
    (loop [store store, xs triples]
      (if xs
        (let [[s p o] (first xs)
              t [(or (u/canonical aliases s) s)
                 (or (u/canonical aliases p) p)
                 (or (u/canonical aliases o) o)]]
          (recur (add-triple store t) (next xs)))
        (AliasMemoryStore. store aliases))))

  PModel
  (subject? [_ x]
    (subject? store (or (u/canonical aliases x) x)))
  (predicate? [_ x]
    (predicate? store (or (u/canonical aliases x) x)))
  (object? [_ x]
    (object? store (or (u/canonical aliases x) x)))
  (indexed? [_ x]
    (indexed? store (or (u/canonical aliases x) x)))
  (subjects [_] (subjects store))
  (predicates [_] (predicates store))
  (objects [_] (objects store))
  (model-size [_] (model-size store))

  PModelSelect
  (select
    [_] (select _ nil nil nil))
  (select
    [_ s p o]
    (let [s (or (u/canonical aliases s) s)
          p (or (u/canonical aliases p) p)
          o (or (u/canonical aliases o) o)]
      (select store s p o)))

  u/PUnionFind
  (canonical [_ p] (u/canonical aliases p))
  (canonical? [_ p] (u/canonical? aliases p))
  (component [_ p] (u/component aliases p))
  (disjoint-components [_] (u/disjoint-components aliases))
  (register [_ p] (AliasMemoryStore. store (u/register aliases p)))
  (unregister
    [_ p]
    (if (u/canonical? aliases p)
      (let [q (first (disj (u/component aliases p) p))
            aliases (u/unregister aliases p)
            store (if q (rewrite-alias store p q) store)]
        (AliasMemoryStore. store aliases))
      (AliasMemoryStore. store (u/unregister aliases p))))
  (unified? [_ p q] (u/unified? aliases p q))
  (union
    [_ p q]
    (if (and p q)
      (let [aliases (u/union aliases p q)
            canon (u/canonical aliases p)
            store (if (= p canon)
                    (rewrite-alias store q canon)
                    (rewrite-alias store p canon))]
        (AliasMemoryStore. store aliases))
      (err/illegal-arg! (str "aliases must be both non-nil values: " [p q])))))

(defrecord PlainDataset [models]
  PModelUpdate
  (add-triple [_ s]
    (add-triple _ :default s))
  (add-triple [_ g s]
    (update-in _ [:models g] add-triple s))
  (add-triples [_ triples]
    (add-triples _ :default triples))
  (add-triples [_ g triples]
    (update-in _ [:models g] add-triples triples))
  (remove-triple [_ s]
    (remove-triple _ :default s))
  (remove-triple [_ g s]
    (update-in _ [:models g] remove-triple s))
  (remove-triples [_ triples]
    (remove-triples _ :default triples))
  (remove-triples [_ g triples]
    (update-in _ [:models g] remove-triples triples))
  (remove-subject [_ s]
    (remove-subject _ :default s))
  (remove-subject [_ g s]
    (update-in _ [:models g] remove-subject s))

  PModel
  (subject? [_ x]
    (some #(subject? % x) (vals models)))
  (predicate? [_ x]
    (some #(predicate? % x) (vals models)))
  (object? [_ x]
    (some #(object? % x) (vals models)))
  (indexed? [_ x]
    (some #(indexed? % x) (vals models)))
  (subjects [_]
    (set (mapcat subjects (vals models))))
  (predicates [_]
    (set (mapcat predicates (vals models))))
  (objects [_]
    (set (mapcat objects (vals models))))
  (model-size [_]
    (reduce + (map model-size (vals models))))

  PModelSelect
  (select [_]
    (select _ nil nil nil))
  (select [_ s p o]
    (mapcat #(select % s p o) (vals models)))
  (select [_ g s p o]
    (if-let [g (models g)] (select g s p o)))

  PDataset
  (update-model [_ id m]
    (assoc-in _ [:models id] m))
  (remove-model [_ id]
    (update-in _ [:models] dissoc id))
  (get-model [_ id]
    (models id)))

(defrecord WatchedModel
    [model pre-hooks post-hooks]
  PModelWatch
  (add-pre-commit-hook
    [_ id hook-fn]
    (assoc-in _ [:pre-hooks id] hook-fn))
  (remove-pre-commit-hook
    [_ id]
    (update-in _ [:pre-hooks] dissoc id))
  (add-post-commit-hook
    [_ id hook-fn]
    (assoc-in _ [:post-hooks id] hook-fn))
  (remove-post-commit-hook
    [_ id]
    (update-in _ [:post-hooks] dissoc id))

  PModel
  (subject?
    [_ x] (subject? model x))
  (predicate?
    [_ x] (predicate? model x))
  (object?
    [_ x] (object? model x))
  (indexed?
    [_ x] (indexed? model x))
  (subjects
    [_] (subjects model))
  (predicates
    [_] (predicates model))
  (objects
    [_] (objects model))
  (model-size
    [_] (model-size model))

  PModelUpdate
  (add-triple
    [_ t]
    (let [t (reduce-kv
             (fn [t k v] (if-let [t' (v :add model t k)] t' (reduced nil)))
             t pre-hooks)]
      (if-not (seq (apply select model t))
        (let [m' (add-triple model t)
              m' (reduce-kv
                  (fn [m k v] (if-let [m' (v :add m t k)] m' (reduced nil)))
                  m' post-hooks)]
          (if m' (assoc _ :model m') _))
        _)))
  (add-triples
    [_ triples]
    (reduce add-triple model triples))
  (remove-triple
    [_ t]
    (let [t (reduce-kv
             (fn [t k v] (if-let [t' (v :remove model t k)] t' (reduced nil)))
             t pre-hooks)]
      (if (seq (apply select model t))
        (let [m' (remove-triple model t)
              m' (reduce-kv
                  (fn [m k v] (if-let [m' (v :remove m t k)] m' (reduced nil)))
                  m' post-hooks)]
          (if m' (assoc _ :model m') _))
        _)))
  (remove-triples
    [_ triples]
    (reduce add-triple model triples))
  (update-triple
    [_ t t'])
  (remove-subject
    [_ s])

  PModelSelect
  (select
    [_] (select model nil nil nil))
  (select
    [_ s p o] (select model s p o))
  (select-with-alts
    [_ s p o] (select-with-alts model s p o)))
(defn watched-model
 [model] (WatchedModel. model {} {}))

(defn plain-store
  [& triples]
  (add-triples (PlainMemoryStore. (hash-map) (hash-map) (hash-map) 0) triples))

(defn alias-store
  [store aliases & triples]
  (add-triples (reduce (partial apply u/union) (AliasMemoryStore. store (u/disjoint-set)) aliases) triples))

(defn triple-graph
  []
  (let [root' (TripleNode. root nil)]
    (add-node (TripleGraph. {} {} 0 0) root')))
(defn plain-dataset
  [& {:as models}]
  (PlainDataset. (assoc models :default (plain-store))))

#+clj (defn regexp? [x] (instance? java.util.regex.Pattern x))

(defn regexp-matches
  [ds f re]
  ;;(into #{} (filter #(if (string? %) (re-find x %)) (f ds)))
  (into #{} (filter #(re-find re (if (string? %) % (str %))) (f ds))))

(defn search
  [ds s p o]
  (->> [s p o]
       (map #(if (regexp? %2) (regexp-matches ds % %2) %2)
            [subjects predicates objects])
       (apply select-with-alts ds)))

(defn triple-seq-associative
  "Converts a single nested map into a seq of triples.
  Each key must have another map as value. Toplevel keys become
  subjects, value map keys predicates, inner map values objects. Each
  predicate key can define a seq of values to produce multiple
  triples."
  [coll]
  (mapcat
   (fn [[s v]]
     (mapcat
      (fn [[p o]]
        (if (sequential? o)
          (mapv (fn [o] [s p o]) o)
          [[s p o]]))
      v))
   coll))
(defn triple-seq-sequential
 [coll]
 (mapcat
  (fn [triple]
    (if (map? triple)
      (triple-seq triple)
      (->> triple
           (map #(if (sequential? %) % [%]))
           (apply d/cartesian-product))))
  coll))
(extend-protocol PTripleSeq
  #+clj  clojure.lang.Sequential
  #+cljs PersistentVector
  (triple-seq [_] (triple-seq-sequential _))
  #+cljs List
  #+cljs (triple-seq [_] (triple-seq-sequential _))
  #+cljs LazySeq
  #+cljs (triple-seq [_] (triple-seq-sequential _))
  #+cljs IndexedSeq
  #+cljs (triple-seq [_] (triple-seq-sequential _))
  #+clj  clojure.lang.IPersistentMap
  #+cljs PersistentHashMap
  (triple-seq [_] (triple-seq-associative _))
  #+cljs PersistentArrayMap
  #+cljs (triple-seq [_] (triple-seq-associative _)))

(extend-protocol PModelConvert
  #+clj  clojure.lang.Sequential
  #+cljs PersistentVector
  (as-model [_] (apply plain-store (triple-seq-sequential _)))
  #+cljs List
  #+cljs (as-model [_] (apply plain-store (triple-seq-sequential _)))
  #+cljs LazySeq
  #+cljs (as-model [_] (apply plain-store (triple-seq-sequential _)))
  #+cljs IndexedSeq
  #+cljs (as-model [_] (apply plain-store (triple-seq-sequential _)))
  #+clj  clojure.lang.IPersistentMap
  #+cljs PersistentHashMap
  (as-model [_] (apply plain-store (triple-seq-associative _)))
  #+cljs PersistentArrayMap
  #+cljs (as-model [_] (apply plain-store (triple-seq-associative _))))

(defn select-from-seq
  [_ s p o]
  (-> (if s
        (if p
          (if o
            (fn [[s' p' o']] (and (= s s') (= p p') (= o o')))
            (fn [[s' p']] (if (= s s') (= p p'))))
          (if o
            (fn [[s' _ o']] (if (= s s') (= o o')))
            (fn [t] (= s (first t)))))
        (if p
          (if o
            (fn [[_ p' o']] (if (= p p') (= o o')))
            (fn [t] (= p (nth t 1))))
          (if o
            (fn [t] (= o (nth t 2)))
            identity)))
      (filter _)))

(extend-protocol PModelSelect
  #+clj  clojure.lang.Sequential
  #+cljs PersistentVector
  (select [_] (select-from-seq _ nil nil nil))
  (select [_ s p o] (select-from-seq _ s p o))
  #+cljs List
  #+cljs (select [_] (select-from-seq _ nil nil nil))
  #+cljs (select [_ s p o] (select-from-seq _ s p o))
  #+cljs LazySeq
  #+cljs (select [_] (select-from-seq _ nil nil nil))
  #+cljs (select [_ s p o] (select-from-seq _ s p o))
  #+cljs IndexedSeq
  #+cljs (select [_] (select-from-seq _ nil nil nil))
  #+cljs (select [_ s p o] (select-from-seq _ s p o)))
