(ns thi.ng.trio.vocabs.utils
  (:require
   [thi.ng.trio.core :as api]
   #+clj [clojure.java.io :as io]
   #+clj [clojure.edn :as edn]
   [clojure.string :as str]))

(defn expand-prefixes
  [prefixes x]
  (if (string? x)
    (let [[p n] (str/split x #":")]
      (if-let [p (prefixes p)]
        (str p n)
        x))
    x))

(defn expand-prefixes-in-triple
  [prefixes [s p o]]
  (api/triple
   (expand-prefixes prefixes s)
   (expand-prefixes prefixes p)
   (expand-prefixes prefixes o)))

(defn find-prefix
  [prefixes ^String uri]
  (loop [[[pre puri] & more] (seq prefixes)]
    (when pre
      (if (== 0 (.indexOf uri ^String puri))
        [pre (subs uri (count puri))]
        (recur more)))))

(defn vocabs-from-model
  [prefixes graph]
  (->> graph
       (api/subjects)
       (reduce
        (fn [vocabs uri]
          (let [[p n] (find-prefix prefixes uri)]
            (if p
              (assoc-in vocabs [(keyword p) (keyword n)] uri)
              vocabs)))
        {})))

(defn triple-seq-with-prefixes
  [prefixes triples]
  (->> triples
       (api/triple-seq)
       (mapv #(expand-prefixes-in-triple prefixes %))))

#+clj
(defn load-vocab-triples
  [src]
  (let [{:keys [prefixes triples]}
        (->> src
             (io/input-stream)
             (slurp)
             (edn/read-string))]
    {:prefixes prefixes
     :triples (triple-seq-with-prefixes prefixes triples)}))

#+clj
(defn load-vocabs-as-model
  [src]
  (let [{:keys [prefixes triples]} (load-vocab-triples src)]
    (vocabs-from-model prefixes (api/as-model triples))))

(defn make-vocab
  [uri xs]
  (->> xs (map (fn [x] [x (str uri (name x))])) (into {})))
