(ns thi.ng.trio.query
  (:require
   [thi.ng.trio.core :as api]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.error :as err]
   [thi.ng.common.data.unionfind :as uf]
   [thi.ng.validate.core :as v]
   [clojure.set :as set]
   [clojure.core.reducers :as r]))

(defn compile-filter-fn
  "Takes a map of qvar bindings as keys and filter fns as values.
  Returns fn which accepts a single result map and returns true if all
  given filter fns succeed. Filter fns must accept a single arg, i.e.
  the qvar's value in the result map. Filter fns are only called if
  the qvar is bound. If a filter is defined for a qvar which does not
  ever appear in a solution, the related query will produce no
  results."
  [filters]
  (if filters
    (if (fn? filters)
      filters
      (fn [res]
        (every?
         (fn [[k f]] (let [rv (res k)] (if-not (nil? rv) (f rv) false)))
         filters)))))
(defn compile-bind-fn
  "Takes a map of qvar bindings as keys and bind fns as values.
  Returns a fn which accepts a single result map and returns updated
  map with injected/updated qvars. Bind fns must accept two args: the
  full result map and the value of the fn's associated qvars. The fn's
  return value is used as new binding for the qvar unless nil. Bind
  fns are called regardless if qvar is currently bound (non-nil). This
  is in order to allow for the injection of new vars into the
  solution."
  [bindings]
  (if bindings
    (if (fn? bindings)
      bindings
      (fn [res]
        (reduce-kv
         (fn [acc k f]
           (let [rv (f res (res k))]
             (if (nil? rv) acc (assoc acc k rv))))
         res bindings)))))
(defn query-opts
  "Takes a query options map and injects compiled versions for
  :filter & :bind keys."
  [opts]
  (-> opts
      (update-in [:filter] compile-filter-fn)
      (update-in [:bind] compile-bind-fn)))
(def ^:dynamic *auto-qvar-prefix* "?__q")

(defn qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  [x] (and (symbol? x) (re-matches #"^\?.*" (name x))))

(defn auto-qvar?
  "Returns true, if x is an auto-generated qvar
  (a symbol prefixed with *auto-qvar-prefix*)"
  [x] (and (symbol? x) (zero? (.indexOf ^String (name x) *auto-qvar-prefix*))))

(defn auto-qvar
  "Creates a new auto-named qvar (symbol)."
  [] (gensym *auto-qvar-prefix*))

(defn qvar-name
  [x] (-> x name (subs 1)))
(defn resolve-path-pattern
  "Takes a path triple pattern where the predicate is a seq of preds.
  Returns seq of query patterns with injected temp qvars for inbetween
  patterns. E.g.

      [?s [p1 p2 p3] ?o]
      => ([?s p1 ?__q0] [?__q0 p2 ?__q1] [?__q1 p3 ?o])"
  [[s p o]]
  (let [vars (->> auto-qvar
                  (repeatedly (dec (count p)))
                  (cons s))]
    (->> (concat (interleave vars p) [o])
         (d/successive-nth 3 2))))

(defn resolve-patterns
  [patterns]
  (mapcat
   (fn [[_ p :as t]]
     (if (vector? p)
       (resolve-path-pattern t)
       [t]))
   patterns))
(defn produce-patterns-with-bound-vars
  "Takes a triple pattern (possibly with variables) and a map of
  possible value sets (each *must* be a set or single value) for each var.
  Produces lazy seq of resulting triple query patterns using cartesian
  product of all values.

      (produce-patterns-with-bound-vars
        [?a :type ?b]
        {?a #{\"me\" \"you\"} ?b #{\"foo\" \"bar\"})
      => ((\"me\" :type \"foo\") (\"me\" :type \"bar\")
          (\"you\" :type \"foo\") (\"you\" :type \"bar\"))"
  [[s p o] bindings]
  (let [s (or (bindings s) s)
        p (or (bindings p) p)
        o (or (bindings o) o)]
    (if (some set? [s p o])
      (d/cartesian-product
       (if (set? s) s #{s}) (if (set? p) p #{p}) (if (set? o) o #{o}))
      [[s p o]])))
(defn sort-patterns
  "Sorts a seq of triple patterns in dependency order using any
  re-occuring vars. Triples with least qvars will be in head
  position."
  [patterns]
  (let [q (map #(let [v (d/filter-tree qvar? %)] [(count v) v %]) patterns)
        singles (->> q (filter #(= 1 (first %))) (mapcat second) set)]
    (->> q
         (sort-by (fn [[c v]] (- (* c 4) (count (filter singles v)))))
         (map peek))))
(defn triple-verifier
  "Takes a triple pattern (potentially with vars) and 3 booleans to
  indicate which SPO is a var. Returns fn which accepts a result triple
  and returns false if any of the vars clash (e.g. a qvar is used multiple
  times but result has different values in each position or likewise, if
  different vars relate to same values)."
  [[ts tp to] vars varp varo]
  (cond
   (and vars varp varo) (cond
                         (= ts tp to) (fn [[rs rp ro]] (= rs rp ro))
                         (= ts tp) (fn [[rs rp ro]] (and (= rs rp) (not= rs ro)))
                         (= ts to) (fn [[rs rp ro]] (and (= rs ro) (not= rs rp)))
                         (= tp to) (fn [[rs rp ro]] (and (= rp ro) (not= rs rp)))
                         :default (constantly true))
   (and vars varp) (if (= ts tp)
                     (fn [[rs rp]] (= rs rp))
                     (fn [[rs rp]] (not= rs rp)))
   (and vars varo) (if (= ts to)
                     (fn [[rs _ ro]] (= rs ro))
                     (fn [[rs _ ro]] (not= rs ro)))
   (and varp varo) (if (= tp to)
                     (fn [[_ rp ro]] (= rp ro))
                     (fn [[_ rp ro]] (not= rp ro)))
   :default (constantly true)))
(defn unique-bindings?
  "Returns true if all values in the given map are unique, i.e.
  no two keys are mapped to the same value."
  [map] (== (count (into #{} (vals map))) (count map)))

(defn accumulate-result-vars
  "Takes a seq of query result maps and returns a map of all found
  qvars as keys and their value sets."
  ([res] (accumulate-result-vars {} nil res))
  ([acc vars res]
     (let [acc (reduce
                (fn [acc b]
                  (merge-with
                   (fn [a b] (if (set? a) (conj a b) (hash-set a b)))
                   acc (if vars (select-keys b vars) b)))
                acc res)]
       (reduce-kv
        (fn [acc k v] (if (set? v) acc (assoc acc k #{v})))
        acc acc))))

(defn select-renamed-keys
  "Similar to clojure.core/select-keys, but instead of key seq takes a
  map of keys to be renamed (keys in that map are the original keys to
  be selected, their values the renamed keys in returned map)."
  [ret map aliases]
  (loop [ret ret, keys aliases]
    (if keys
      (let [kv (first keys)
            entry (find map (key kv))]
        (recur
         (if entry
           (assoc ret (val kv) (val entry))
           ret)
         (next keys)))
      ret)))

(defn order-asc
  [vars res]
  (if (coll? vars)
    (sort-by (fn [r] (reduce #(conj % (r %2)) [] vars)) res)
    (sort-by (fn [r] (get r vars)) res)))

(defn order-desc
  [vars res]
  (if (coll? vars)
    (sort-by
     (fn [r] (reduce #(conj % (r %2)) [] vars))
     #(- (compare % %2))
     res)
    (sort-by #(get % vars) #(- (compare % %2)) res)))

(defn distinct-result-set
  [res]
  (->> res
       (reduce
        (fn [acc r]
          (let [vs (set (vals r))]
            (if (acc vs) acc (assoc acc vs r))))
        {})
       (vals)))

(defn keywordize-result-vars
  [res]
  (map
   (fn [r] (into {} (map (fn [[k v]] [(-> k qvar-name keyword) v]) r)))
   res))
(defn triples->dot
  "Takes a seq of triples and returns them as digraph spec in
  Graphviz .dot format."
  [triples]
  (apply
   str
   (concat
    "digraph G {\n"
    "node[color=\"black\",style=\"filled\",fontname=\"Inconsolata\",fontcolor=\"white\"];\n"
    "edge[fontname=\"Inconsolata\",fontsize=\"9\"];\n"
    (map
     (fn [t]
       (let [[s p o] (map #(if (string? %) % (pr-str %)) t)]
         (str "\"" s "\" -> \"" o "\" [label=\"" p "\"];\n")))
     triples)
    "}")))

(defn join
  [a b]
  (->> (set/join a b)
       (mapcat
        (fn [k]
          (if (unique-bindings? k)
            [k])))))

(defn join-optional
  [a b]
  (loop [old (transient #{}), new (transient #{}), kb b]
    (if kb
      (let [kb' [(first kb)]
            [old new] (loop [old old, new new, ka a]
                        (if ka
                          (let [ka' (first ka)
                                j (first (set/join [ka'] kb'))]
                            (if j
                              (recur (conj! old ka') (conj! new j) (next ka))
                              (recur old new (next ka))))
                          [old new]))]
        (recur old new (next kb)))
      (let [new (persistent! new)]
        (if (seq new)
          (into (apply disj (set a) (persistent! old)) new)
          a)))))

 (defn minus
   [a b]
   (let [vars (accumulate-result-vars b)]
     (reduce-kv
      (fn [res k v]
        (let [v (if (coll? v) v [v])]
          (reduce
           (fn [res v] (vec (remove (fn [r] (= (r k) v)) res))) res v)))
      a vars)))

 (defn union
   [a b] (set (concat a b)))

(defn unbound-var?
  [bindings v? x] (and v? (not (bindings x))))

(defn bind-translator
  [vs? vp? vo? [s p o]]
  (if vs?
    (if vp?
      (if vo?
        (fn [[s' p' o']] {s s' p p' o o'})
        (fn [[s' p']] {s s' p p'}))
      (if vo?
        (fn [[s' _ o']] {s s' o o'})
        (fn [r] {s (first r)})))
    (if vp?
      (if vo?
        (fn [[_ p' o']] {p p' o o'})
        (fn [r] {p (r 1)}))
      (if vo?
        (fn [r] {o (r 2)})
        (fn [_] {})))))

(defn unbound-vars-translator
  [bindings vs? vp? vo? [s p o]]
  (if (unbound-var? bindings vs? s)
    (if (unbound-var? bindings vp? p)
      (if (unbound-var? bindings vo? o)
        (fn [p] [nil nil nil])
        (fn [p] [nil nil (p 2)]))
      (if (unbound-var? bindings vo? o)
        (fn [p] [nil (p 1) nil])
        (fn [p] (assoc p 0 nil))))
    (if (unbound-var? bindings vp? p)
      (if (unbound-var? bindings vo? o)
        (fn [p] [(p 0) nil nil])
        (fn [p] (assoc p 1 nil)))
      (if (unbound-var? bindings vo? o)
        (fn [p] (assoc p 2 nil))
        identity))))

(defn select-with-bindings
  [ds bindings opts [s p o :as t]]
  (let [patterns (produce-patterns-with-bound-vars t bindings)
        _        (prn :patterns patterns)
        vs?      (qvar? s), vp? (qvar? p), vo? (qvar? o)
        vmap     (bind-translator vs? vp? vo? t)
        pmap     (unbound-vars-translator bindings vs? vp? vo? t)
        verify   (triple-verifier t vs? vp? vo?)
        flt      (compile-filter-fn (opts :filter))
        res-fn   (if flt
                   #(if (verify %)
                      (let [vbinds (vmap %)]
                        (if (flt vbinds) vbinds)))
                   #(if (verify %) (vmap %)))]
    (loop [acc (transient []), ps patterns]
      (if ps
        (let [[qs qp qo] (pmap (vec (first ps)))
              _   (prn :select qs qp qo)
              res (api/select ds qs qp qo)]
          (if (seq res)
            (recur
             (loop [acc acc, res res]
               (if res
                 (let [r (res-fn (first res))]
                   (if r
                     (recur (conj! acc r) (next res))
                     (recur acc (next res))))
                 acc))
             (next ps))
            (recur acc (next ps))))
        (persistent! acc)))))

(defn select-join
  ([ds patterns]
     (select-join ds {} {} patterns))
  ([ds bindings {bind :bind flt :filter opt? :optional?} patterns]
     (let [[p & ps] (sort-patterns patterns)
           res      (select-with-bindings ds bindings {} p)
           join-fn  (if opt? join-optional join)
           flt      (compile-filter-fn flt)
           bind     (compile-bind-fn bind)]
       (if (seq res)
         (loop [res res, ps ps]
           (if ps
             (let [binds (merge-with into (accumulate-result-vars res) bindings)
                   r' (select-with-bindings ds binds {} (first ps))
                   res (if (seq r') (join-fn res r'))]
               (if (seq res)
                 (recur res (next ps))))
             (cond->> res
                      flt (filter flt)
                      bind (map bind))))))))

(defn subvec-slices
  "Takes a min & max count and returns function accepting a vector as
  single arg. When called, returns vector of subvec slices each starting
  at index 0 and with an increasing length from min to max."
  [n1 n2]
  (fn [path]
    (mapv #(subvec path 0 %) (range n1 (inc (min (count path) n2))))))

(defn dfs-forward*
  [ds s p sv acc min max]
  (if (<= (count acc) max)
    (let [acc (conj acc sv)
          o (auto-qvar)
          r (select-with-bindings ds {s sv} {} [s p o])]
      (if (seq r)
        (let [visited (set acc)
              ovals (filter (comp not visited) (set (map o r)))]
          (if (seq ovals)
            (                      mapcat
                   (fn [ov] (dfs-forward* ds o p ov acc min max))
                   ovals)
            [acc]))
        (if (> (count acc) min) [acc])))
    [acc]))

(defn dfs-backward*
  [ds o p ov acc min max]
  (if (<= (count acc) max)
    (let [acc (conj acc ov)
          s (auto-qvar)
          r (select-with-bindings ds {o ov} {} [s p o])]
      (if (seq r)
        (let [visited (set acc)
              svals (filter (comp not visited) (set (map s r)))]
          (if (seq svals)
            (                      mapcat
                   (fn [sv] (dfs-backward* ds s p sv acc min max))
                   svals)
            [acc]))
        (if (> (count acc) min) [acc])))
    [acc]))

(defn select-transitive
  ([ds [s p o :as t]]
     (if (vector? p)
       (select-transitive ds t (count p) (count p))
       (select-transitive ds t 1 1000000)))
  ([ds [s p o] mind maxd]
     (let [mind (max mind 1)
           maxd (max maxd 1)]
       (if (= mind maxd)
         (let [p (if (vector? p) p [p])
               p (take mind (cycle p))]
           (select-join ds {} {} (resolve-path-pattern [s p o])))
         (let [vs? (qvar? s)
               vo? (qvar? o)
               v (auto-qvar)
               conf (cond
                     (and vs? vo?) {:s s ;; [?s p ?o]
                                    :o v
                                    :bmap {}
                                    :lookup (fn [b] [(b v) (b s)])
                                    :bind (fn [p] {s (first p) o (peek p)})
                                    :search dfs-forward*}
                     vo?           {:s v ;; [x p ?o]
                                    :o o
                                    :bmap {v s}
                                    :lookup (fn [b] [(b o) (b v)])
                                    :bind (fn [p] {o (peek p)})
                                    :search dfs-forward*}
                     vs?           {:s s ;; [?s p x]
                                    :o v
                                    :bmap {v o}
                                    :lookup (fn [b] [(b s) (b v)])
                                    :bind (fn [p] {s (peek p)})
                                    :search dfs-backward*})
               {:keys [bind search]} conf
               slices (subvec-slices (inc mind) (inc maxd))]
           (->> (select-with-bindings ds (:bmap conf) {} [(:s conf) p (:o conf)])
                (r/map    (:lookup conf))
                (r/mapcat #(search ds v p (% 0) [(% 1)] mind maxd))
                (r/mapcat slices)
                (r/map    bind)
                (into #{})))))))

(def validator-select
  {:select    (v/alts
               [(v/symbol) (v/keyword) (v/sequential)]
               "must be a qvar, keyword or seq")
   :from      (v/optional
               (v/alts
                [(v/sequential) (v/satisfies api/PModel)]
                "must satisfy the thi.ng.trio.core.PModel protocol"))
   :query     {:* (v/map)}
   :bind      (v/optional
               (v/alts
                [(v/map) (v/function)]
                "must be a map or fn"))
   :group     (v/optional
               (v/alts
                [(v/symbol) (v/sequential) (v/function)]
                "must be a qvar, qvar seq or fn"))
   :aggregate (v/optional (v/map))
   :filter    (v/optional
               (v/alts
                [(v/map) (v/function)]
                "must be a map or fn"))})
(defn validate
  [q spec]
  (try
    (let [[_ err] (v/validate q spec)]
      (when err (err/throw! (str "error compiling query: " err))))
    (catch Exception e
      (err/throw! (str "error compiling query: " (.getMessage e))))))

(defmulti compile-query-step
  (fn [qfn q type] type))
(defmethod compile-query-step :select
  [qfn {:keys [select bind order-asc order-desc] :as q} _]
  (prn :select select)
  (validate q validator-select)
  (cond->
   (compile-query-step qfn q :query-dispatch)
   bind       (compile-query-step bind :bind)
   order-asc  (compile-query-step order-asc :order-asc)
   order-desc (compile-query-step order-desc :order-desc)
   true       (compile-query-step q :project-vars)))
(defmethod compile-query-step :construct-triples
  [qfn {:keys [construct] :as q} _]
  (fn [res]
    (->> res
         qfn
         (mapcat
          (fn [r]
            (map
             (fn [[s p o]]
               (let [s (if (qvar? s) (r s) s)
                     p (if (qvar? p) (r p) p)
                     o (if (qvar? o) (r o) o)]
                 (if (and s p o) (api/triple s p o))))
             construct)))
         (filter identity)
         (set))))

(defmethod compile-query-step :construct
  [qfn q _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :group)
              (assoc :select :*))]
    (-> qfn
        (compile-query-step q :select)
        (compile-query-step q :construct-triples))))
(defmethod compile-query-step :ask
  [qfn q _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :bind :group :aggregate)
              (assoc :select :*))
        qfn (compile-query-step qfn q :select)]
    (fn [res] (if (seq (qfn res)) true false))))
(defmethod compile-query-step :describe
  [qfn {:keys [from describe] :as q} _]
  (let [q (-> q
              (dissoc :order-asc :order-desc :group :aggregate)
              (assoc :select :*))
        qfn (compile-query-step qfn q :select)
        describe (if (sequential? describe) describe [describe])]
    (fn [res]
      (let [vars (select-keys (accumulate-result-vars (qfn res)) describe)]
        (if (seq vars)
          (reduce
           (fn [acc v]
             (let [vals (vars v)]
               (-> acc
                   (into (mapcat #(api/select from % nil nil) vals))
                   (into (mapcat #(api/select from nil nil %) vals)))))
           #{} (keys vars)))))))
(defn query-step
  [qfn join-fn {:keys [from values] :as q} patterns]
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)
            b (select-join from values opts patterns)
            res' (if (= ::empty a) b (join-fn a b))]
        res'))))

(defmethod compile-query-step :join
  [qfn {:keys [from values where] :as q} _]
  (prn :join where values)
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)]
        (if (or (= ::empty a) (seq a))
          (let [b (select-join from values opts where)
                res' (if (= ::empty a) b (join a b))]
            res')
          a)))))

(defmethod compile-query-step :minus
  [qfn {:keys [from values minus] :as q} _]
  (prn :minus minus values)
  (let [opts (query-opts q)]
    (fn [res]
      (let [a (qfn res)]
        (if (seq a)
          (thi.ng.trio.query/minus a (select-join from values opts minus))
          a)))))

(defmethod compile-query-step :optional
  [qfn {:keys [from values optional] :as q} _]
  (prn :join-opt optional values)
  (query-step qfn join-optional q optional))

(defmethod compile-query-step :union
  [qfn {:keys [from values union] :as q} _]
  (prn :union union values)
  (query-step qfn thi.ng.trio.query/union q union))
(defmethod compile-query-step :query-dispatch
  [qfn {:keys [from values query]} _]
  (prn :query-dispatch query)
  (if (and (sequential? query) (seq query))
    (loop [qfn qfn, query query]
      (if query
        (let [q (first query)
              from (or (:from q) from)
              q (cond
                 (satisfies? api/PModel from)
                 (assoc q :from from)
                 (satisfies? api/PModelConvert from)
                 (assoc q :from (api/as-model from)))
              q (if (:values q) q (assoc q :values (or values {})))
              qfn (cond
                   (:where q)    (compile-query-step qfn q :join)
                   (:optional q) (compile-query-step qfn q :optional)
                   (:minus q)    (compile-query-step qfn q :minus)
                   (:union q)    (compile-query-step qfn q :union))]
          (recur qfn (next query)))
        qfn))
    (err/illegal-arg! "Query spec must contain :query key w/ seq of sub-queries")))
(defmethod compile-query-step :bind
  [qfn bind _]
  (prn :bindings bind)
  (let [bind (compile-bind-fn bind)]
    (fn [res] (map bind (qfn res)))))

(defmethod compile-query-step :order-asc
  [qfn order _]
  (prn :order-asc order)
  (fn [res] (order-asc order (qfn res))))

(defmethod compile-query-step :order-desc
  [qfn order _]
  (prn :order-desc order)
  (fn [res] (order-desc order (qfn res))))
(defn renamed-vars
  [binding-map]
  (reduce-kv
   (fn [vars k v]
     (if (and (map? v) (:as v))
       (assoc vars (:as v) v)
       (assoc vars k v)))
   {} binding-map))

(defn selected-var-keys
  [qvars]
  (reduce
   (fn [vars v]
     (if (map? v)
       (merge vars v)
       (assoc vars v v)))
   {} qvars))

(defn parse-var-bind-spec
  [v spec] (if (map? spec) [(or (:use spec) v) (:fn spec)] [nil spec]))

(defn compute-select-bindings
  [res svars]
  (reduce-kv
   (fn [acc v spec]
     (let [[k f] (parse-var-bind-spec v spec)
           v' (if k
                (if (fn? f) (f (res k)) (res k))
                (f res))]
       (if v'
         (assoc acc v v')
         acc)))
   {} svars))

(defn validate-selected-vars
  [svars]
  (if (svars :*)
    (if (> (count svars) 1)
      (err/illegal-arg! (str "can't use :* selector with other vars: " (vals svars)))
      (dissoc svars :*))
    svars))

(defmethod compile-query-step :project-vars
  [qfn {:keys [select group aggregate filter] :as q} _]
  (prn :project-vars select)
  (let [select     (if (sequential? select) select [select])
        svars      (selected-var-keys select)
        catch-all? (svars :*)
        svars      (validate-selected-vars svars)
        agg-only?  (every? (set (keys aggregate)) (keys svars))
        filter     (compile-filter-fn filter)]
    (if group
      (compile-query-step qfn [group aggregate svars filter agg-only? catch-all?] :project-groups)
      (if (seq aggregate)
        (if (and agg-only? (not catch-all?))
          (compile-query-step qfn [aggregate svars filter] :project-vars-aggregates)
          (compile-query-step qfn [aggregate svars catch-all? filter] :project-vars-mixed))
        (if catch-all?
          (compile-query-step qfn filter :project-vars-all)
          (compile-query-step qfn [svars filter] :project-vars-simple))))))
(defn compile-group-fn
  [group]
  (if (fn? group)
    group
    (if (sequential? group)
      (fn [res] (reduce #(conj % (res %2)) [] group))
      (fn [res] (res group)))))

(defn compute-aggregates
  [vars vals]
  (reduce-kv
   (fn [inner v spec]
     (let [[k f] (parse-var-bind-spec v spec)]
       (if k
         (assoc inner v (f (map #(get % k) vals)))
         (assoc inner v (f vals)))))
   {} vars))

(defn project-aggregates-only
  [avars svars filter]
  (fn [vals]
    (let [agg (compute-aggregates avars vals)]
      (if (or (nil? filter) (filter agg))
        (compute-select-bindings agg svars)))))

(defn project-aggregates-mixed
  [avars svars all? flt]
  (fn [vals]
    (let [agg  (compute-aggregates avars vals)
          vals (map #(merge % agg) vals)
          vals (if flt (filter flt vals) vals)]
      (if (seq vals)
        (if all?
          vals
          (map #(compute-select-bindings % svars) vals))))))

(defn project-aggregates-simple
  [vars flt]
  (fn [vals]
    (let [vals (if flt (filter flt vals) vals)]
      (if (seq vals)
        (map #(compute-select-bindings % vars) vals)))))

(defn project-aggregates-all
  [flt]
  (fn [vals]
    (let [vals (if flt (filter flt vals) vals)]
      (if (seq vals) vals))))

(defmethod compile-query-step :project-vars-aggregates
  [qfn [avars svars filter] _]
  (prn :aggregates-only avars filter)
  (let [project-fn (project-aggregates-only avars svars filter)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defmethod compile-query-step :project-vars-mixed
  [qfn [avars svars all? flt] _]
  (prn :mixed-aggregates avars svars all? flt)
  (let [project-fn (project-aggregates-mixed avars svars all? flt)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defmethod compile-query-step :project-vars-simple
  [qfn [vars flt] _]
  (prn :simple-aggregates vars flt)
  (let [project-fn (project-aggregates-simple vars flt)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defmethod compile-query-step :project-vars-all
  [qfn flt _]
  (prn :all-aggregates flt)
  (let [project-fn (project-aggregates-all flt)]
    (fn [res] (->> (qfn res) (project-fn)))))

(defn project-groups-with
  [project-fn res]
  (reduce-kv
   (fn [acc k vals]
     (if-let [v (project-fn vals)]
       (assoc acc k v)
       acc))
   {} res))

(defmethod compile-query-step :group-aggregates-only
  [qfn [avars svars filter] _]
  (prn :aggregates-only avars filter)
  (let [project-fn (project-aggregates-only avars svars filter)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :group-aggregates-mixed
  [qfn [avars svars all? flt] _]
  (prn :mixed-aggregates avars svars all? flt)
  (let [project-fn (project-aggregates-mixed avars svars all? flt)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :group-aggregates-simple
  [qfn [vars flt] _]
  (prn :simple-aggregates vars flt)
  (let [project-fn (project-aggregates-simple vars flt)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :group-aggregates-all
  [qfn flt _]
  (prn :all-aggregates flt)
  (let [project-fn (project-aggregates-all flt)]
    (fn [res] (->> (qfn res) (project-groups-with project-fn)))))

(defmethod compile-query-step :project-groups
  [qfn [group aggregate svars filter agg-only? catch-all?] _]
  (prn :project-groups group aggregate)
  (let [group (compile-group-fn group)
        qfn   (fn [res] (group-by group (qfn res)))]
    (if (seq aggregate)
      (if (and agg-only? (not catch-all?))
        (compile-query-step qfn [aggregate svars filter] :group-aggregates-only)
        (compile-query-step qfn [aggregate svars catch-all? filter] :group-aggregates-mixed))
      (if catch-all?
        (compile-query-step qfn filter :group-aggregates-all)
        (compile-query-step qfn [svars filter] :group-aggregates-simple)))))

(defn compile-query
  [q]
  (if-let [type (some #{:select :ask :construct :describe} (keys q))]
    (compile-query-step identity q type)
    (err/illegal-arg! "Unsupported query type")))

(defn query
  [q]
  (let [res ((if (fn? q) q (compile-query q)) ::empty)]
    (if (seq res) res)))

;;;;;;;;;;;; This file autogenerated from src/cljx/thi/ng/trio/query.cljx
