(ns thi.ng.morphogen.core
  #+cljs (:require-macros [thi.ng.macromath.core :as mm])
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.core.vector :as v :refer [vec3]]
   [thi.ng.geom.core.utils :as gu]
   [thi.ng.geom.types.utils :as tu]
   [thi.ng.geom.circle :as c]
   [thi.ng.geom.polygon :as p]
   [thi.ng.geom.quad :as q]
   [thi.ng.geom.plane :as pl]
   [thi.ng.geom.aabb :as a]
   [thi.ng.geom.cuboid :as cu]
   [thi.ng.geom.basicmesh :as bm]
   [thi.ng.geom.mesh.ops :as ops]
   [thi.ng.common.data.core :as d]
   [thi.ng.common.math.core :as m :refer [*eps* TWO_PI PI HALF_PI]]
   #+clj [thi.ng.macromath.core :as mm]
   #+clj [thi.ng.geom.mesh.io :as mio]
   #+clj [clojure.java.io :as io]
   #+clj [clojure.pprint :refer [pprint]]
   ))

(declare operator child-path)

(def face-opposite {:e :w, :w :e, :n :s, :s :n, :f :b, :b :f})
(defn classify-node-at
  [op-tree cursor]
  (let [n (get-in op-tree (child-path cursor))]
    (cond
     (:op n) :operator
     n :leaf
     :else nil)))
(defn quad-normal
  "Takes 4 points (or a seq of 4 points), returns vector perdendicular
  to the 2 diagonals of the quad"
  ([[a b c d]] (quad-normal a b c d))
  ([a b c d] (gu/ortho-normal (g/- c a) (g/- d b))))

(defn offset-face-points
  [[a b c d e f g h] side n]
  (case side
    :e [a b (g/+ c n) (g/+ d n) e f (g/+ g n) (g/+ h n)]
    :w [(g/+ a n) (g/+ b n) c d (g/+ e n) (g/+ f n) g h]
    :n [a b c d (g/+ e n) (g/+ f n) (g/+ g n) (g/+ h n)]
    :s [(g/+ a n) (g/+ b n) (g/+ c n) (g/+ d n) e f g h]
    :f [a (g/+ b n) (g/+ c n) d e (g/+ f n) (g/+ g n) h]
    :b [(g/+ a n) b c (g/+ d n) (g/+ e n) f g (g/+ h n)]))

(defn child-path
  "Takes a seq of child indices and constructs a lookup path/vector
  for them by interleaving `:out` in the seq:
      (child-path [1 0 2]) => [:out 1 :out 0 :out 2]"
  [path] (vec (interleave (repeat :out) path)))

(defn inject
  "Almost like assoc-in, but transforms lookup path with `child-path`."
  [t path t']
  (assoc-in t (child-path path) t'))

(defn apply-recursively
  "Recursively injects tree into itself `n` times, starting at given
  child path. At each subsequent level, the original tree given is
  injected at index `id` of the `:out` child node vector. The initial
  path is simply given as a seq of indices and will be translated into
  an actual lookup path using the `child-path` fn."
  [tree n path id]
  (loop [t' tree, path (child-path path), n (dec n)]
    (if (pos? n)
      (recur (assoc-in t' path tree) (into path [:out id]) (dec n))
      t')))

(defn map-leaves
  "Takes a fn and operator tree, applies f to all leaf nodes. The fn
  must accept 3 args: the leaf's parent node, the child index of the
  leaf in the parent and the tree depth. The leaf will be replaced
  with the fn's return value."
  ([f tree] (map-leaves f tree 0))
  ([f tree depth]
     (->> (:out tree)
          (interleave (range))
          (partition 2)
          (reduce
           (fn [acc [i c]]
             (cond
              (seq (:out c)) (assoc-in acc [:out i] (map-leaves f c (inc depth)))
              (map? c) (assoc-in acc [:out i] (f acc i depth))
              :default acc))
           tree))))

(defprotocol PNode
  (parent [_] "Returns the node's parent or `nil` if node is the root")
  (tree-depth [_]  "Returns the node's tree depth")
  (face-vertices [_ face] "Returns vertices for given face ID")
  (face-topology [_] "Returns number of vertices used for each face in the node"))

(defrecord BoxNode [points parent depth]
PNode
(parent [_] parent)
(tree-depth [_] depth)
(face-vertices
 [{[a b c d e f g h] :points} side]
 (case side
   :e [c d h g]
   :w [a b f e]
   :n [e f g h]
   :s [a d c b]
   :f [b c g f]
   :b [d a e h]))
(face-topology [_] 4)

g/PBounds
(bounds [_] (tu/bounding-box (:points _)))
(width  [_] (gu/axis-range 0 (:points _)))
(height [_] (gu/axis-range 1 (:points _)))
(depth  [_] (gu/axis-range 2 (:points _)))

g/PVertexAccess
(vertices [_] points)

g/PFaceAccess
(faces
 [{[a b c d e f g h] :points}]
 [[b c g f]  ;; front
  [d a e h]  ;; back
  [a b f e]  ;; west
  [c d h g]  ;; east
  [e f g h]  ;; north
  [a d c b]] ;; south
 )
g/PSubdivide
(subdivide
 [_ {:keys [cols rows slices] :or {cols 1 rows 1 slices 1}}]
 (let [ru (d/successive-nth 2 (m/norm-range cols))
       rv (d/successive-nth 2 (m/norm-range rows))
       rw (d/successive-nth 2 (m/norm-range slices))
       map-p (fn [p] (->> p (gu/map-trilinear points) (map #(m/roundto % *eps*)) vec3))]
   (for [[w1 w2] rw, [v1 v2] rv, [u1 u2] ru]
     (mapv map-p [[u1 v1 w1] [u1 v1 w2] [u2 v1 w2] [u2 v1 w1]
                  [u1 v2 w1] [u1 v2 w2] [u2 v2 w2] [u2 v2 w1]]))))
)
(defn seed-box
  [x]
  (let [points (cond
                (number? x) (g/vertices (a/aabb x))
                (sequential? x) (mapv vec3 x)
                (satisfies? g/PVertexAccess x) (g/vertices x))]
    (BoxNode. points nil 0)))

(defmulti operator
  (fn [g-node op-node]
   (if (:op op-node) [(type g-node) (:op op-node)])))

;; leaf node operator (no-op)
(defmethod operator nil [_ _] nil)

(defmethod operator [BoxNode :sd]
  [^BoxNode node {:keys [args]}]
  (let [depth (inc (tree-depth node))]
    (->> (g/subdivide node args)
         (mapv #(BoxNode. % node depth)))))

(defn subdivide-inset___
  [[a b c d e f g h :as points] {i :inset dir :dir :or {i 0.1 dir :y}}]
  (let [ii (- 1.0 i)
        map-points (fn [base uv]
                     (mapcat
                      (fn [[u v]]
                        [(gu/map-trilinear points (assoc (vec3) uv [u v]))
                         (gu/map-trilinear points (assoc base uv [u v]))])
                      [[i i] [i ii] [ii ii] [ii i]]))]
    (condp = dir
      :x (let [[a1 a2 b1 b2 c1 c2 d1 d2] (map-points v/V3X :yz)]
           [[b c d a b1 b2 a2 a1]
            [c1 c2 d2 d1 f g h e]
            [b c b2 b1 f g c2 c1]
            [a1 a2 d a d1 d2 h e]
            [b1 b2 a2 a1 c1 c2 d2 d1]])
      :y (let [[a1 a2 b1 b2 c1 c2 d1 d2] (map-points v/V3Y :xz)]
           [[b1 b c c1 b2 f g c2]
            [a a1 d1 d e a2 d2 h]
            [a b b1 a1 e f b2 a2]
            [d1 c1 c d d2 c2 g h]
            [a1 b1 c1 d1 a2 b2 c2 d2]])
      :z (let [[a1 a2 b1 b2 c1 c2 d1 d2] (map-points v/V3Z :xy)]
           [[a b c d a1 a2 d2 d1]
            [b1 b2 c2 c1 e f g h]
            [a b a2 a1 e f b2 b1]
            [d1 d2 c d c1 c2 g h]
            [a1 a2 d2 d1 b1 b2 c2 c1]]))))

(defn subdivide-inset
  [[a b c d e f g h :as points] {i :inset dir :dir :or {i 0.1 dir :y}}]
  (case dir
    :x (let [[a2 b2 f2 e2] (q/inset-quad [a b f e] i)
             [c2 d2 h2 g2] (q/inset-quad [c d h g] i)]
         [[b c d a b2 c2 d2 a2]
          [f2 g2 h2 e2 f g h e]
          [b c c2 b2 f g g2 f2]
          [a2 d2 d a e2 h2 h e]
          [b2 c2 d2 a2 f2 g2 h2 e2]])
    :y (let [[a2 b2 c2 d2] (q/inset-quad [a b c d] i)
             [e2 f2 g2 h2] (q/inset-quad [e f g h] i)]
         [[b2 b c c2 f2 f g g2]
          [a a2 d2 d e e2 h2 h]
          [a b b2 a2 e f f2 e2]
          [d2 c2 c d h2 g2 g h]
          [a2 b2 c2 d2 e2 f2 g2 h2]])
    :z (let [[a2 d2 h2 e2] (q/inset-quad [a d h e] i)
             [b2 c2 g2 f2] (q/inset-quad [b c g f] i)
             p' [a2 b2 c2 d2 e2 f2 g2 h2]]
         (if (some nil? p')
           (do
             ;;(prn points)
             ;;(prn p')
             ;;(prn "----")
             [points])
           [[a b c d a2 b2 c2 d2]
            [e2 f2 g2 h2 e f g h]
            [a b b2 a2 e f f2 e2]
            [d2 c2 c d h2 g2 g h]
            [a2 b2 c2 d2 e2 f2 g2 h2]]))))

(defmethod operator [BoxNode :sd-inset]
  [^BoxNode node {:keys [args]}]
  (let [depth (inc (tree-depth node))]
    (->> (subdivide-inset (:points node) args)
         (mapv #(BoxNode. % node depth)))))

(defn- offset
  [^thi.ng.geom.core.vector.Vec3 a ^thi.ng.geom.core.vector.Vec3 b ^double len]
  (let [^thi.ng.geom.core.vector.Vec3 d (g/- a b)
        m (/ len (g/mag d))]
    (thi.ng.geom.core.vector.Vec3.
     (mm/madd (.-x d) m (.-x a))
     (mm/madd (.-y d) m (.-y a))
     (mm/madd (.-z d) m (.-z a)))))

(defmethod operator [BoxNode :extrude]
  [^BoxNode node {{:keys [dir len] :or {dir :n len 1.0}} :args}]
  (let [c1 (gu/centroid (face-vertices node dir))
        c2 (gu/centroid (face-vertices node (face-opposite dir)))
        n (g/normalize (g/- c1 c2) len)]
    [(BoxNode.
      (offset-face-points (:points node) dir n)
      node (inc (tree-depth node)))]))

(defmethod operator [BoxNode :ext-prop]
  [{[a b c d e f g h] :points :as node}
   {{:keys [dir len] :or {dir :n len 1.0}} :args}]
  [(BoxNode.
    (case dir
      :e [a b (offset c b len) (offset d a len)
          e f (offset g f len) (offset h e len)]
      :w [(offset a d len) (offset b c len) c d
          (offset e h len) (offset f g len) g h]
      :f [a (offset b a len) (offset c d len) d
          e (offset f e len) (offset g h len) h]
      :b [(offset a b len) b c (offset d c len)
          (offset e f len) f g (offset h g len)]
      :n [a b c d (offset e a len) (offset f b len)
          (offset g c len) (offset h d len)]
      :s [(offset a e len) (offset b f len)
          (offset c g len) (offset d h len)
          e f g h])
    node (inc (tree-depth node)))])
(defn reflect-on-plane
  "Reflects point p on plane defined by point q & normal n.
  Normal vector must be normalized."
  [^thi.ng.geom.core.vector.Vec3 p
   ^thi.ng.geom.core.vector.Vec3 q
   ^thi.ng.geom.core.vector.Vec3 n]
  (let [^thi.ng.geom.core.vector.Vec3 r (g/- q p)
        d (* (g/dot r n) 2.0)]
    (thi.ng.geom.core.vector.Vec3.
     (+ (mm/msub (.-x n) d (.-x r)) (.-x q))
     (+ (mm/msub (.-y n) d (.-y r)) (.-y q))
     (+ (mm/msub (.-z n) d (.-z r)) (.-z q)))))

(defmethod operator [BoxNode :reflect]
  [{[a b c d e f g h] :points :as node}
   {{:keys [dir] :or {dir :n}} :args}]
  [node
   (BoxNode.
    (case dir
      :e (let [n (gu/ortho-normal c d g)]
           [d c (reflect-on-plane b c n) (reflect-on-plane a d n)
            h g (reflect-on-plane f g n) (reflect-on-plane e h n)])
      :w (let [n (gu/ortho-normal a b f)]
           [(reflect-on-plane d a n) (reflect-on-plane c b n) b a
            (reflect-on-plane h e n) (reflect-on-plane g f n) f e])
      :s (let [n (gu/ortho-normal a c b)]
           [(reflect-on-plane e a n) (reflect-on-plane f b n)
            (reflect-on-plane g c n) (reflect-on-plane h d n)
            a b c d])
      :n (let [n (gu/ortho-normal e f g)]
           [e f g h
            (reflect-on-plane a e n) (reflect-on-plane b f n)
            (reflect-on-plane c g n) (reflect-on-plane d h n)])
      :f (let [n (gu/ortho-normal b c g)]
           [b (reflect-on-plane a b n) (reflect-on-plane d c n) c
            f (reflect-on-plane e f n) (reflect-on-plane h g n) g])
      :b (let [n (gu/ortho-normal a e h)]
           [(reflect-on-plane b a n) a d (reflect-on-plane c d n)
            (reflect-on-plane f e n) e h (reflect-on-plane g h n)]))
    node (inc (tree-depth node)))])

(defmethod operator [BoxNode :scale-edge]
  [{[a b c d e f g h] :points :as node}
   {{:keys [edge sym scale len] :or {scale 0.5}} :args}]
  (let [scale-if (fn [sid p q s]
                   (if (= sid sym)
                     (let [c (g/mix p q)]
                       [(g/madd (g/- p c) s c) (g/madd (g/- q c) s c)])
                     [p q]))
        scale (fn [p q s1 i j s2 k l]
                (let [ll (g/dist p q)
                      dpq (or len (* ll scale))
                      s (/ dpq ll)
                      c (g/mix p q)
                      p' (g/madd (g/- p c) s c)
                      q' (g/madd (g/- q c) s c)
                      [i j] (scale-if s1 i j (/ dpq (g/dist i j)))
                      [k l] (scale-if s2 k l (/ dpq (g/dist k l)))]
                  [p' q' i j k l]))]
    [(BoxNode.
      (case edge
        ;; bottom
        :ab (let [[a b c d e f] (scale a b :x c d :y e f)]
              [a b c d e f g h])
        :bc (let [[b c a d f g] (scale b c :z a d :y f g)]
              [a b c d e f g h])
        :cd (let [[c d a b g h] (scale c d :x a b :y g h)]
              [a b c d e f g h])
        :ad (let [[a d b c e h] (scale a d :z b c :y e h)]
              [a b c d e f g h])
        ;; top
        :ef (let [[e f g h a b] (scale e f :x g h :y a b)]
              [a b c d e f g h])
        :fg (let [[f g e h b c] (scale f g :z e h :y b c)]
              [a b c d e f g h])
        :gh (let [[g h e f c d] (scale g h :x e f :y c d)]
              [a b c d e f g h])
        :eh (let [[e h f g a d] (scale e h :z f g :y a d)]
              [a b c d e f g h])
        ;; left
        :ae (let [[a e d h b f] (scale a e :x d h :z b f)]
              [a b c d e f g h])
        :bf (let [[b f c g a e] (scale b f :x c g :z a e)]
              [a b c d e f g h])
        ;; right
        :cg (let [[c g b f d h] (scale c g :x b f :z d h)]
              [a b c d e f g h])
        :dh (let [[d h a e c g] (scale d h :x a e :z c g)]
              [a b c d e f g h]))
      node (inc (tree-depth node)))]))

(defn make-planar
  [a b c d]
  (let [pabc (pl/plane-from-points a b c)
        pabd (pl/plane-from-points a b d)
        pacd (pl/plane-from-points a c d)
        pbcd (pl/plane-from-points b c d)]
    (mapv #(g/dist % %2) [pbcd pacd pabd pabc] [a b c d])))

(defmethod operator [BoxNode :scale-side]
  [{[a b c d e f g h] :points :as node}
   {{:keys [side scale] :or {scale 0.5}} :args}]
  (let [s (* (- 1.0 scale) 0.5)
        [fa fb fc fd] (face-vertices node side)
        [fa fb fc fd] (mapv (fn [[p q]] (g/mix p q s)) [[fa fc] [fb fd] [fc fa] [fd fb]])]
    [(BoxNode.
      (case side
        :e [a b fa fb e f fd fc]
        :w [fa fb c d fd fc g h]
        :n [a b c d fa fb fc fd]
        :s [fa fd fc fb e f g h]
        :f [a fa fb d e fd fc h]
        :b [fb b c fa fc f g fd])
      node (inc (tree-depth node)))]))
(defmethod operator [BoxNode :skew]
  [{[a b c d e f g h] :points :as node}
   {{:keys [side ref offset] :or {offset 0.5}} :args}]
  (let [n (if (v/vec3? offset)
            offset
            (g/* (quad-normal (face-vertices node ref)) offset))]
    ;;(prn side ref n)
    [(BoxNode.
      (case side
        :e [a b (g/+ c n) (g/+ d n) e f (g/+ g n) (g/+ h n)]
        :w [(g/+ a n) (g/+ b n) c d (g/+ e n) (g/+ f n) g h]
        :n [a b c d (g/+ e n) (g/+ f n) (g/+ g n) (g/+ h n)]
        :s [(g/+ a n) (g/+ b n) (g/+ c n) (g/+ d n) e f g h]
        :f [a (g/+ b n) (g/+ c n) d e (g/+ f n) (g/+ g n) h]
        :b [(g/+ a n) b c (g/+ d n) (g/+ e n) f g (g/+ h n)])
      node (inc (tree-depth node)))]))

(defmethod operator [BoxNode :skew2]
  [{[a b c d e f g h] :points :as node}
   {{:keys [side dir offset] :or {offset 0.5}} :args}]
  (let [[fa fb fc fd] (face-vertices node side)
        skew-vec (if (v/vec3? offset)
                   (constantly offset)
                   (fn [a b c d]
                     (g/normalize (g/- (g/mix a b) (g/mix c d)) offset)))]
    [(BoxNode.
      (case side
        :e (let [n (case dir
                     :z (skew-vec fa fd fb fc)
                     :y (skew-vec fc fd fb fa))]
             [a b (g/+ c n) (g/+ d n) e f (g/+ g n) (g/+ h n)])
        :w (let [n (case dir
                     :z (skew-vec fb fc fa fd)
                     :y (skew-vec fc fd fb fa))]
             [(g/+ a n) (g/+ b n) c d (g/+ e n) (g/+ f n) g h])
        :n (let [n (case dir
                     :z (skew-vec fb fc fa fd)
                     :x (skew-vec fc fd fa fb))]
             [a b c d (g/+ e n) (g/+ f n) (g/+ g n) (g/+ h n)])
        :s (let [n (case dir
                     :z (skew-vec fc fd fa fb)
                     :x (skew-vec fb fc fa fd))]
             [(g/+ a n) (g/+ b n) (g/+ c n) (g/+ d n) e f g h])
        :f (let [n (case dir
                     :y (skew-vec fd fc fa fb)
                     :x (skew-vec fb fc fa fd))]
             [a (g/+ b n) (g/+ c n) d e (g/+ f n) (g/+ g n) h])
        :b (let [n (case dir
                     :x (skew-vec fd fc fa fb)
                     :y (skew-vec fb fc fa fd))]
             [(g/+ a n) b c (g/+ d n) (g/+ e n) f g (g/+ h n)]))
      node (inc (tree-depth node)))]))
(defmethod operator [BoxNode :split-displace]
  [{[a b c d e f g h] :points :as node}
   {{:keys [dir ref offset] :or {offset 0.5}} :args}]
  (let [sd-dir ({:x :cols :y :rows :z :slices} dir)
        children (operator node {:op :sd :args {sd-dir 2}})]
    (mapcat
     (fn [c side]
       (operator c {:op :skew2 :args {:side side :dir ref :offset offset}}))
     children
     (case dir
       :x [:e :w]
       :y [:n :s]
       :z [:f :b]))))

(defmethod operator [BoxNode :split-displace2]
  [{[a b c d e f g h] :points :as node}
   {{:keys [dir ref offset] :or {offset 0.5}} :args}]
  (let [sd-dir ({:x :cols :y :rows :z :slices} dir)
        children (operator node {:op :sd :args {sd-dir 2}})
        offset (g/* (quad-normal (face-vertices node ({:x :e :y :n :z :f} ref))) offset)]
    (mapcat
     (fn [c side]
       (operator c {:op :skew2 :args {:side side :dir ref :offset offset}}))
     children
     (case dir
       :x [:e :w]
       :y [:n :s]
       :z [:f :b]))))

(defn operator-output
  [n out empty?]
  (let [default (vec (repeat n (if empty? nil {})))]
    (cond
     (map? out) (reduce-kv assoc default out)
     (sequential? out) (vec out)
     :default default)))

(defn subdiv
  [& {:keys [cols rows slices out empty?] :or {cols 1 rows 1 slices 1}}]
  {:op :sd
   :args {:cols cols :rows rows :slices slices}
   :out (operator-output (* cols rows slices) out empty?)})

(defn subdiv-inset
  [& {:keys [dir inset out empty?] :or {dir :y inset 0.25}}]
  {:op :sd-inset
   :args {:dir dir :inset inset}
   :out (operator-output 5 out empty?)})

(defn reflect
  [& {:keys [dir out empty?] :or {dir :n}}]
  {:op :reflect
   :args {:dir dir}
   :out (operator-output 2 out empty?)})

(defn extrude
  [& {:keys [dir len out empty?] :or {dir :n len 1.0}}]
  {:op :extrude
   :args {:dir dir :len len}
   :out (operator-output 1 out empty?)})

(defn extrude-prop
  [& {:keys [dir len out empty?] :or {dir :n len 1.0}}]
  {:op :ext-prop
   :args {:dir dir :len len}
   :out (operator-output 1 out empty?)})

(defn scale-edge
  [edge sym & {:keys [scale len out] :or {scale 0.5}}]
  {:op :scale-edge
   :args {:edge edge :sym sym :scale scale :len len}
   :out (operator-output 1 out false)})

(defn skew
  [side ref & {:keys [offset out] :or {offset 0.25}}]
  {:op :skew
   :args {:side side :ref ref :offset offset}
   :out (operator-output 1 out false)})

(defn split-displace
  [dir ref & {:keys [offset out] :or {offset 0.25}}]
  {:op :split-displace2
   :args {:dir dir :ref ref :offset offset}
   :out (operator-output 2 out false)})
(defn reflect-seq
  "Takes a seq of direction keys and optional `leaf` tree. Builds a
  tree encoding a nested reflection sequence in the order given. If
  `leaf` is specified, injects it at the end of the nested
  reflection."
  [dirs & [leaf]]
  (reduce
    (fn [t dir] (reflect :dir dir :out [{} t])) (or leaf {})
    (reverse dirs)))

(def ^:private kw-out :out)

(defn- walk*
  [node tree max-depth acc]
  ;;(prn :d (tree-depth node) (:points node) tree)
  (if (< (tree-depth node) max-depth)
    (let [children (operator node tree)]
      (if children
        (loop [acc acc, children children, out (get tree kw-out)]
          (if (and out children)
            (let [ctree (first out)]
              (recur
               (if ctree (walk* (first children) ctree max-depth acc) acc)
               (next children) (next out)))
            acc))
        (conj! acc (g/faces node))))
    (conj! acc (g/faces node))))

(defn walk
  ([seed tree] (walk seed tree 1e6))
  ([seed tree max-depth] (persistent! (walk* seed tree max-depth (transient [])))))
(defn compute-tree-map*
  [node tree acc path]
  (let [children (operator node tree)]
    (if children
      (loop [acc (assoc! acc path node), children children, out (get tree kw-out), i 0]
        (if (and out children)
          (let [c (first children)
                ctree (first out)
                cpath (conj path i)]
            (recur
             (if ctree
               (compute-tree-map* c ctree acc cpath)
               (assoc! acc cpath c))
             (next children) (next out) (inc i)))
          acc))
      (assoc! acc path node))))

(defn compute-tree-map
  [seed tree]
  (persistent! (compute-tree-map* seed tree (transient {}) [])))
(defn- operator-seq*
  [f]
  (fn opseq*
    [acc node]
    (reduce opseq* (f acc node) (:out node))))

(defn operator-seq
  [node]
  (->> node
       ((operator-seq*
         (fn [acc node]
           (conj! acc
                  (cond
                   (:op node)  (:op node)
                   (nil? node) :delete
                   :else       :leaf))))
        (transient []))
       (persistent!)))

(defn operator-seq-no-leaves
  [node]
  (->> node
       ((operator-seq*
         (fn [acc node]
           (if-let [op (:op node)] (conj! acc op) acc)))
        (transient []))
       (persistent!)))

(defn circle-lattice-seg
  [n h wall]
  (let [theta (/ m/PI n)
        off (vec3 0 0 h)
        points (g/vertices (g/rotate (g/as-polygon (c/circle) n) (- (- HALF_PI) theta)))
        [b c] (map vec3 points)
        [a d] (map vec3 (p/inset-polygon points (- wall)))
        [f g] (map #(g/+ off %) [b c])
        [e h] (map #(g/+ off %) [a d])]
    [b f g c a e h d]))

(defn sphere-lattice-seg
  [n h inset wall]
  (let [theta (/ m/PI n)
        off (vec3 0 0 h)
        points (g/vertices (g/rotate (g/as-polygon (c/circle) n) (- (- HALF_PI) theta)))
        [b c] (map vec3 points)
        [a d] (map vec3 (p/inset-polygon points (- wall)))
        [f g] (map #(g/+ off %) (p/inset-polygon points (- inset)))
        [e h] (map #(g/+ off %) (p/inset-polygon points (- (- inset) wall)))]
    [b f g c a e h d]))

(defn sphere-lat
  [resu resv wall]
  (let [r1 (- 1.0 wall)
        lat2 (/ PI resv)
        lat1 (- lat2)
        lon2 (/ PI resu)
        lon1 (- lon2)]
    (->> [(vec3 r1 lat1 lon1)
          (vec3 1 lat1 lon1)
          (vec3 1 lat1 lon2)
          (vec3 r1 lat1 lon2)
          (vec3 r1 lat2 lon1)
          (vec3 1 lat2 lon1)
          (vec3 1 lat2 lon2)
          (vec3 r1 lat2 lon2)]
         (mapv g/as-cartesian))))

(defn union-mesh
  ([meshes]
     (union-mesh (bm/basic-mesh) 1e-3 meshes))
  ([target eps meshes]
     (-> (reduce g/into target meshes)
         (ops/canonicalize-vertices eps)
         (first)
         (ops/remove-internal))))

#+clj
(defn save-mesh
  ([seed tree] (save-mesh seed tree "p.ply" 1e6))
  ([seed tree path] (save-mesh seed tree path 1e6))
  ([seed tree path max-depth]
     (with-open [o (io/output-stream path)]
       (->> (walk seed tree max-depth)
            (union-mesh)
            (g/tessellate)
            (mio/write-ply o)))))
