(ns thi.ng.geom.webgl.shaders
  (:require-macros
    [cljs-log.core :refer [info warn]])
  (:require
    [thi.ng.math.core :as m]
    [thi.ng.geom.core :as g]
    [thi.ng.geom.matrix]
    [thi.ng.geom.webgl.constants :as glc]
    [thi.ng.geom.webgl.utils :as glu]
    [thi.ng.typedarrays.core :as ta]
    [thi.ng.dstruct.streams :as streams]
    [thi.ng.glsl.core :as glsl :include-macros true]
    [thi.ng.color.core :as col]
    [thi.ng.xerror.core :as err]
    [clojure.string :as str]))

  (def uniform-types
    {:float       ["1fv" ta/float32]
     :int         ["1iv" ta/int32]
     :vec2        ["2fv" ta/float32 2]
     :vec3        ["3fv" ta/float32 3]
     :vec4        ["4fv" ta/float32 4]
     :ivec2       ["2iv" ta/int32 2]
     :ivec3       ["3iv" ta/int32 3]
     :ivec4       ["4iv" ta/int32 4]
     :bool        ["1iv" ta/int32]
     :bvec2       ["2iv" ta/int32 2]
     :bvec3       ["3iv" ta/int32 3]
     :bvec4       ["4iv" ta/int32 4]
     :mat2        ["Matrix2fv" ta/float32 4]
     :mat3        ["Matrix3fv" ta/float32 9]
     :mat4        ["Matrix4fv" ta/float32 16]
     :sampler2D   ["1iv" ta/int32]
     :samplerCube ["1iv" ta/int32]})
  
  (defn init-shader-uniforms
    [^WebGLRenderingContext gl prog uniforms]
    (reduce
     (fn [umap [id type]]
       (let [loc                (.getUniformLocation gl prog (name id))
             [type default opt] (if (sequential? type) type [type])
             [u-type u-cast]    (uniform-types type)
             setter             (aget gl (str "uniform" u-type))]
         ;; TODO add check if valid uniform
         (assoc umap id
                {:type type
                 :default default
                 :setter (cond
                           (#{:mat2 :mat3 :mat4} type)
                           (fn [x]
                             (.call setter gl loc (boolean opt)
                                    (if (ta/typed-array? x)
                                      x (u-cast x))))
                           (= :vec2 type)
                           (fn [x]
                             (.call setter gl loc
                                    (cond
                                      (ta/typed-array? x) x
                                      (satisfies? streams/IBuffer x) (streams/get-buffer x)
                                      (number? x) (u-cast [x x])
                                      :else (u-cast x))))
                           (= :vec3 type)
                           (fn [x]
                             (.call setter gl loc
                                    (cond
                                      (ta/typed-array? x) x
                                      (satisfies? streams/IBuffer x) (streams/get-buffer x)
                                      (number? x) (-> x col/int24 col/as-rgba deref (subvec 0 3) u-cast)
                                      :else (u-cast x))))
                           (= :vec4 type)
                           (fn [x]
                             (.call setter gl loc
                                    (cond
                                      (ta/typed-array? x) x
                                      (satisfies? streams/IBuffer x) (streams/get-buffer x)
                                      (number? x) (-> x col/int24 col/as-rgba deref u-cast)
                                      (string? x) (-> x col/css col/as-rgba deref u-cast)
                                      :else (u-cast x))))
                           :else
                           (fn [x]
                             (.call setter gl loc
                                    (if (ta/typed-array? x)
                                      x (u-cast (if (not (sequential? x)) [x] x))))))
                 :loc loc})))
     {} uniforms))
  
  (defn set-uniform
    [shader id val]
    (if-let [u-spec (get-in shader [:uniforms id])]
      ((get u-spec :setter) val)
      (warn "Unknown shader uniform: " id)))
  
  (defn apply-default-uniforms
    [shader uniforms]
    (->> (keys uniforms)
         (apply dissoc (get shader :uniforms))
         (transduce
          (filter #(-> % val (get :default)))
          (completing #(set-uniform shader (key %2) (-> %2 val (get :default))))
          nil)))

  (defn init-shader-attribs
    [^WebGLRenderingContext gl prog attribs]
    (reduce
     (fn [amap id]
       (assoc amap id (.getAttribLocation gl prog (name id))))
     {} attribs))
  
  (defn set-attribute
    [^WebGLRenderingContext gl shader id {:keys [buffer stride size type normalized? offset]}]
    (if-let [loc (get-in shader [:attribs id])]
      (doto gl
        (.bindBuffer glc/array-buffer buffer)
        (.enableVertexAttribArray loc)
        (.vertexAttribPointer
         loc
         size
         (or type glc/float)
         (boolean normalized?)
         (or stride 0)
         (or offset 0)))
      (warn (str "Unknown shader attribute: " id))))
  
  (defn disable-attribute
    [^WebGLRenderingContext gl shader id]
    (if-let [loc (get-in shader [:attribs id])]
      (do (.disableVertexAttribArray gl loc) gl)
      (warn (str "Unknown shader attribute: " id))))

    (def header-prefix
      "
  #ifdef GL_FRAGMENT_PRECISION_HIGH
  precision highp int;
  precision highp float;
  #else
  precision mediump int;
  precision mediump float;
  #endif
  #ifndef PI
  #define PI      3.141592653589793
  #endif
  #ifndef TWO_PI
  #define TWO_PI  6.283185307179586
  #endif
  #ifndef HALF_PI
  #define HALF_PI 1.570796326794896
  #endif
  #ifndef RAD
  #define RAD     0.008726646259972
  #endif
  ")
  (defn compile-glsl-vars
    [qualifier coll]
    (->> coll
         (map
          (fn [[id type]]
            (str qualifier " "
                 (name (if (sequential? type) (first type) type)) " "
                 (name id) ";\n")))
         (apply str)))
  
  (defn parse-and-throw-error
    [^WebGLRenderingContext gl shader src]
    (let [src-lines (vec (str/split-lines src))
          errors (->> shader
                      (.getShaderInfoLog gl)
                      (str/split-lines)
                      (map
                       (fn [line]
                         (let [[_ ln msg] (re-find #"ERROR: \d+:(\d+): (.*)" line)]
                           (when ln
                             (str "line " ln ": " msg "\n"
                                  (src-lines (dec (js/parseInt ln 10))))))))
                      (filter identity)
                      (str/join "\n"))]
      (.deleteShader gl shader)
      (err/throw! (str "Error compiling shader:\n" errors))))
  
  (defn compile-shader
    [^WebGLRenderingContext gl src type]
    (if-let [shader (.createShader gl type)]
      (do
        (.shaderSource gl shader src)
        (.compileShader gl shader)
        (if (.getShaderParameter gl shader glc/compile-status)
          shader
          (parse-and-throw-error gl shader src)))
      (err/throw! "Can't create shader")))
  
  (defn make-shader-from-spec
    [^WebGLRenderingContext gl {:keys [vs fs uniforms attribs varying prefix state]}]
    (let [u-src  (compile-glsl-vars "uniform" uniforms)
          a-src  (compile-glsl-vars "attribute" attribs)
          v-src  (compile-glsl-vars "varying" varying)
          prefix (str (or prefix header-prefix) u-src v-src)
          vs     (compile-shader gl (str prefix a-src vs) glc/vertex-shader)
          fs     (compile-shader gl (str prefix fs) glc/fragment-shader)
          prog   (.createProgram gl)]
      (doto gl
        (.attachShader prog vs)
        (.attachShader prog fs)
        (.linkProgram prog))
      (if (.getProgramParameter gl prog glc/link-status)
        (let [uniforms (init-shader-uniforms gl prog uniforms)
              attribs (init-shader-attribs gl prog (keys attribs))]
          (doto gl
            (.deleteShader vs)
            (.deleteShader fs))
          {:program  prog
           :uniforms uniforms
           :attribs  attribs
           :varying  varying
           :state    state})
        (err/throw! (str "Shader failed to link:" (.getProgramInfoLog gl prog))))))
  
  (defn make-shader-from-dom
    [^WebGLRenderingContext gl {:keys [vs fs] :as spec}]
    (make-shader-from-spec
     gl (assoc spec
          :vs (glu/get-script-text vs)
          :fs (glu/get-script-text fs))))
