(ns thi.ng.geom.webgl.buffers
  (:require
   [thi.ng.geom.webgl.core :as gl]
   [thi.ng.geom.webgl.constants :as glc]
   [thi.ng.xerror.core :as err]))

(declare check-fbo)

(defrecord FBO [^WebGLRenderingContext gl id]
  gl/IBind
  (bind [_] (.bindFramebuffer gl glc/framebuffer id) _)
  (bind [_ _] (err/unsupported!))
  (unbind [_] (.bindFramebuffer gl glc/framebuffer nil) _)

  gl/IRelease
  (release [_] (.deleteFramebuffer gl id) _)

  gl/IFramebuffer
  (set-fbo-color-texture
    [_ {:keys [id]}]
    (.framebufferTexture2D gl glc/framebuffer glc/color-attachment0 glc/texture-2d id 0)
    (check-fbo gl)
    _)
  (set-fbo-depth-buffer
    [_ {:keys [id]}]
    (.framebufferRenderbuffer gl glc/framebuffer glc/depth-attachment glc/renderbuffer id)
    (check-fbo gl)
    _))

(defn- check-fbo
  [^WebGLRenderingContext gl]
  (condp = (.checkFramebufferStatus gl glc/framebuffer)
    glc/framebuffer-unsupported
    (err/throw! "FBO unsupported")
    glc/framebuffer-incomplete-attachment
    (err/throw! "FBO incomplete attachment")
    glc/framebuffer-incomplete-dimensions
    (err/throw! "FBO incomplete dimensions")
    glc/framebuffer-incomplete-missing-attachment
    (err/throw! "FBO incomplete missing attachment")
    gl))

(defn make-fbo
  [^WebGLRenderingContext gl]
  (FBO. gl (.createFramebuffer gl)))

(defrecord RenderBuffer [^WebGLRenderingContext gl id format width height]
  gl/IBind
  (bind [_] (.bindRenderbuffer gl glc/renderbuffer id) _)
  (bind [_ _] (err/unsupported!))
  (unbind [_] (.bindRenderbuffer gl glc/renderbuffer nil) _)

  gl/IRelease
  (release [_] (.deleteRenderbuffer gl id) _)

  gl/IConfigure
  (configure [_ {:keys [format width height]}]
    (gl/bind _)
    (.renderbufferStorage gl glc/renderbuffer format width height)
    (gl/unbind _)
    (RenderBuffer. gl id format width height)))

(defn make-render-buffer
  ([^WebGLRenderingContext gl]
   (make-render-buffer gl nil))
  ([^WebGLRenderingContext gl opts]
   (let [buf (RenderBuffer. gl (.createRenderbuffer gl) nil nil nil)]
     (if opts (gl/configure buf opts) buf))))

(defn make-depth-buffer
  ([^WebGLRenderingContext gl size]
   (make-depth-buffer gl size size))
  ([^WebGLRenderingContext gl width height]
   (make-render-buffer gl {:format glc/depth-component16 :width width :height height})))

(defrecord Texture2D [^WebGLRenderingContext gl id target]
  gl/IBind
  (bind
      [_] (gl/bind _ 0))
  (bind
      [_ unit]
    (.activeTexture gl (+ glc/texture0 unit))
    (.bindTexture gl target id)
    _)
  (unbind
      [_] (gl/unbind _ 0))
  (unbind
      [_ unit]
    (.activeTexture gl (+ glc/texture0 unit))
    (.bindTexture gl target nil)
    _)
  gl/IRelease
  (release
      [_] (.deleteTexture gl id) _)
  gl/IConfigure
  (configure
      [_ {:keys [width height image pixels format type filter wrap flip premultiply]
          :or   {type glc/unsigned-byte format glc/rgb} :as config}]
    (when filter
      (let [[min mag] (if (sequential? filter) filter [filter filter])]
        (.texParameteri gl target glc/texture-min-filter min)
        (.texParameteri gl target glc/texture-mag-filter mag)))
    (when wrap
      (let [[ws wt] (if (sequential? wrap) wrap [wrap wrap])]
        (.texParameteri gl target glc/texture-wrap-s ws)
        (.texParameteri gl target glc/texture-wrap-t wt)))
    (when (not (nil? flip))
      (.pixelStorei gl glc/unpack-flip-y-webgl flip))
    (when (not (nil? premultiply))
      (.pixelStorei gl glc/unpack-premultiply-alpha-webgl premultiply))
    (cond
      image
      (.texImage2D gl target 0 format format type image)
      (and width height)
      (.texImage2D gl target 0 format width height 0 format type pixels))
    _))

(defn make-texture
  [^WebGLRenderingContext gl opts]
  (let [tex  (Texture2D. gl (.createTexture gl) (opts :target glc/texture-2d))
        opts (merge {:format glc/rgba :type glc/unsigned-byte} opts)]
    (gl/bind tex)
    (gl/configure tex opts)))

(defn make-canvas-texture
  ([^WebGLRenderingContext gl canvas]
   (make-canvas-texture gl canvas {}))
  ([^WebGLRenderingContext gl canvas opts]
   (->> opts
        (merge {:image       canvas
                :format      glc/rgba
                :filter      [glc/linear glc/linear]
                :wrap        [glc/clamp-to-edge glc/clamp-to-edge]
                :flip        false
                :premultiply true})
        (make-texture gl))))

(defn load-texture
  [^WebGLRenderingcontext gl opts]
  (let [tex (->> opts
                 (merge {:format glc/rgba
                         :filter [glc/linear glc/linear]
                         :wrap   glc/clamp-to-edge})
                 (make-texture gl))
        img (js/Image.)]
    (set! (.-onload img)
          (fn []
            (doto tex
              (gl/bind)
              (gl/configure (merge {:flip false :image img} opts)))
            (when-let [cb (:callback opts)] (cb tex img))))
    (set! (.-src img) (:src opts))
    tex))
