(ns thi.ng.geom.basicmesh
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.utils :as gu]
   [thi.ng.geom.vector :as v :refer [vec2 vec3]]
   [thi.ng.geom.matrix :refer [M44]]
   [thi.ng.geom.meshface :as mf]
   [thi.ng.geom.types :as types]
   [thi.ng.dstruct.core :as d]
   [thi.ng.math.core :as m :refer [*eps*]]
   [thi.ng.xerror.core :as err]
   [clojure.core.reducers :as r]))

(declare basic-mesh)

(defn- add-face*
  [mesh [fverts]]
  (thi.ng.geom.types.BasicMesh.
   (into (:vertices mesh) fverts)
   (conj (:faces mesh) (thi.ng.geom.meshface.MeshFace. fverts nil))
   (:fnormals mesh)))

(defn basic-mesh
  "Builds a new 3d mesh data structure and (optionally) populates it with
  the given items (a seq of existing meshes and/or faces). Faces are defined
  as vectors of their vertices."
  [] (thi.ng.geom.types.BasicMesh. #{} #{} {}))

(extend-type thi.ng.geom.types.BasicMesh
g/IArea
(area
 [_] (gu/total-area-3d (mf/xf-face-verts _) (:faces _)))


g/IBounds
(bounds [_] (gu/bounding-box (seq (:vertices _))))
(width [_]  (gu/axis-range 0 (:vertices _)))
(height [_] (gu/axis-range 1 (:vertices _)))
(depth [_]  (gu/axis-range 2 (:vertices _)))
g/IBoundingSphere
(bounding-sphere
 [_] (gu/bounding-sphere (g/centroid _) (:vertices _)))
g/ICenter
(center
 ([_]   (g/center _ (vec3)))
 ([_ o] (g/transform _ (g/translate M44 (m/- o (g/centroid _))))))
(centroid
 [_]    (gu/centroid (seq (:vertices _))))
g/IFlip
(flip [_] (gu/map-mesh (fn [f] [(vec (rseq f))]) _))
g/IVertexAccess
(vertices
 [_] (:vertices _))
g/IEdgeAccess
(edges
 [_]
 (into
  #{}
  (comp
   (map #(g/vertices % _))
   (mapcat #(d/successive-nth 2 (conj % (first %))))
   (map set))
  (:faces _)))
g/IFaceAccess
(faces
 ([_] (:faces _))
 ([_ opts]
  (if opts
    (map #(g/raw % _) (:faces _))
    (:faces _))))
(add-face
 [_ face] (add-face* _ face))
(vertex-faces
 [_ v]
 (sequence
  (comp
   (map #(g/vertices % _))
   (filter
    #(pos? #?(:clj (.indexOf ^clojure.lang.PersistentVector % v)
              :cljs (d/index-of % v))))
   (:faces _))))
(remove-face
 [_ f]
 (err/unsupported!)) ;; TODO implement
g/INormalAccess
(face-normals
 [_ force?] (if (seq (:fnormals _)) (:fnormals _) (if force? (:fnormals (g/compute-face-normals _)))))
(face-normal
 [_ f] ((:fnormals _) f))
(vertex-normals
 [_ force?] (if force? (err/unsupported!)))
(vertex-normal
 [_ v] (err/unsupported!))
(compute-face-normals
 [_]
 (loop [fnorms (transient {}), faces (:faces _)]
   (if faces
     (let [f (first faces)]
       (recur (assoc! fnorms f (gu/ortho-normal (g/vertices f _))) (next faces)))
     (assoc _ :fnormals (persistent! fnorms)))))
(compute-vertex-normals
 [_] (err/unsupported!))
g/IGeomContainer
(into
 [_ faces] (gu/into-mesh _ add-face* faces))
g/IClear
(clear*
 [_] (basic-mesh))
g/IMeshConvert
(as-mesh
 ([_] _)
 ([_ opts] (g/into (:mesh opts) (:faces _))))
g/ITessellate
(tessellate
 ([_]      (g/tessellate _ {}))
 ([_ opts] (gu/map-mesh (or (:fn opts) (gu/tessellate-face gu/tessellate-with-first)) _)))
g/IScale
(scale
 ([_ s]
    (gu/transform-mesh _ add-face* #(m/* % s)))
 ([_ sx sy sz]
    (gu/transform-mesh _ add-face* #(m/* % sx sy sz))))
(scale-size
 [_ s]
 (let [c (g/centroid _)]
   (gu/transform-mesh _ add-face* #(m/madd (m/- % c) s c))))
g/ITranslate
(translate
 [_ t] (gu/transform-mesh _ add-face* #(m/+ % t)))
g/ITransform
(transform
 [_ m]
 (gu/transform-mesh _ add-face* m))
g/IVolume
(volume
 [_] (gu/total-volume (mf/xf-face-verts _) (:faces _)))
)
