(ns thi.ng.geom.physics.core
                                                         
  (:require
   [thi.ng.geom.core :as g]
   [thi.ng.geom.core.vector :as v :refer [vec2 vec3]]
   [thi.ng.common.data.core :as d]
         [thi.ng.macromath.core :as mm]))

(defprotocol PTimeStep
  (timestep [_ delta]))

(defprotocol PBehavioral
  (add-behaviors [_ bmap])
  (remove-behavior [_ id])
  (clear-behaviors [_])
  (apply-behaviors [_ delta]))

(defprotocol PConstrained
  (add-constraints [_ cmap])
  (remove-constraint [_ id])
  (clear-constraints [_])
  (apply-constraints [_ delta]))

(defprotocol PParticle
  (add-force [_ f])
  (add-velocity [_ v])
  (apply-force [_ delta])
  (clear-force [_])
  (clear-velocity [_])
  (scale-velocity [_ s])
  (position [_])
  (set-position [_ p])
  (velocity [_])
  (lock [_])
  (unlock [_])
  (locked? [_]))

(defprotocol PPhysics
  (update-particles [_ delta])
  (update-springs [_ delta]))

(defn apply-to-particle
  [p delta fns]
  (when-not (locked? p)
    (loop [f (seq fns)] (when f ((first f) p delta) (recur (next f))))))

(defn apply-to-particles
  [particles delta fns]
  (loop [ps (seq particles)]
    (when ps
      (apply-to-particle (first ps) delta fns)
      (recur (next ps)))))

(deftype VerletParticle
         
  [^:unsynchronized-mutable pos
   ^:unsynchronized-mutable prev
   ^:unsynchronized-mutable force
   ^:unsynchronized-mutable locked?
   ^:unsynchronized-mutable behaviors
   ^:unsynchronized-mutable constraints
   ^double inv-weight
   _meta]
        
                
                 
                  
                    
                      
                        
             
         
        clojure.lang.IObj
              
  (      meta             
    [_] _meta)
                  
  (      withMeta                  
    [_ meta']
    (VerletParticle. pos prev force locked? behaviors constraints inv-weight meta'))
  PBehavioral
  (add-behaviors
    [_ bmap] (set! behaviors (merge behaviors bmap)) _)
  (apply-behaviors
    [_ delta] (when (seq behaviors) (apply-to-particle _ delta (vals behaviors))) _)
  (remove-behavior
    [_ id] (set! behaviors (dissoc behaviors id)) _)
  (clear-behaviors
    [_] (set! behaviors {}) _)
  PConstrained
  (add-constraints
    [_ cmap] (set! constraints (merge constraints cmap)) _)
  (apply-constraints
    [_ delta] (when (seq constraints) (apply-to-particle _ delta (vals constraints))) _)
  (remove-constraint
    [_ id] (set! constraints (dissoc constraints id)) _)
  (clear-constraints
    [_] (set! constraints {}) _)
  PParticle
  (lock
    [_] (set! locked? true) _)
  (unlock
    [_] (set! locked? false) _)
  (locked?
    [_] locked?)
  (position
    [_] pos)
  (set-position
    [_ p] (set! pos p) _)
  (velocity
    [_] (g/- pos prev))
  (add-force
    [_ f] (set! force (g/+ force f)) _)
  (clear-force
    [_] (set! force (g/clear* force)))
  (apply-force
    [_ delta]
    (let [pos' (g/madd force (* inv-weight (* delta delta)) (g/msub pos 2.0 prev))]
      (set! prev pos)
      (set! pos pos')
      (set! force (g/clear* force)))
    _)
  (scale-velocity
    [_ s] (set! prev (g/mix prev pos s)) _)
  PTimeStep
  (timestep
    [_ delta]
    (if-not locked?
      (-> (apply-behaviors _ delta)
          (apply-force delta)
          (apply-constraints delta))
      _))
  Object
  (toString
    [_]
    (pr-str
     {:pos pos
      :prev prev
      :force force
      :locked? locked?
      :inv-weight inv-weight
      :behaviors behaviors
      :constraints constraints})))
(defrecord Spring
    [^VerletParticle a
     ^VerletParticle b
     ^double rest-len
     ^double strength]
  PTimeStep
  (timestep
    [_ delta]
    (let [aw   (.-inv-weight ^VerletParticle a)
          bw   (.-inv-weight ^VerletParticle b)
          pa   (position a)
          pb   (position b)
          diff (g/- pb pa)
          dist (+ (g/mag diff) 1e-6)
          nd   (* (/ (- dist rest-len) (* dist (+ aw bw))) (* strength delta))]
      (if-not (locked? a)
        (set-position a (g/madd diff (* nd aw) pa)))
      (if-not (locked? b)
        (set-position b (g/madd diff (* (- nd) bw) pb))))))

(defrecord PullbackSpring
    [^VerletParticle a
     ^VerletParticle b
     ^double rest-len
     ^double min-len
     ^double strength]
  PTimeStep
  (timestep
    [_ delta]
    (let [pa   (position a)
          pb   (position b)
          diff (g/- pb pa)
          dist (+ (g/mag diff) 1e-6)]
      (if (> dist min-len)
        (let [aw (.-inv-weight ^VerletParticle a)
              bw (.-inv-weight ^VerletParticle b)
              nd (* (/ (- dist rest-len) (* dist (+ aw bw))) (* strength delta))]
          (if-not (locked? a)
            (set-position a (g/madd diff (* nd aw) pa)))
          (if-not (locked? b)
            (set-position b (g/madd diff (* (- nd) bw) pb))))))))
(defrecord VerletPhysics
    [particles springs behaviors constraints drag]
  PBehavioral
  (add-behaviors
    [_ bmap] (update-in _ [:behaviors] merge bmap))
  (apply-behaviors
    [_ delta] (apply-to-particles particles delta (vals behaviors)) _)
  (remove-behavior
    [_ id] (update-in _ [:behaviors] dissoc id))
  (clear-behaviors
    [_] (assoc _ behaviors {}))
  PConstrained
  (add-constraints
    [_ cmap] (update-in _ [:constraints] merge cmap))
  (apply-constraints
    [_ delta] (apply-to-particles particles delta (vals constraints)) _)
  (remove-constraint
    [_ id] (update-in _ [:constraints] dissoc id))
  (clear-constraints
    [_] (assoc _ constraints {}))
  PPhysics
  (update-particles
    [_ delta]
    (let [drag' (* delta drag)]
      (loop [ps (seq particles)]
        (when ps
          (-> ps first (scale-velocity drag') (timestep delta))
          (recur (next ps)))))
    _)
  (update-springs
    [_ delta] (loop [s (seq springs)] (when s (timestep (first s) delta) (recur (next s)))) _)
  PTimeStep
  (timestep
    [_ iter]
    (let [delta (/ 1.0 iter)]
      (loop [i iter]
        (when (pos? i)
          (-> (apply-behaviors _ delta)
              (update-springs delta)
              (update-particles delta)
              (apply-constraints delta))
          (recur (dec i)))))
    _))

(defn physics
  [{:keys [particles springs behaviors constraints drag]
    :or   {particles #{}, springs #{}, behaviors {}, constraints {}, drag 0.0}}]
  (VerletPhysics.
   (set particles) (set springs) behaviors constraints drag))

(defn particle
  [pos & [weight lock?]]
  (let [w (double (or weight 1.0))]
    (VerletParticle. pos pos (g/clear* pos) lock? nil nil (/ 1.0 w) nil)))

(defn spring
  [a b rlen strength]
  (Spring. a b (double rlen) (double strength)))

(defn gravity
  [force]
  (fn [p delta]
    (add-force p (g/* force delta))))

(defn attract
  [pos r strength]
  (let [rsq (* r r)]
    (fn [p delta]
      (let [d (g/- pos (position p))
            l (g/mag-squared d)]
        (if (and (> l 0.0) (< l rsq))
          (add-force p (g/* d (/ (* (- 1.0 (/ l rsq)) (* strength delta))
                                 (Math/sqrt l)))))))))

(defn attract-particle
  [p r strength]
  (let [rsq (* r r)]
    (fn [q delta]
      (if-not (= p q)
        (let [d (g/- (position p) (position q))
              l (g/mag-squared d)]
          (if (and (> l 0.0) (< l rsq))
            (add-force q (g/* d (/ (* (- 1.0 (/ l rsq)) (* strength delta))
                                   (Math/sqrt l))))))))))

(defn align
  [vel strength]
  (fn [p delta]
    (let [d (g/- vel (velocity p))]
      (add-force p (g/* d (* strength delta))))))

(defn shape-constraint*
  [pred shape]
  (fn [p delta]
    (let [pos (position p)]
      (if (pred pos) (set-position p (g/closest-point shape pos))))))

(defn shape-constraint-inside
  [shape] (shape-constraint* #(not (g/contains-point? shape %)) shape))

(defn shape-constraint-outside
  [shape] (shape-constraint* #(g/contains-point? shape %) shape))

(defn shape-constraint-boundary
  [shape] (shape-constraint* (constantly true) shape))

;;;;;;;;;;;; This file autogenerated from src/cljx/thi/ng/geom/physics/core.cljx
