(ns thi.ng.geom.mesh.ops
  (:require
   [thi.ng.math.core :as m]
   [thi.ng.geom.core :as g]
   [thi.ng.geom.core.utils :as gu]
   [thi.ng.geom.core.intersect :as isec]
   [thi.ng.geom.core.vector :as v :refer [vec2 vec3]]
   [thi.ng.geom.aabb :as a]
   [thi.ng.geom.sphere :as s]
   [thi.ng.geom.line :as l]
   [thi.ng.geom.gmesh :as gm]
   [thi.ng.geom.spatialtree :as st]
   [thi.ng.geom.types.utils :as tu]
   [thi.ng.dstruct.core :as d]
   [clojure.set :as set]))

(defn find-in-tree
  "Takes a query radius `eps`, returns a fn which queries an octree
  with a spherical region around `p` using the pre-configured radius.
  Returns the closest point found (if any)."
  [eps]
  (fn [tree p]
    (->> (st/select-with-sphere tree p eps)
         (sort-by #(g/dist-squared p %))
         (first))))

(defn unique-point-tree
  [points eps]
  (let [finder (find-in-tree eps)
        [tree dupes] (reduce
                      (fn [[t dupes :as state] p]
                        (if (finder t p)
                          [t (conj! dupes p)]
                          [(g/add-point t p) dupes]))
                      [(st/octree (tu/bounding-box (seq points))) (transient [])]
                      points)]
    [tree (persistent! dupes)]))

(defn collapse-edges
  [m eps]
  (let [eps* (* eps eps)]
    (->> (:edges m)
         (keys)
         (filter (fn [e] (let [[a b] (seq e)] (< (g/dist-squared a b) eps*))))
         (reduce
          (fn [m e] (if (-> m :edges e) (gm/merge-vertices* m (first e) (second e)) m))
          m))))

(defn canonicalize-vertices
  [mesh eps]
  (let [[tree dupes] (unique-point-tree (g/vertices mesh) eps)
        finder (find-in-tree eps)]
    [(reduce (fn [m f] (g/add-face m (mapv #(finder tree %) f))) (g/clear* mesh) (g/faces mesh))
     ;;(g/into (g/clear* mesh) (map (fn [f] (mapv #(finder tree %) f)) (g/faces mesh)))
     dupes]))

(defn face-permutations
  [f] (take (count f) (iterate #(d/rotate-left 1 %) f)))

(defn remove-internal
  "Takes a mesh and removes all faces which have coincident vertices,
  but opposite orientation. Returns updated mesh."
  [mesh]
  (->> (g/faces mesh)
       (reduce
        (fn [acc f]
          (let [dupes (filter acc (face-permutations (reverse f)))]
            (if (seq dupes)
              (reduce disj acc (cons f dupes))
              acc)))
        (set (g/faces mesh)))
       (assoc mesh :faces)))

(defn remove-internal-with-edges
  [mesh]
  (let [mesh (g/compute-face-normals mesh)]
    (->> (g/edges mesh)
         (reduce
          (fn [acc e]
            (let [ef (-> mesh :edges e)]
              (if (> (count ef) 1)
                (let [efn (select-keys (:fnormals mesh) ef)
                      ef (set ef)
                      acc (reduce
                           (fn [acc [f n]]
                             (let [ni (g/- n)
                                   dup (filter #(m/delta= ni (efn %)) (disj ef f))]
                               (if (seq dup) (apply disj acc dup) acc)))
                           acc efn)]
                  acc)
                acc)))
          (set (g/faces mesh)))
         (assoc mesh :faces))))

(defn make-watertight
  [{:keys [vertices edges] :as m} eps]
  (let [split-face (fn [v e [fa fb fc :as f]]
                     (cond
                      (= e #{fa fb}) [f [fa v fc] [v fb fc]]
                      (= e #{fb fc}) [f [fa fb v] [v fc fa]]
                      :default [f [fc v fb] [v fa fb]]))
        update-face (fn [m [f f1 f2]]
                      (-> (g/remove-face m f)
                          (g/add-face f1)
                          (g/add-face f2)))
        eps* (* eps eps)]
    (reduce
     (fn [m v]
       (let [vedges (into #{} (map (fn [n] #{v n}) (g/vertex-neighbors m v)))
             coeff #(gu/closest-point-coeff v % %2)
             v-on-edge? (fn [a b]
                          (let [t (coeff a b)
                                p (if (m/in-range? 0.01 0.99 t) (g/mix a b t))]
                            (if (and p (< (g/dist-squared v p) eps*)) p)))]
         (loop [edges (set/difference (into #{} (keys (:edges m))) vedges)]
           (if-let [e (first edges)]
             (let [[a b] (seq e) p (v-on-edge? a b)]
               (if p
                 (reduce update-face m (map #(split-face v e %) (-> m :edges e)))
                 (recur (rest edges))))
             m))))
     m (keys vertices))))
