(ns thi.ng.fabric.facts.core
  #?@(:clj
      [(:require
        [thi.ng.fabric.core :as f]
        [thi.ng.xerror.core :as err]
        [thi.ng.strf.core :as strf]
        [clojure.set :as set]
        [clojure.data.int-map :as imap]
        [clojure.core.async :as a :refer [go go-loop chan close! <! >! alts! timeout]]
        [taoensso.timbre :refer [debug info warn]])]
      :cljs
      [(:require-macros
        [cljs.core.async.macros :refer [go go-loop]]
        [cljs-log.core :refer [debug info warn]])
       (:require
        [thi.ng.fabric.core :as f]
        [thi.ng.xerror.core :as err]
        [thi.ng.strf.core :as strf]
        [clojure.set :as set]
        [cljs.core.async :refer [chan close! <! >! alts! timeout]])]))

(def ^:private MAX_LIMIT #?(:clj Long/MAX_VALUE :cljs (.-MAX_VALUE js/Number)))

(declare index-selection make-index-selections qvar? add-query! add-query-join!)

(defprotocol IFactGraph
  (fact-indices [_])
  (fact-transform [_])
  (add-fact! [_ t])
  (remove-fact! [_ t]))

(defprotocol ICache
  (cached [_ type k])
  (cache! [_ type k v])
  (expire! [_ type k]))

(defprotocol IFactQuery
  (raw-pattern [_])
  (query-spec [_]))

(defprotocol IQueryResult
  (result-vertex [_]))

(defprotocol ITwoWayTransform
  (transform [_ x])
  (untransform [_ x]))

(declare map->facts)

(defn map->facts*
  [s]
  (fn [[p o]]
    (cond
      (sequential? o) (mapv (fn [o] [s p o]) o)
      (map? o)        (let [bn (strf/new-uuid)] (cons [s p bn] (map->facts o bn)))
      :else           [[s p o]])))

(defn map->facts
  ([fact-map]
   (mapcat (fn [[s v]] (mapcat (map->facts* s) v)) fact-map))
  ([fact-map s]
   (mapcat (map->facts* s) fact-map)))

(defn- signal-fact
  [vertex op] [op (f/vertex-id vertex) @vertex])

(defn- collect-index
  [spo]
  (let [tx (map (fn [[op id t]] [op id (nth t spo)]))
        rf (completing
            (fn [acc [op id x]]
              (case op
                :add    (assoc acc x (conj (or (acc x) #{}) id))
                :remove (if-let [idx (acc x)]
                          (if (= #{id} idx)
                            (dissoc acc x)
                            (assoc acc x (disj idx id)))
                          acc)
                (do (warn "ignoring unknown index signal op:" op)
                    acc))))]
    (f/collect-pure
     (fn [val incoming]
       ;;(debug :old-index val)
       (let [val (transduce tx rf val incoming)]
         ;;(debug :new-index val)
         val)))))

(defn- signal-index-select
  [vertex [idx sel]]
  [idx (if sel (@vertex sel [nil]) (->> @vertex vals (mapcat identity) (set)))])

(def ^:private collect-select
  (f/collect-pure
   (fn [val incoming]
     (let [val (reduce (fn [acc [idx res]] (assoc acc idx res)) val incoming)]
       ;;(debug :coll-select val incoming)
       val))))

(defn- score-collect-min-signal-vals
  [num]
  (fn [vertex]
    (if (> num (count (vals (peek (f/uncollected-signals vertex))))) 0 1)))

(defn- score-collect-min-signals
  [num]
  (fn [vertex]
    (if (> num (count (f/uncollected-signals vertex))) 0 1)))

(defn- collect-basic-query-results
  [g opts]
  (let [ftx (fact-transform g)
        tx  (map #(untransform ftx @(f/vertex-for-id g %)))
        tx  (if-let [flt (:filter opts)]
              (comp tx (filter flt))
              tx)
        tx  (if-let [lim (:limit opts)]
              (comp tx (take lim))
              tx)]
    (f/collect-pure
     (fn [_ incoming]
       (let [res (vals (peek incoming))]
         ;;(debug :agg-incoming res)
         (if (every? #(not= [nil] %) res)
           (->> res
                (into #{} (map #(disj % nil)))
                (sort-by count)
                (reduce set/intersection)
                (into #{} tx))
           #{}))))))

(defn- score-collect-join
  [vertex]
  (if (and (seq (f/uncollected-signals vertex))
           (== (count (f/signal-map vertex)) 2))
    1 0))

(defn- collect-inference
  [g production]
  (fn [vertex]
    (let [prev @vertex
          in   (reduce into #{} (f/uncollected-signals vertex))
          adds (set/difference in prev)]
      (debug (f/vertex-id vertex) :additions adds)
      (run! #(production g vertex %) adds)
      (f/update-value! vertex #(set/union % adds)))))

(defn collect-into-set
  [tx] (f/collect-pure (fn [_ in] (into #{} tx (peek in)))))

(def ^:dynamic *auto-qvar-prefix* "?__q")

(defn qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  [x] (and (symbol? x) (= \? (.charAt ^String (name x) 0))))

(defn auto-qvar?
  "Returns true, if x is an auto-generated qvar (a symbol prefixed
  with *auto-qvar-prefix*)"
  [x] (and (symbol? x) (zero? (.indexOf ^String (name x) ^String *auto-qvar-prefix*))))

(defn auto-qvar
  "Creates a new auto-named qvar (symbol)."
  [] (gensym *auto-qvar-prefix*))

(defn qvar-name
  [x] (-> x name (subs 1)))

(defn resolve-path-pattern
  "Takes a path triple pattern and max depth. The pattern's predicate
  must be a seq of preds. Returns a 2-elem vector [patterns vars],
  where `patterns` is a seq of query patterns with injected temp qvars
  for inbetween patterns and `vars` the autogenerated qvars
  themselves.

  Example:

    [?s [p1 p2 p3] ?o]
    => [([?s p1 ?__q0] [?__q0 p2 ?__q1] [?__q1 p3 ?o]) (?__q0 ?__q1)]"
  [[s p o] maxd]
  (let [avars (repeatedly maxd auto-qvar)
        vars  (cons s avars)]
    [(->> (concat (interleave vars (take maxd (cycle p))) [o])
          (partition 3 2))
     avars]))

(defn pattern-var-count
  [pattern]
  (count (filter qvar? pattern)))

(defn sort-patterns
  [patterns]
  (sort-by pattern-var-count patterns))

(defn select-keys*
  "Like c.c/select-keys, but doesn't retain map's meta"
  {:static true}
  [map keyseq]
  (loop [ret {} keys (seq keyseq)]
    (if keys
      #?(:clj
         (let [entry (. clojure.lang.RT (find map (first keys)))]
           (recur
            (if entry (conj ret entry) ret)
            (next keys)))
         :cljs
         (let [key   (first keys)
               entry (get map key ::not-found)]
           (recur
            (if (= entry ::not-found) ret (assoc ret key entry))
            (next keys))))
      ret)))

(defn index*
  "Like clojure.set/index, but using select-keys w/o metadata retention."
  [xrel ks]
  (persistent!
   (reduce
    (fn [m x]
      (let [ik (select-keys* x ks)]
        (assoc! m ik (conj (get m ik #{}) x))))
    (transient {}) xrel)))

(defn join
  "Based on clojure.set/join. Does not join when there're no shared
  keys and no key mapping, enforced result set limit, uses
  transients."
  [xrel yrel]
  (if (and (seq xrel) (seq yrel))
    (let [ks (set/intersection (set (keys (first xrel))) (set (keys (first yrel))))]
      (if (seq ks)
        (let [[r s] (if (<= (count xrel) (count yrel))
                      [xrel yrel]
                      [yrel xrel])
              idx (index* r ks)]
          (persistent!
           (reduce
            (fn [ret x]
              (let [found (idx (select-keys* x ks))]
                (if found
                  (reduce #(conj! % (conj %2 x)) ret found)
                  ret)))
            (transient #{}) s)))
        #{}))
    #{}))

(defn join-optional
  [a b]
  (loop [old (transient #{}), new (transient #{}), kb b]
    (if kb
      (let [kb'       [(first kb)]
            [old new] (loop [old old, new new, ka a]
                        (if ka
                          (let [ka' (first ka)
                                j   (first (join [ka'] kb'))]
                            (if j
                              (recur (conj! old ka') (conj! new j) (next ka))
                              (recur old new (next ka))))
                          [old new]))]
        (recur old new (next kb)))
      (let [new (persistent! new)]
        (if (seq new)
          (into (apply disj (set a) (persistent! old)) new)
          a)))))

(defn result-pre-aggregator
  [agg]
  (fn [results]
    (let [ares (agg results)]
      (map #(merge % ares) results))))

(defn result-post-aggregator
  [agg cfn spec]
  (let [sel (:select spec)
        rfn (if (or (nil? sel) (= :* sel))
              (let [sel (if (sequential? sel) sel [sel])]
                (fn [acc k res]
                  (let [ares (agg res)]
                    (assoc acc k (mapv #(merge % ares) res)))))
              (let [avars (set (keys (:aggregate* spec)))]
                (if (and (seq avars) (every? avars sel))
                  (fn [acc k res]
                    (assoc acc k (select-keys (merge (first res) (agg res)) sel)))
                  (fn [acc k res]
                    (let [ares (agg res)]
                      (assoc acc k (into (empty res) (map #(select-keys (merge % ares) sel)) res)))))))]
    (fn [_ in]
      (reduce-kv rfn {} (cfn _ in)))))

(defn result-transducer
  [spec tx]
  (let [tx (if-let [flt (:filter spec)]
             (let [tx' (filter flt)]
               (if tx (comp tx tx') tx'))
             tx)
        tx (if-let [lim (:limit spec)]
             (let [tx' (take lim)]
               (if tx (comp tx tx') tx'))
             tx)
        tx (if-let [sel (and (not (:group-by spec)) (:select spec))]
             (if-not (= :* sel)
               (let [sel (if (sequential? sel) sel [sel])
                     tx' (comp (map #(select-keys % sel)) (filter seq))
                     agg (set (keys (:aggregate* spec)))
                     tx' (if (and (seq agg) (every? agg sel))
                           (comp tx' (take 1)) tx')]
                 (if tx (comp tx tx') tx'))
               tx)
             tx)]
    (or tx (map identity))))

(defn group-by-reducer
  [spec grp]
  (let [type (if (:order spec) [] #{})]
    (fn
      ([] {})
      ([res] res)
      ([res input] (update res (grp input) #(conj (or % type) input))))))

(defn remove-recursively
  [g res parent deps type]
  (if (f/none-or-single-user? res parent)
    (do (debug type)
        (f/remove-vertex! g res)
        (run! #(f/remove-from-graph! % g res) deps)
        true)
    (do (f/disconnect-neighbor! res parent)
        false)))

(defn as-query-entity
  [res qspec deps type]
  (reify
    #?@(:clj
         [clojure.lang.IDeref (deref [_] @res)]
         :cljs
         [IDeref (-deref [_] @res)])
    IFactQuery
    (raw-pattern
      [_] nil)
    (query-spec
      [_] qspec)
    IQueryResult
    (result-vertex
      [_] res)
    f/IGraphComponent
    (add-to-graph!
      [_ g] (err/unsupported!))
    (remove-from-graph!
      [_ g] (f/remove-from-graph! _ g nil))
    (remove-from-graph!
      [_ g parent]
      (remove-recursively g res parent deps type))))
(defmulti bind-translator
  (fn [qvars? pattern] (count pattern)))

(defmethod bind-translator 3
  [[vs? vp? vo?] [s p o]]
  (if vs?
    (if vp?
      (if vo?
        (fn [r] {s (r 0) p (r 1) o (r 2)})
        (fn [r] {s (r 0) p (r 1)}))
      (if vo?
        (fn [r] {s (r 0) o (r 2)})
        (fn [r] {s (r 0)})))
    (if vp?
      (if vo?
        (fn [r] {p (r 1) o (r 2)})
        (fn [r] {p (r 1)}))
      (if vo?
        (fn [r] {o (r 2)})
        (fn [_] {})))))

(defmethod bind-translator 4
  [[vt? vs? vp? vo?] [t s p o]]
  (if vt?
    (if vs?
      (if vp?
        (if vo?
          (fn [r] {t (r 0) s (r 1) p (r 2) o (r 3)})
          (fn [r] {t (r 0) s (r 1) p (r 2)}))
        (if vo?
          (fn [r] {t (r 0) s (r 1) o (r 3)})
          (fn [r] {t (r 0) s (r 1)})))
      (if vp?
        (if vo?
          (fn [r] {t (r 0) p (r 2) o (r 3)})
          (fn [r] {t (r 0) p (r 2)}))
        (if vo?
          (fn [r] {t (r 0) o (r 3)})
          (fn [r] {t (r 0)}))))
    (if vs?
      (if vp?
        (if vo?
          (fn [r] {s (r 1) p (r 2) o (r 3)})
          (fn [r] {s (r 1) p (r 2)}))
        (if vo?
          (fn [r] {s (r 1) o (r 3)})
          (fn [r] {s (r 1)})))
      (if vp?
        (if vo?
          (fn [r] {p (r 2) o (r 3)})
          (fn [r] {p (r 2)}))
        (if vo?
          (fn [r] {o (r 3)})
          (fn [_] {}))))))
(defmulti fact-verifier
  (fn [qvars? pattern] (count pattern)))

(defmethod fact-verifier 3
  [[vs? vp? vo?] [s p o]]
  (cond
    (and vs? vp? vo?) (cond
                        (= s p o) #(= (% 0) (% 1) (% 2))
                        (= s p) #(and (= (% 0) (% 1)) (not= (% 0) (% 2)))
                        (= s o) #(and (= (% 0) (% 2)) (not= (% 0) (% 1)))
                        (= p o) #(and (= (% 1) (% 2)) (not= (% 0) (% 1)))
                        :else nil)
    (and vs? vp?)     (if (= s p) #(= (% 0) (% 1)) #(not= (% 0) (% 1)))
    (and vs? vo?)     (if (= s o) #(= (% 0) (% 2)) #(not= (% 0) (% 2)))
    (and vp? vo?)     (if (= p o) #(= (% 1) (% 2)) #(not= (% 1) (% 2)))
    :else             nil))

(defmethod fact-verifier 4
  [[vt? vs? vp? vo?] [t s p o]]
  (cond
    (and vt? vs? vp? vo?)
    (cond
      (= t s p o)           #(= (% 0) (% 1) (% 2) (% 3))
      (= t s p)             #(and (not= (% 0) (% 3)) (= (% 0) (% 1) (% 2)))
      (= t s o)             #(and (not= (% 0) (% 2)) (= (% 0) (% 1) (% 3)))
      (= t p o)             #(and (not= (% 0) (% 1)) (= (% 0) (% 2) (% 3)))
      (= s p o)             #(and (not= (% 0) (% 1)) (= (% 1) (% 2) (% 3)))
      (and (= t s) (= p o)) #(and (= (% 0) (% 1)) (= (% 2) (% 3)))
      (and (= t p) (= s o)) #(and (= (% 0) (% 2)) (= (% 1) (% 3)))
      (and (= t o) (= s p)) #(and (= (% 0) (% 3)) (= (% 1) (% 2)))
      (= t s)               #(let [t (first %)] (and (= t (% 1)) (not= t (% 2)) (not= t (% 3))))
      (= t p)               #(let [t (first %)] (and (= t (% 2)) (not= t (% 1)) (not= t (% 3))))
      (= t o)               #(let [o (peek %)]  (and (= o (% 0)) (not= o (% 1)) (not= o (% 2))))
      (= s p)               #(let [s (nth % 1)] (and (= s (% 2)) (not= s (% 3)) (not= s (% 0))))
      (= s o)               #(let [o (peek %)]  (and (= o (% 1)) (not= o (% 2)) (not= o (% 0))))
      (= p o)               #(let [o (peek %)]  (and (= o (% 2)) (not= o (% 1)) (not= o (% 0))))
      :else                 nil)
    (and vt? vs? vp?)
    (cond
      (= t s p)             #(= (% 0) (% 1) (% 2))
      (= t s)               #(and (= (% 0) (% 1)) (not= (% 0) (% 2)))
      (= t p)               #(and (= (% 0) (% 2)) (not= (% 0) (% 1)))
      (= s p)               #(and (= (% 1) (% 2)) (not= (% 1) (% 0)))
      :else                 nil)
    (and vt? vs? vo?)
    (cond
      (= t s o)             #(= (% 0) (% 1) (% 3))
      (= t s)               #(and (= (% 0) (% 1)) (not= (% 0) (% 3)))
      (= t o)               #(and (= (% 0) (% 3)) (not= (% 0) (% 1)))
      (= s o)               #(and (= (% 1) (% 3)) (not= (% 1) (% 0)))
      :else                 nil)
    (and vt? vp? vo?)
    (cond
      (= t p o)             #(= (% 0) (% 2) (% 3))
      (= t p)               #(and (= (% 0) (% 2)) (not= (% 0) (% 3)))
      (= t o)               #(and (= (% 0) (% 3)) (not= (% 0) (% 1)))
      (= p o)               #(and (= (% 2) (% 3)) (not= (% 2) (% 0)))
      :else                 nil)
    (and vs? vp? vo?)
    (cond
      (= s p o)             #(= (% 1) (% 2) (% 3))
      (= s p)               #(and (= (% 1) (% 2)) (not= (% 1) (% 3)))
      (= s o)               #(and (= (% 1) (% 3)) (not= (% 1) (% 2)))
      (= p o)               #(and (= (% 2) (% 3)) (not= (% 2) (% 1)))
      :else                 nil)
    (and vt? vs?)           (if (= t s) #(= (% 0) (% 1)) #(not= (% 0) (% 1)))
    (and vt? vp?)           (if (= t p) #(= (% 0) (% 2)) #(not= (% 0) (% 2)))
    (and vt? vo?)           (if (= t o) #(= (% 0) (% 3)) #(not= (% 0) (% 3)))
    (and vs? vp?)           (if (= s p) #(= (% 1) (% 2)) #(not= (% 1) (% 2)))
    (and vs? vo?)           (if (= s o) #(= (% 1) (% 3)) #(not= (% 1) (% 3)))
    (and vp? vo?)           (if (= p o) #(= (% 2) (% 3)) #(not= (% 2) (% 3)))
    :else                   nil))

(deftype FactVertex
    [id fact new-edges outs]
  #?@(:clj
       [clojure.lang.IDeref
        (deref
         [_] fact)]
       :cljs
       [IDeref
        (-deref
         [_] fact)])
  f/IVertex
  (vertex-id
    [_] id)
  (set-value!
    [_ val] (err/unsupported!))
  (update-value!
    [_ f] (err/unsupported!))
  (previous-value
    [_] fact)
  (collect!
    [_] (err/unsupported!))
  (score-collect
    [_] 0)
  (connect-to!
    [_ v sig-fn opts]
    (swap! outs assoc v [sig-fn opts])
    (swap! new-edges inc)
    (debug id "edge to" (f/vertex-id v) "(" (pr-str opts) ") new:" @new-edges)
    _)
  (neighbors
    [_] (keys @outs))
  (disconnect-neighbor!
    [_ v]
    (when v
      (debug id "disconnect from" (f/vertex-id v))
      (swap! outs dissoc v))
    _)
  (disconnect-all!
    [_]
    (run! #(f/disconnect-neighbor! _ %) (keys @outs))
    _)
  (new-edge-count
    [_] @new-edges)
  (score-signal
    [_] @new-edges)
  (signal!
    [_ handler]
    (reset! new-edges 0)
    (handler _ @outs))
  (receive-signal
    [_ src sig] (err/unsupported!))
  (signal-map
    [_] (err/unsupported!))
  (uncollected-signals
    [_] (err/unsupported!)))
(defn fact-vertex
  [id fact _]
  (FactVertex. id fact (atom 0) (atom {})))

(defn- index-vertex
  [g idx]
  (f/add-vertex!
   g {} {::f/collect-fn      (collect-index idx)
         ::f/score-signal-fn f/score-signal-with-new-edges}))

(def identity-transform
  (reify ITwoWayTransform
    (transform [_ x] x)
    (untransform [_ x] x)))
(defn prefix-vector-transform
  [prefixes]
  (let [prefixes' (rseq (vec (sort-by #(count (peek %)) prefixes)))]
    (reify ITwoWayTransform
      (transform
        [_ x]
        (if (string? x)
          (reduce
           (fn [acc p]
             (if (zero? (.indexOf ^String x ^String (peek p)))
               (reduced [(first p) (subs x (count (peek p)))])
               acc))
           x prefixes')
          x))
      (untransform
        [_ x]
        (if (vector? x)
          (if-let [p (prefixes (first x))]
            (str p (nth x 1))
            x)
          x)))))
(defn prefix-string-transform
  [prefixes]
  (let [prefixes' (rseq (vec (sort-by #(count (peek %)) prefixes)))]
    (reify ITwoWayTransform
      (transform
        [_ x]
        (if (string? x)
          (reduce
           (fn [acc p]
             (if (zero? (.indexOf ^String x ^String (peek p)))
               (reduced (str (first p) \: (subs x (count (peek p)))))
               acc))
           x prefixes')
          x))
      (untransform
        [_ x]
        (if (string? x)
          (let [[[_ p n]] (re-seq #"^(\w[\w\-_]+):([\w\-_]*)$" x)]
            (if-let [p (prefixes p)]
              (str p n)
              x))
          x)))))
(defn global-index-transform
  []
  (let [index (atom #?(:clj  {:fwd {} :rev (imap/int-map) :id 0}
                       :cljs {:fwd {} :rev {} :id 0}))]
    (reify ITwoWayTransform
      (transform
        [_ x]
        (if (or (nil? x) (qvar? x))
          x
          (or ((@index :fwd) x)
              (let [curr (volatile! nil)]
                (swap! index
                       #(let [id (:id %)]
                          (vreset! curr id)
                          (-> %
                              (update :id inc)
                              (update :fwd assoc x id)
                              (update :rev assoc id x))))
                @curr))))
      (untransform
        [_ id] ((@index :rev) id id)))))
(defn compose-transforms
  [& transforms]
  (let [rtx (reverse transforms)]
    (reify ITwoWayTransform
      (transform [_ x]
        (reduce #(transform %2 %) x transforms))
      (untransform [_ x]
        (reduce #(untransform %2 %) x rtx)))))

(defn one-way-transform
  [tx]
  (reify ITwoWayTransform
    (transform [_ x] (transform tx x))
    (untransform [_ x] x)))

(defn combine-transforms
  ([tx len]
   (apply combine-transforms (repeat len tx)))
  ([txs txp txo]
   (reify ITwoWayTransform
     (transform [_ fact]
       [(transform txs (first fact))
        (transform txp (nth fact 1))
        (transform txo (nth fact 2))])
     (untransform [_ fact]
       [(untransform txs (first fact))
        (untransform txp (nth fact 1))
        (untransform txo (nth fact 2))])))
  ([txt txs txp txo]
   (reify ITwoWayTransform
     (transform [_ fact]
       [(transform txt (first fact))
        (transform txs (nth fact 1))
        (transform txp (nth fact 2))
        (transform txo (nth fact 3))])
     (untransform [_ fact]
       [(untransform txt (first fact))
        (untransform txs (nth fact 1))
        (untransform txp (nth fact 2))
        (untransform txo (nth fact 3))]))))

(defrecord FactGraph
    [g indices facts cache ftx]
  f/IComputeGraph
  (add-vertex!
    [_ val vspec] (f/add-vertex! g val vspec))
  (remove-vertex!
    [_ v] (f/remove-vertex! g v))
  (vertex-for-id
    [_ id] (f/vertex-for-id g id))
  (vertices
    [_] (f/vertices g))
  (add-edge!
    [_ src dest sig opts] (f/add-edge! g src dest sig opts))
  f/IWatch
  (add-watch!
    [_ type id f] (f/add-watch! g type id f) _)
  (remove-watch!
    [_ type id] (f/remove-watch! g type id) _)
  IFactGraph
  (fact-indices
    [_] indices)
  (fact-transform
    [_] ftx)
  (add-fact!
    [_ f]
    (let [f' (transform ftx f)]
      (or (@facts f')
          (let [v (f/add-vertex! g f' nil fact-vertex)]
            (debug :add-fact f f')
            (run! #(f/add-edge! g v % signal-fact :add) indices)
            (swap! facts assoc f' v)
            v))))
  (remove-fact!
    [_ f]
    (let [f' (transform ftx f)]
      (if-let [v (@facts f')]
        (do
          (debug :remove-fact f f')
          (run! #(f/add-edge! g v % signal-fact :remove) indices)
          (swap! facts dissoc f')
          (f/remove-vertex! g v)
          v)
        (warn "attempting to remove unknown fact:" f))))
  ICache
  (cached
    [_ type k] (get-in @cache [type k]))
  (cache!
    [_ type k v] (swap! cache assoc-in [type k] v) v)
  (expire!
    [_ type k] (swap! cache update type dissoc k) nil))
(defn fact-graph
  "Creates a new FactGraph instance configured with given options map:
  :graph     - backing IComputeGraph (default fabric.core/compute-graph)
  :len       - fact length (default 3)
  :index     - index vertex ctor (default index-vertex)
  :transform - fact transform (default none)"
  ([]
   (fact-graph {}))
  ([{:keys [graph len index transform]
     :or   {graph     (f/compute-graph)
            len       3
            index     index-vertex
            transform identity-transform}}]
   (map->FactGraph
    {:indices (mapv #(index graph %) (range len))
     :facts   (atom {})
     :cache   (atom {})
     :ftx     transform
     :g       graph})))

(defn index-selection
  [g sel]
  (if-let [sel' (cached g ::index-sel sel)]
    (do (debug :reuse-index-sel sel) sel')
    (let [index  ((fact-indices g) (first sel))
          vertex (f/add-vertex!
                  g nil
                  {::f/score-signal-fn f/score-signal-with-new-edges
                   ::f/collect-fn      (f/collect-pure (fn [_ in] (peek in)))})
          isel   (reify
                   IQueryResult
                   (result-vertex
                     [_] vertex)
                   f/IGraphComponent
                   (add-to-graph!
                     [_ g] _)
                   (remove-from-graph!
                     [_ g] (f/remove-from-graph! _ g nil))
                   (remove-from-graph!
                     [_ g parent]
                     (if (f/none-or-single-user? vertex parent)
                       (do (debug :remove-index-sel sel)
                           (expire! g ::index-sel sel)
                           (f/disconnect-neighbor! index vertex)
                           (f/remove-vertex! g vertex)
                           true)
                       (do (f/disconnect-neighbor! vertex parent)
                           false))))]
      (f/add-edge! g index vertex signal-index-select sel)
      (cache! g ::index-sel sel isel))))

(defn make-index-selections
  [g pattern]
  (into [] (map-indexed #(index-selection g [% %2])) pattern))
(defn add-query!
  ([g pattern opts]
   (add-query! g (fact-transform g) pattern opts))
  ([g ptx pattern opts]
   (let [pattern (transform ptx pattern)]
     (if-let [q (and (empty? opts) (cached g ::queries pattern))]
       (do (debug :reuse-basic-query pattern) q)
       (let [[s p o] pattern
             sels (make-index-selections g pattern)
             acc  (f/add-vertex!
                   g {} {::f/collect-fn collect-select})
             res  (f/add-vertex!
                   g nil
                   {::f/collect-fn       (collect-basic-query-results g opts)
                    ::f/score-signal-fn  f/score-signal-with-new-edges
                    ::f/score-collect-fn (score-collect-min-signal-vals (count sels))})
             q    (reify
                    #?@(:clj
                         [clojure.lang.IDeref (deref [_] @res)]
                         :cljs
                         [IDeref (-deref [_] @res)])
                    IFactQuery
                    (raw-pattern
                      [_] pattern)
                    (query-spec
                      [_] (merge {:select :* :q [{:where pattern}]} opts))
                    IQueryResult
                    (result-vertex
                      [_] res)
                    f/IGraphComponent
                    (add-to-graph!
                      [_ g] (err/unsupported!))
                    (remove-from-graph!
                      [_ g] (f/remove-from-graph! _ g nil))
                    (remove-from-graph!
                      [_ g parent]
                      (if (f/none-or-single-user? res parent)
                        (do (debug :remove-query pattern)
                            (expire! g ::queries pattern)
                            (f/remove-vertex! g res)
                            (f/remove-vertex! g acc)
                            (run! #(f/remove-from-graph! % g acc) sels)
                            true)
                        (do (f/disconnect-neighbor! res parent)
                            false))))]
         (run! #(f/add-edge! g (result-vertex %) acc f/signal-forward nil) sels)
         (f/add-edge! g acc res f/signal-forward nil)
         (if-not (seq opts)
           (cache! g ::queries pattern q)
           q))))))
(defn add-param-query!
  ([g pattern opts]
   (add-param-query! g (fact-transform g) pattern opts))
  ([g ptx pattern opts]
   (let [pattern (transform ptx pattern)]
     (if-let [q (and (empty? opts) (cached g ::queries pattern))]
       (do (debug :reuse-param-query pattern) q)
       (let [qvars?  (mapv qvar? pattern)
             raw     (mapv #(if-not (qvar? %) %) pattern)
             vmap    (bind-translator qvars? pattern)
             verify  (fact-verifier qvars? pattern)
             tx      (if verify
                       (map #(if (verify %) (vmap %)))
                       (map vmap))
             tx      (comp tx (filter identity))
             tx      (if-let [flt (:filter opts)]
                       (comp tx (filter flt))
                       tx)
             tx      (if-let [lim (:limit opts)]
                       (comp tx (take lim))
                       tx)
             coll-fn (f/collect-pure
                      (fn [_ incoming]
                        (if-let [res (seq (peek incoming))]
                          (into #{} tx res)
                          #{})))
             sub-q   (add-query! g identity-transform raw {})
             result  (f/add-vertex!
                      g nil
                      {::f/collect-fn      coll-fn
                       ::f/score-signal-fn f/score-signal-with-new-edges})
             pq      (reify
                       #?@(:clj
                            [clojure.lang.IDeref (deref [_] @result)]
                            :cljs
                            [IDeref (-deref [_] @result)])
                       IFactQuery
                       (raw-pattern
                         [_] raw)
                       (query-spec
                         [_]) ;; TODO
                       IQueryResult
                       (result-vertex
                         [_] result)
                       f/IGraphComponent
                       (add-to-graph!
                         [_ g] (err/unsupported!))
                       (remove-from-graph!
                         [_ g] (f/remove-from-graph! _ g nil))
                       (remove-from-graph!
                         [_ g parent]
                         (if (f/none-or-single-user? result parent)
                           (do (debug :remove-param-query pattern)
                               (expire! g ::queries pattern)
                               (f/remove-vertex! g result)
                               (f/remove-from-graph! sub-q g result)
                               true)
                           (do (f/disconnect-neighbor! result parent)
                               false))))]
         (f/add-edge! g (result-vertex sub-q) result f/signal-forward nil)
         (if-not (seq opts)
           (cache! g ::queries pattern pq)
           pq))))))
(defn add-join!
  ([g lhs rhs opts]
   (add-join! g join lhs rhs opts))
  ([g join-fn lhs rhs opts]
   (let [lhs-v  (result-vertex lhs)
         rhs-v  (result-vertex rhs)
         lhs-id (f/vertex-id lhs-v)
         rhs-id (f/vertex-id rhs-v)
         tx     (if-let [flt (:filter opts)]
                  (filter flt))
         tx     (if-let [lim (:limit opts)]
                  (let [tx' (take lim)]
                    (if tx (comp tx tx') tx'))
                  tx)
         cfn    (if tx #(into #{} tx (join-fn % %2)) join-fn)
         res    (f/add-vertex!
                 g nil
                 {::f/collect-fn
                  (fn [vertex]
                    (let [sig-map (f/signal-map vertex)
                          a (sig-map lhs-id)
                          b (sig-map rhs-id)]
                      (debug (f/vertex-id vertex) :join-sets a b)
                      (f/set-value! vertex (cfn a b))))
                  ::f/score-collect-fn score-collect-join
                  ::f/score-signal-fn  f/score-signal-with-new-edges})
         jq     (as-query-entity res nil [lhs rhs] :remove-query-join)]
     (f/add-edge! g lhs-v res f/signal-forward nil)
     (f/add-edge! g rhs-v res f/signal-forward nil)
     jq)))

(defn add-query-join!
  ([g patterns opts]
   (add-query-join! g (fact-transform g) patterns opts))
  ([g ptx patterns opts]
   (let [[a b & more :as p] patterns ;;(sort-patterns patterns)
         _  (assert (and a b) "Requires min. 2 query patterns")
         jq (reduce
             #(add-join! g join % (add-param-query! g ptx %2 {}) {})
             (add-join!
              g join
              (add-param-query! g ptx a {})
              (add-param-query! g ptx b {})
              (if (< 1 (count more)) {} opts))
             (butlast more))]
     (if-let [p (last more)]
       (add-join! g join jq (add-param-query! g ptx p {}) opts)
       jq))))

(defn add-query-join-optional!
  ([g patterns opts]
   (add-query-join-optional! g (fact-transform g) patterns opts))
  ([g ptx patterns opts]
   (let [[a b & more] patterns
         _  (assert (and a b) "Requires min. 2 query patterns")
         jq (reduce
             #(add-join! g join-optional % (add-param-query! g ptx %2 {}) {})
             (add-join!
              g join-optional
              (add-param-query! g ptx a {})
              (add-param-query! g ptx b {})
              (if (< 1 (count more)) {} opts))
             (butlast more))]
     (if-let [p (last more)]
       (add-join! g join-optional jq (add-param-query! g ptx p {}) opts)
       jq))))
(defn add-query-union!
  [g queries opts]
  (assert (< 1 (count queries)) "min. 2 queries required")
  (let [tx  (if-let [flt (:filter opts)]
              (filter flt))
        tx  (if-let [lim (:limit opts)]
              (let [tx' (take lim)]
                (if tx (comp tx tx') tx'))
              tx)
        res (f/add-vertex!
             g nil
             {::f/collect-fn
              (fn [vertex]
                (let [subs (vals (f/signal-map vertex))
                      res  (reduce #(if (seq %2) (into % %2) %) subs)]
                  (f/set-value! vertex (if tx (into #{} tx res) res))))
              ::f/score-signal-fn f/score-signal-with-new-edges})
        q   (as-query-entity res nil queries :remove-query-union)]
    (run! #(f/add-edge! g (result-vertex %) res f/signal-forward nil) queries)
    q))
(defn add-path-query!
  ([g path-pattern]
   (let [len (count (nth path-pattern 1))]
     (add-path-query! g path-pattern len len)))
  ([g path-pattern mind maxd]
   (add-path-query! g (fact-transform g) path-pattern mind maxd))
  ([g ptx path-pattern mind maxd]
   (assert (pos? mind) "min depth must be >= 1")
   (assert (<= mind maxd) "min depth must be <= max depth")
   (let [[patterns avars] (resolve-path-pattern path-pattern maxd)
         [?s _ ?o] path-pattern
         vs? (qvar? ?s)
         vo? (qvar? ?o)
         req (take mind patterns)
         opt (drop mind (take maxd patterns))
         req (if (seq req)
               (if (== 1 (count req))
                 (add-param-query! g ptx (first req) {})
                 (add-query-join! g ptx req {})))
         opt (if (seq opt)
               (if (== 1 (count opt))
                 (add-param-query! g ptx (first opt) {})
                 (add-query-join-optional! g ptx opt {})))
         q   (if (and req opt)
               (add-join! g join-optional req opt {}) ;; TODO opts
               (or req opt))
         tx  (cond
               (or (= mind maxd) (and vs? (not vo?)))
               (let [qv (filter qvar? path-pattern)]
                 (map #(select-keys* % qv)))

               (and vo? (not vs?))
               (let [rv (take (dec mind) avars)]
                 (mapcat #(map (fn [v] {?o v}) (vals (apply dissoc % rv)))))

               :else
               (let [rv (cons ?s (take (dec mind) avars))]
                 (mapcat
                  #(let [s (% ?s)] (map (fn [v] {?s s ?o v}) (vals (apply dissoc % rv)))))))
         res (f/add-vertex!
              g #{}
              {::f/collect-fn (f/collect-pure (fn [_ in] (into #{} tx (peek in))))})
         pq  (as-query-entity res nil [q] :remove-path-query)]
     (f/add-edge! g (result-vertex q) res f/signal-forward nil)
     pq)))
(defn add-query-result!
  [g {:keys [aggregate order group-by] :as spec} q]
  (let [in-tx (if (and aggregate (not group-by))
                (result-pre-aggregator aggregate))
        in-tx (if order
                (if in-tx
                  #(sort-by order (in-tx %))
                  #(sort-by order %))
                in-tx)
        tx    (result-transducer spec nil)
        cfn   (if group-by
                (let [gfn (group-by-reducer spec group-by)]
                  (if in-tx
                    (fn [_ in] (transduce tx gfn {} (in-tx (peek in))))
                    (fn [_ in] (transduce tx gfn {} (peek in)))))
                (if in-tx
                  (fn [_ in] (into (if order [] #{}) tx (in-tx (peek in))))
                  (fn [_ in] (into #{} tx (peek in)))))
        cfn   (if (and aggregate group-by)
                (result-post-aggregator aggregate cfn spec)
                cfn)
        res   (f/add-vertex!
               g nil
               {::f/collect-fn      (f/collect-pure cfn)
                ::f/score-signal-fn f/score-signal-with-new-edges})
        qres (as-query-entity res nil [q] :remove-result)]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    qres))

(defn add-counter!
  [g src]
  (let [v (f/add-vertex!
           g nil
           {::f/collect-fn      (f/collect-pure (fn [_ in] (count (peek in))))
            ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g src v f/signal-forward nil)
    v))

(defn add-rule!
  [g {:keys [id query patterns transform production collect-fn] :as opts}]
  (let [id      (or id (f/random-id))
        coll-fn (or collect-fn (collect-inference g production))
        query   (or query
                    (let [tx (or transform (fact-transform g))
                          q-opts (select-keys opts [:filter :limit])]
                      (if (< 1 (count patterns))
                        (add-query-join! g tx patterns q-opts)
                        (add-param-query! g tx (first patterns) q-opts))))
        inf     (f/add-vertex! g #{} {::f/collect-fn coll-fn})
        rule    (reify
                  #?@(:clj
                       [clojure.lang.IDeref (deref [_] @query)]
                       :cljs
                       [IDeref (-deref [_] @query)])
                  IFactQuery
                  (raw-pattern
                    [_] nil)
                  (query-spec
                    [_]) ;; TODO
                  IQueryResult
                  (result-vertex
                    [_] (result-vertex query))
                  f/IGraphComponent
                  (add-to-graph!
                    [_ g] (err/unsupported!))
                  (remove-from-graph!
                    [_ g] (f/remove-from-graph! _ g nil))
                  (remove-from-graph!
                    [_ g parent]
                    (if (f/none-or-single-user? inf parent)
                      (do (debug :remove-rule id)
                          (f/remove-vertex! g inf)
                          (f/remove-from-graph! query g inf)
                          true)
                      (do (f/disconnect-neighbor! inf parent)
                          false))))]
    (f/add-edge! g (result-vertex query) inf f/signal-forward nil)
    (cache! g ::rules id rule)))

(def fact-log-transducer
  (comp
   (filter (fn [[op v]] (and (#{:add-vertex :remove-vertex} op) (vector? @v))))
   (map (fn [[op v]] [({:add-vertex :+ :remove-vertex :-} op) @v]))))

(defn add-fact-graph-logger
  [g log-fn]
  (let [ch        (chan 1024 fact-log-transducer)
        watch-id  (f/random-id)
        log->chan #(go (>! ch %))]
    (go-loop []
      (let [t (<! ch)]
        (when t
          (log-fn t)
          (recur))))
    (f/add-watch! g :add-vertex watch-id log->chan)
    (f/add-watch! g :remove-vertex watch-id log->chan)
    {:graph g :chan ch :watch-id watch-id}))

(defn remove-fact-graph-logger
  [{:keys [graph watch-id chan]}]
  (f/remove-watch! graph :add-vertex watch-id)
  (f/remove-watch! graph :remove-vertex watch-id)
  (close! chan))
