(ns thi.ng.fabric.facts.core
  #?@(:clj
      [(:require
        [thi.ng.fabric.core :as f]
        [clojure.set :as set]
        [clojure.core.async :as a :refer [go go-loop chan close! <! >! alts! timeout]]
        [taoensso.timbre :refer [debug info warn]])]
      :cljs
      [(:require-macros
        [cljs.core.async.macros :refer [go go-loop]]
        [cljs-log.core :refer [debug info warn]])
       (:require
        [thi.ng.fabric.core :as f]
        [clojure.set :as set]
        [cljs.core.async :refer [chan close! <! >! alts! timeout]])]))

(declare index-selection add-query! add-query-join!)

(defprotocol IFactGraph
  (fact-indices [_])
  (add-fact! [_ t])
  (remove-fact! [_ t]))

(defprotocol ICache
  (cached [_ type k])
  (cache! [_ type k v])
  (expire! [_ type k]))

(defprotocol IFactQuery
  (raw-pattern [_])
  (result-vertex [_]))

  

(defn- signal-fact
  [vertex op] [op (f/vertex-id vertex) @vertex])

(defn- collect-index
  [spo]
  (f/collect-pure
   (fn [val incoming]
     ;;(debug :update-index spo incoming)
     (debug :old-index val)
     (let [val (transduce
                (map (fn [[op id t]] [op id (nth t spo)]))
                (completing
                 (fn [acc [op id x]]
                   (case op
                     :add    (update acc x (fnil conj #{}) id)
                     :remove (if-let [idx (acc x)]
                               (if (= #{id} idx)
                                 (dissoc acc x)
                                 (update acc x disj id))
                               acc)
                     (do (warn "ignoring unknown index signal op:" op)
                         acc))))
                val incoming)]
       (debug :new-index val)
       val))))

(defn- signal-index-select
  [vertex [idx sel]]
  [idx (if sel (@vertex sel [nil]) (->> @vertex vals (mapcat identity) (set)))])

(def ^:private collect-select
  (f/collect-pure
   (fn [val incoming]
     (let [val (reduce (fn [acc [idx res]] (assoc acc idx res)) val incoming)]
       ;;(debug :coll-select val incoming)
       val))))

(defn- score-collect-min-signal-vals
  [num]
  (fn [vertex]
    (if (> num (count (vals (peek (f/uncollected-signals vertex))))) 0 1)))

(defn- score-collect-min-signals
  [num]
  (fn [vertex]
    (if (> num (count (f/uncollected-signals vertex))) 0 1)))

(defn- aggregate-select
  [g]
  (f/collect-pure
   (fn [_ incoming]
     (let [res (vals (peek incoming))]
       ;;(debug :agg-incoming res)
       (if (every? #(not= [nil] %) res)
         (->> res
              (into #{} (map #(disj % nil)))
              (sort-by count)
              (reduce set/intersection)
              (into #{}
                    (comp (map #(f/vertex-for-id g %))
                          (filter identity)
                          (map deref))))
         #{})))))

(defn- score-collect-join
  [^thi.ng.fabric.core.Vertex vertex]
  (if (and (seq (f/uncollected-signals vertex))
           (== (count (f/signal-map vertex)) 2))
    1 0))

(defn- collect-inference
  [g production]
  (fn [^thi.ng.fabric.core.Vertex vertex]
    (let [prev @vertex
          in   (reduce into #{} (f/uncollected-signals vertex))
          adds (set/difference in prev)
          inferred (mapcat production adds)]
      (debug (f/vertex-id vertex) :additions adds)
      (doseq [[op t :as inf] inferred]
        (case op
          :+ (do (debug :add-fact t)
                 (add-fact! g t))
          :- (do (debug :remove-fact t)
                 (remove-fact! g t))
          (warn "invalid inference:" inf)))
      (swap! (.-value vertex) set/union adds))))

(defn- qvar?
  "Returns true, if x is a qvar (a symbol prefixed with '?')"
  [x] (and (symbol? x) (= \? (.charAt ^String (name x) 0))))

(defn- bind-translator
  [vs? vp? vo? s p o]
  (if vs?
    (if vp?
      (if vo?
        (fn [r] {s (r 0) p (r 1) o (r 2)})
        (fn [r] {s (r 0) p (r 1)}))
      (if vo?
        (fn [r] {s (r 0) o (r 2)})
        (fn [r] {s (r 0)})))
    (if vp?
      (if vo?
        (fn [r] {p (r 1) o (r 2)})
        (fn [r] {p (r 1)}))
      (if vo?
        (fn [r] {o (r 2)})
        (fn [_] {})))))

(defn- fact-verifier
  [ts tp to vars varp varo]
  (cond
    (and vars varp varo) (cond
                           (= ts tp to) (fn [r] (= (r 0) (r 1) (r 2)))
                           (= ts tp) (fn [r] (and (= (r 0) (r 1)) (not= (r 0) (r 2))))
                           (= ts to) (fn [r] (and (= (r 0) (r 2)) (not= (r 0) (r 1))))
                           (= tp to) (fn [r] (and (= (r 1) (r 2)) (not= (r 0) (r 1))))
                           :else (constantly true))
    (and vars varp)      (if (= ts tp)
                           (fn [r] (= (r 0) (r 1)))
                           (fn [r] (not= (r 0) (r 1))))
    (and vars varo)      (if (= ts to)
                           (fn [r] (= (r 0) (r 2)))
                           (fn [r] (not= (r 0) (r 2))))
    (and varp varo)      (if (= tp to)
                           (fn [r] (= (r 1) (r 2)))
                           (fn [r] (not= (r 1) (r 2))))
    :else                (constantly true)))

(def ^:private fact-vertex-spec
  {::f/score-collect-fn (constantly 0)
   ::f/score-signal-fn  f/score-signal-with-new-edges})

(defn- index-vertex
  [g spo]
  (f/add-vertex!
   g {} {::f/collect-fn      (collect-index spo)
         ::f/score-signal-fn f/score-signal-with-new-edges}))

(defrecord FactGraph
    [g indices facts cache]
  f/IComputeGraph
  (add-vertex!
    [_ val vspec] (f/add-vertex! g val vspec))
  (remove-vertex!
    [_ v] (f/remove-vertex! g v))
  (vertex-for-id
    [_ id] (f/vertex-for-id g id))
  (vertices
    [_] (f/vertices g))
  (add-edge!
    [_ src dest sig opts] (f/add-edge! g src dest sig opts))
  f/IWatch
  (add-watch!
    [_ type id f] (f/add-watch! g type id f) _)
  (remove-watch!
    [_ type id] (f/remove-watch! g type id) _)
  IFactGraph
  (fact-indices
    [_] indices)
  (add-fact!
    [_ t]
    (or (@facts t)
        (let [{:keys [subj pred obj]} indices
              v (f/add-vertex! g t fact-vertex-spec)]
          (f/add-edge! g v subj signal-fact :add)
          (f/add-edge! g v pred signal-fact :add)
          (f/add-edge! g v obj  signal-fact :add)
          (swap! facts assoc t v)
          v)))
  (remove-fact!
    [_ t]
    (if-let [v (@facts t)]
      (let [{:keys [subj pred obj]} indices]
        (f/add-edge! g v subj signal-fact :remove)
        (f/add-edge! g v pred signal-fact :remove)
        (f/add-edge! g v obj  signal-fact :remove)
        (swap! facts dissoc t)
        (f/remove-vertex! g v)
        v)
      (warn "attempting to remove unknown fact:" t)))
  ICache
  (cached
    [_ type k] (get-in @cache [type k]))
  (cache!
    [_ type k v] (swap! cache assoc-in [type k] v) v)
  (expire!
    [_ type k] (swap! cache update type dissoc k) nil))
(defn fact-graph
  [g]
  (map->FactGraph
   {:indices {:subj (index-vertex g 0)
              :pred (index-vertex g 1)
              :obj  (index-vertex g 2)}
    :facts   (atom {})
    :cache   (atom {})
    :g       g}))

#?(:clj (prefer-method print-method clojure.lang.IRecord clojure.lang.IDeref))

(defrecord FactIndexSelection [index sel vertex]
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [v (f/add-vertex!
             g nil
             {::f/score-signal-fn f/score-signal-with-new-edges
              ::f/collect-fn      (f/collect-pure (fn [_ in] (peek in)))})]
      (f/add-edge! g index v signal-index-select sel)
      (cache! g ::index-sel sel (assoc _ :vertex v))))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? vertex parent)
      (do (warn :remove-index-sel sel)
          (f/disconnect-neighbor! index vertex)
          (f/remove-vertex! g vertex)
          true)
      (do (f/disconnect-neighbor! vertex parent)
          false))))
(defrecord BasicFactQuery [acc selections result pattern]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] pattern)
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [{:keys [subj pred obj]} (fact-indices g)
          [s p o] pattern
          sel-s (index-selection g [0 s])
          sel-p (index-selection g [1 p])
          sel-o (index-selection g [2 o])
          acc   (f/add-vertex! g {} {::f/collect-fn collect-select})
          res   (f/add-vertex!
                 g nil
                 {::f/collect-fn       (aggregate-select g)
                  ::f/score-signal-fn  f/score-signal-with-new-edges
                  ::f/score-collect-fn (score-collect-min-signal-vals 3)})
          this  (assoc _ :acc acc :result res :selections [sel-s sel-p sel-o])]
      ;; TODO add index selection vertices, use existing if possible
      (f/add-edge! g (:vertex sel-s) acc f/signal-forward nil)
      (f/add-edge! g (:vertex sel-p) acc f/signal-forward nil)
      (f/add-edge! g (:vertex sel-o) acc f/signal-forward nil)
      (f/add-edge! g acc             res f/signal-forward nil)
      (cache! g ::queries pattern this)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-query pattern)
          (expire! g ::queries pattern)
          (f/remove-vertex! g result)
          (f/remove-vertex! g acc)
          (run! #(f/remove-from-graph! % g acc) selections)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))
(defrecord ParametricFactQuery
    [sub-query result pattern]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] (mapv #(if-not (qvar? %) %) pattern))
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [[s p o]    pattern
          vs?        (qvar? s), vp? (qvar? p), vo? (qvar? o)
          vmap       (bind-translator vs? vp? vo? s p o)
          verify     (fact-verifier s p o vs? vp? vo?)
          res-tx     (comp (map #(if (verify %) (vmap %))) (filter identity))
          collect-fn (f/collect-pure
                      (fn [_ incoming]
                        (if-let [res (seq (peek incoming))]
                          (into #{} res-tx res)
                          #{})))
          sub-q      (add-query! g (raw-pattern _))
          res        (f/add-vertex!
                      g nil
                      {::f/collect-fn      collect-fn
                       ::f/score-signal-fn f/score-signal-with-new-edges})
          this       (assoc _ :sub-query sub-q :result res)]
      (f/add-edge! g (result-vertex sub-q) res f/signal-forward nil)
      (cache! g ::queries pattern this)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-param-query pattern)
          (expire! g ::queries pattern)
          (f/remove-vertex! g result)
          (f/remove-from-graph! sub-query g result)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))
(defrecord FactQueryJoin [lhs rhs result]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] nil)
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [lhs-v  (result-vertex lhs)
          rhs-v  (result-vertex rhs)
          lhs-id (f/vertex-id lhs-v)
          rhs-id (f/vertex-id rhs-v)
          join   (f/add-vertex!
                  g nil
                  {::f/collect-fn
                   (fn [vertex]
                     (let [sig-map (f/signal-map vertex)
                           a (sig-map lhs-id)
                           b (sig-map rhs-id)]
                       (debug (f/vertex-id vertex) :join-sets a b)
                       (f/set-value! vertex (set/join a b))))
                   ::f/score-collect-fn score-collect-join
                   ::f/score-signal-fn  f/score-signal-with-new-edges})]
      (f/add-edge! g lhs-v join f/signal-forward nil)
      (f/add-edge! g rhs-v join f/signal-forward nil)
      (assoc _ :result join)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-query-join)
          (f/remove-vertex! g result)
          (f/remove-from-graph! lhs g result)
          (f/remove-from-graph! rhs g result)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))
(defrecord FactQueryUnion [sub-queries result]
  #?@(:clj
       [clojure.lang.IDeref (deref [_] (when result @result))]
       :cljs
       [IDeref (-deref [_] (when result @result))])
  IFactQuery
  (raw-pattern
    [_] nil)
  (result-vertex
    [_] result)
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [res (f/add-vertex!
               g nil
               {::f/collect-fn
                (fn [vertex]
                  (let [subs (vals (f/signal-map vertex))]
                    (f/set-value! vertex (reduce (fn [acc res] (if (seq res) (into acc res) acc)) subs))))
                ::f/score-signal-fn f/score-signal-with-new-edges})]
      (run! #(f/add-edge! g (result-vertex %) res f/signal-forward nil) sub-queries)
      (assoc _ :result res)))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? result parent)
      (do (warn :remove-query-union)
          (f/remove-vertex! g result)
          (run! #(f/remove-from-graph! % g result) sub-queries)
          true)
      (do (f/disconnect-neighbor! result parent)
          false))))

(defn index-selection
  [g sel]
  (if-let [sel (cached g ::index-sel sel)]
    (do (debug :reuse-index-sel (:sel sel)) sel)
    (f/add-to-graph!
     (map->FactIndexSelection
      {:index ((fact-indices g) ([:subj :pred :obj] (first sel)))
       :sel   sel}) g)))

(defn add-query!
  [g pattern]
  (if-let [q (cached g ::queries pattern)]
    (do (debug :reuse-basic-query pattern) q)
    (f/add-to-graph!
     (map->BasicFactQuery {:pattern pattern}) g)))

(defn add-param-query!
  [g pattern]
  (if-let [q (cached g ::queries pattern)]
    (do (debug :reuse-param-query pattern) q)
    (f/add-to-graph!
     (map->ParametricFactQuery {:pattern pattern}) g)))

(defn add-join!
  [g lhs rhs]
  (f/add-to-graph!
   (map->FactQueryJoin {:lhs lhs :rhs rhs}) g))

;; TODO pre-sort patterns based on restrictiveness
(defn add-query-join!
  [g a b & more]
  (reduce
   (fn [acc p] (add-join! g acc (add-param-query! g p)))
   (add-join! g (add-param-query! g a) (add-param-query! g b))
   more))

(defn add-query-union!
  [g a b & queries]
  (f/add-to-graph!
   (map->FactQueryUnion {:sub-queries (into [a b] queries)}) g))

(defn add-query-filter!
  [g flt q]
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn      (f/collect-pure (fn [_ in] (filter flt (peek in))))
              ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    res))

(defn add-query-group-by!
  [g gfn q]
  (let [res (f/add-vertex!
             g nil
             {::f/collect-fn      (f/collect-pure (fn [_ in] (group-by gfn (peek in))))
              ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g (result-vertex q) res f/signal-forward nil)
    res))

(defn add-counter!
  [g src]
  (let [v (f/add-vertex!
           g nil
           {::f/collect-fn      (f/collect-pure (fn [_ in] (count (peek in))))
            ::f/score-signal-fn f/score-signal-with-new-edges})]
    (f/add-edge! g src v f/signal-forward nil)
    v))

(defrecord FactInferenceRule
    [id query patterns production inf]
  f/IGraphComponent
  (add-to-graph!
    [_ g]
    (let [q   (apply add-query-join! g patterns)
          inf (f/add-vertex!
               g #{} {::f/collect-fn (collect-inference g production)})]
      (f/add-edge! g (result-vertex q) inf f/signal-forward nil)
      (cache! g ::rules id (assoc _ :query q :inf inf))))
  (remove-from-graph!
    [_ g] (f/remove-from-graph! _ g nil))
  (remove-from-graph!
    [_ g parent]
    (if (f/none-or-single-user? inf parent)
      (do (warn :remove-rule id)
          (f/remove-vertex! g inf)
          (f/remove-from-graph! query g inf)
          true)
      (do (f/disconnect-neighbor! inf parent)
          false))))
(defn add-rule!
  [g id query production]
  (f/add-to-graph!
   (map->FactInferenceRule
    {:id id :patterns query :production production})
   g))

(def fact-log-transducer
  (comp
   (filter (fn [[op v]] (and (#{:add-vertex :remove-vertex} op) (vector? @v))))
   (map (fn [[op v]] [({:add-vertex :+ :remove-vertex :-} op) @v]))))

(defn add-fact-graph-logger
  [g log-fn]
  (let [ch        (chan 1024 fact-log-transducer)
        watch-id  (f/random-id)
        log->chan #(go (>! ch %))]
    (go-loop []
      (let [t (<! ch)]
        (when t
          (log-fn t)
          (recur))))
    (f/add-watch! g :add-vertex watch-id log->chan)
    (f/add-watch! g :remove-vertex watch-id log->chan)
    {:graph g :chan ch :watch-id watch-id}))

(defn remove-fact-graph-logger
  [{:keys [graph watch-id chan]}]
  (f/remove-watch! graph :add-vertex watch-id)
  (f/remove-watch! graph :remove-vertex watch-id)
  (close! chan))
