(ns thi.ng.domus.router
  (:require
   [thi.ng.domus.log :refer [debug info warn]]
   [thi.ng.domus.utils :as utils]
   [thi.ng.validate.core :as v]
   [clojure.string :as str]))

(defn set-location!
  [url] (set! (.-location js/window) url))

(defn get-route
  [] (-> js/window (.-location) (.-hash) (.split "/") (.slice 1)))

(defn format-route
  [base & params]
  (apply str (interpose "/" (into ["#" (name base)] params))))

(defn set-route!
  [id & params]
  (set! (.-hash (.-location js/window)) (apply format-route id params)))

(defn replace-route!
  [id & params]
  (.replace (.-location js/window) (apply format-route id params)))

(defn local?
  []
  (let [host (.-hostname (.-location js/window))]
    (re-find #"localhost|(192\.168\.)|(127\.0\.0\.1)" host)))

(defn match-route*
  [curr route]
  (if (= (count curr) (count route))
    (reduce
     (fn [acc [a b]]
       (cond
        (= a b) acc
        (keyword? b) (assoc acc b a)
        :else (reduced nil)))
     {} (partition 2 (interleave curr route)))))

(defn coerce-route-params
  [specs params]
  (reduce
   (fn [params [k {:keys [coerce]}]]
     (if coerce
       (if-let [pv (try (coerce (params k)) (catch js/Error e))]
         (assoc params k pv)
         (reduced nil))
       params))
   params specs))

(defn validate-route-params
  [specs params]
  (if-let [params (coerce-route-params specs params)]
    (let [valspecs (filter #(comp :validate val) specs)]
      (if (seq valspecs)
        (let [[params err] (->> valspecs
                                (reduce #(assoc % (key %2) (:validate (val %2))) {})
                                (v/validate params))]
          (if-not err params))
        params))))

(defn match-route
  [routes]
  (let [curr (get-route)]
    (some
     (fn [{:keys [match bindings controller]}]
       (if-let [params (match-route* curr match)]
         (if-let [params (if bindings (validate-route-params bindings params) params)]
           {:controller controller
            :params params
            :route curr})))
     routes)))

(defn route-for-id
  [routes id args]
  (if-let [route (some #(if (= id (:id %)) %) routes)]
    (->>  (:match route)
          (reduce
           (fn [acc x] (conj acc (if (keyword? x) (args x) x)))
           [])
          (str/join "/"))))

(defn router
  [routes default-id route-changed]
  (let [routes (filter (complement :disabled) routes)]
    (fn []
      (let [info (match-route routes)]
        (if info
          (route-changed info)
          (do
            (warn "no matching route:" (get-route) "redirect to default...")
            (set-route! (route-for-id routes default-id {}))))))))

(defn start!
  [router]
  (.addEventListener js/window "hashchange" router)
  (router))
