(ns shadow.build.targets.external-index
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [shadow.build.data :as data]
            [shadow.cljs.util :as util]))

(defn into-set [current other]
  (set (into current other)))

(defn merge-js-deps* [out {:keys [js-deps] :as ns-info}]
  (-> out
      (util/reduce->
        (fn [out require]
          (update out require merge {}))
        (get-in ns-info [:meta :external/assets]))
      (util/reduce-kv->
        (fn [out id {:keys [as refer default rename]}]
          (if (or (util/is-absolute? id)
                  (util/is-relative? id))
            out
            (-> out
                (cond->
                  as
                  (assoc-in [id :import-all] true)
                  refer
                  (update-in [id :refer] into-set refer)
                  default
                  (assoc-in [id :default] true)
                  rename
                  (update-in [id :refer] into-set (keys rename))))))
        js-deps)))

(defn merge-js-deps [ns-info-seq]
  (->> ns-info-seq
       (reduce merge-js-deps* {})
       (reduce-kv
         (fn [v require info]
           (conj v (assoc info :require require)))
         [])
       ;; to make things sort of stable
       (sort-by :require)
       (vec)))

(comment
  (merge-js-deps
    '[{:js-deps {"just-refer" {:refer (X Y)}}}
      {:js-deps {"just-refer" {:refer (X Z)}}}
      {:js-deps {"./rel.js" {:as x}} :meta {:external/assets ["somewhere/foo.css"]}}
      {:js-deps {"/abs.js" {:as x}}}
      {:js-deps {"default" {:default D}}}
      {:js-deps {"default" {:refer (A B C)}}}
      {:js-deps {"uses-as" {:as something :refer (X)}}}]))

(defn refer-entries [idx refer]
  (->> refer
       (map-indexed (fn [ridx refer]
                      (str refer " as r_" idx "_" ridx)))
       (str/join ", ")))

(defn refer-obj [idx default refer]
  (-> []
      (cond->
        default
        (conj (str "default: d_" idx)))
      (into (map-indexed (fn [ridx refer] (str refer ": r_" idx "_" ridx)) refer))
      (->> (str/join ", "))))

(defn generate-esm [state idx {:keys [require refer import-all default] :as entry}]
  (-> state
      (update :imports conj (-> "import "
                                (cond->
                                  import-all
                                  (str "* as ns_" idx " ")
                                  (and (not import-all) default)
                                  (str "d_" idx ", ")
                                  (and (not import-all) (seq refer))
                                  (str "{" (refer-entries idx refer) "}"))
                                (str " from \"" require "\";")))

      (update :body conj (str "ALL[\"" require "\"] = "
                              (if import-all
                                (str "ns_" idx)
                                (str "{" (refer-obj idx default refer) "}"))
                              ";"))))

(defn generate-cjs [state idx {:keys [require]}]
  (update state :body conj (str "ALL[\"" require "\"] = require(\"" require "\");")))

(defn join-lines [all]
  (str/join "\n" all))

(defn wrap-boilerplate [{:keys [imports body]}]
  (-> ["// WARNING: DO NOT EDIT!"
       "// THIS FILE WAS GENERATED BY SHADOW-CLJS AND WILL BE OVERWRITTEN!"
       ""]
      (into imports)
      (conj "var ALL = {};")
      (into body)
      (conj (slurp (io/resource "shadow/build/targets/external_boilerplate.js")))
      (conj "")
      (join-lines)))

(defn generate [ext-info]
  (->> ext-info

       ;; FIXME: need to figure out how to do proper compat first
       ;; (:require ["react" :as X]) will be invalid since it is commonjs
       ;; and future webpack would only allow
       ;;   import X from "react";
       ;; which maps to
       ;;   (:require ["react" :default X])
       ;; but that would break everything using :as ... which is everything ...
       ;; (reduce-kv generate-esm {:imports [] :body []})

       (reduce-kv generate-cjs {:imports [] :body []})
       (wrap-boilerplate)))

(comment
  (generate
    '[{:require "react" :import-all true}
      {:require "something" :refer #{X A Y}, :default true}]))

(defn flush-js [{:keys [build-sources] :as state}]
  (let [ext-info
        (->> build-sources
             (map #(data/get-source-by-id state %))
             (filter #(= :cljs (:type %)))
             (map :ns-info)
             (merge-js-deps))]

    (if (= ext-info (::ext-info state))
      state
      (let [output-to (io/file (get-in state [:js-options :external-index] "target/external.js"))]
        (io/make-parents output-to)
        (spit output-to (generate ext-info))
        (assoc state ::ext-info ext-info)))))
