;; *   Silvur
;; *
;; *   Copyright (c) Tsutomu Miyashita. All rights reserved.
;; *
;; *   The use and distribution terms for this software are covered by the
;; *   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; *   which can be found in the file epl-v10.html at the root of this distribution.
;; *   By using this software in any fashion, you are agreeing to be bound by
;; * 	 the terms of this license.
;; *   You must not remove this notice, or any other, from this software.

(ns silvur.datetime
  (:gen-class)
  (:import [java.util Date GregorianCalendar]
           [java.time.format DateTimeFormatter]
           [java.time LocalTime Period DayOfWeek
            ZonedDateTime ZoneOffset ZoneId Instant Duration ZoneRegion LocalDateTime]
           [java.time.temporal ChronoUnit ChronoField TemporalAdjuster TemporalAdjusters
            Temporal TemporalUnit TemporalAmount]))


;; Minimum unit is 'second'
(def UTC (ZoneId/of "UTC"))
(def JST (ZoneId/of "Asia/Tokyo"))
(def NYC (ZoneId/of "America/New_York"))

(def FORMAT {:JP "YYYY/MM/dd HH:mm:ss"
             :US "MM/dd/YYYY HH:mm:ss"})

(defonce ^:dynamic *tz* JST)
(defonce ^:dynamic *precision* (ChronoUnit/SECONDS)) 

(defprotocol TimeExchange
  (minutes-of-week [t])
  (first-date-of-week [t])
  (adjust [t duration])
  (-day [i])
  (-hour [i])
  (-minute [i])
  (-second [i])
  (before [i diff])
  (after [i diff])
  (-year [t])
  (-month [t])
  (-week [t])
  (+time [t duration]))

(defprotocol At
  (at-zone [t zid])
  (at-offset [t]))

(extend-protocol At
  Instant
  (at-zone [^Instant t ^ZoneId zid]
    (.atZone t zid))
  (at-offset [t]
    (.atOffset t (-> ^ZoneId *tz* (.getRules) (.getOffset t))))
  LocalDateTime
  (at-zone [^LocalDateTime t ^ZoneId zid]
    (.atZone t zid))
  (at-offset [^LocalDateTime t]
    (.atOffset t (-> ^ZoneId *tz* (.getRules) (.getOffset t)))))

(defn set-default-precision! [x]
  (alter-var-root #'*precision* (constantly (condp = x
                                              :second (ChronoUnit/SECONDS)
                                              :milli (ChronoUnit/MILLIS)
                                              :nano (ChronoUnit/NANOS)
                                              (ChronoUnit/SECONDS)))))
(defn set-default-tz! [tz]
  (alter-var-root #'*tz* (constantly tz)))

(defn inst< [^long from-epoch]
  (-> ^ChronoUnit
      *precision*
      (.getDuration )
      (.multipliedBy  from-epoch )
      (.addTo (Instant/EPOCH))))

(defn zone-offset
  ([]
   (zone-offset *tz*))
  ([^ZoneId zid ^LocalDateTime ldt]
   (-> ^ZoneId zid (.getRules) (.getTransition ldt))))

(defn inst> [^Instant i]
  (let [d (.plus (.multipliedBy (.getDuration (ChronoUnit/NANOS)) (.getNano i))
                 (.multipliedBy (.getDuration (ChronoUnit/SECONDS)) (.getEpochSecond i)))] 
    (condp = (str *precision*)
      "Seconds" (.toSeconds d)
      "Millis" (.toMillis d)
      "Nanos" (.toNanos d)
      (throw (ex-info "No support" {:precision *precision*})))))


(defmulti -datetime (fn [arg & _]  (class arg)))

(defmethod -datetime Integer [arg & rest]
  (apply -datetime (long arg) rest))

(defmethod -datetime Long [arg & rest]
  (if (and (>= 9999 arg 1970)
           (not (instance? clojure.lang.Keyword (first rest))))
    (LocalDateTime/of arg (nth rest 0 1) (nth rest 1 1) (nth rest 2 0) (nth rest 3 0) (nth rest 4 0) (nth rest 5 0))
    (LocalDateTime/ofInstant (inst< arg) *tz*)))

(defmethod -datetime String [arg & rest]
  (.format ^LocalDateTime (apply -datetime rest) (DateTimeFormatter/ofPattern arg)))

(defmethod -datetime clojure.lang.Keyword [arg & rest]
  (.format ^LocalDateTime (apply -datetime rest) (condp = arg
                                                   :basic-iso-date (DateTimeFormatter/BASIC_ISO_DATE)
                                                   :iso-date (DateTimeFormatter/ISO_DATE)
                                                   :iso-date-time (DateTimeFormatter/ISO_DATE_TIME)
                                                   (DateTimeFormatter/ISO_INSTANT))))


(defmethod -datetime ZonedDateTime [arg & rest] (.toLocalDateTime arg))

(defmethod -datetime LocalDateTime [arg & rest] arg)

(defmethod -datetime clojure.lang.LazySeq [arg & _]
  (map -datetime arg))

(defmethod -datetime clojure.lang.PersistentVector [arg & _]
  (apply -datetime arg))

(defmethod -datetime Date [arg & rest]
  (-datetime (.getTime ^Date arg) :milli))

(defmethod -datetime clojure.lang.PersistentArrayMap [arg & _]
  (-datetime (or (:datetime arg)
                 (:time arg)
                 (:t arg))))

(defmethod -datetime clojure.lang.PersistentHashMap [arg & _]
  (-datetime (or (:datetime arg)
                 (:time arg)
                 (:t arg))))


(defmethod -datetime :default [arg & rest]
  (LocalDateTime/now))


(defn datetime
  ([]
   (-> (LocalDateTime/now)
       (.truncatedTo *precision*)))
  ([arg & rest]
   (apply -datetime arg (->> rest vector (keep identity) flatten))))

(defn datetime*
  ([]
   (datetime* (LocalDateTime/now ^ZoneId *tz*)))
  ([arg & rest]
   (-> ^LocalDateTime
       (apply datetime arg rest)
       (at-offset)
       (.toInstant)
       ;;(inst>)
       )))


(defn date
  ([]
   (date (datetime)))
  ([^LocalDateTime ldt]
   (Date/from (.toInstant (at-offset ldt)))))

(defn vec< [^LocalDateTime ldt]
  [(.getYear ldt) (.getMonthValue ldt) (.getDayOfMonth ldt)
   (.getHour ldt) (.getMinute ldt) (.getSecond ldt)])


(extend-protocol TimeExchange
  Long
  (-year [i]
    (Period/ofYears i))
  (-month [i]
    (Period/ofMonths i))
  (-week [i]
    (Period/ofWeeks i))
  (-day [i]
    (Period/ofDays i))
  (-hour [i]
    (Duration/ofHours i))
  (-minute [i]
    (Duration/ofMinutes i))
  (-second [i]
    (Duration/ofSeconds i))
  (adjust [v duration-or-period]
    (-> (datetime v)
        (.truncatedTo (proxy [TemporalUnit] []
                        (getDuration []
                          duration-or-period)))
        (datetime*)))
  (+time [v duration-or-period]
    (datetime* (+time (datetime v) duration-or-period)))
  (first-date-of-week [i]
    (first-date-of-week (datetime i)))
  (minutes-of-week [i]
    (minutes-of-week (datetime i)))

  java.time.LocalDateTime
  (-year [this]
    (.getYear this))
  (-month [this]
    (.getMonthValue this))
  (-hour [this]
    (.getHour this))
  (-minute [this]
    (.getMinute this))
  (-second [this]
    (.getSecond this))
  (adjust [t duration-or-period]
    (condp instance? duration-or-period
      TemporalAdjuster (.with t duration-or-period)
      Duration (.truncatedTo t (proxy [TemporalUnit] []
                                  (getDuration []
                                    duration-or-period)))
      Period (.with t (proxy [TemporalAdjuster] []
                        (adjustInto [z]
                          (.truncatedTo (.subtractFrom duration-or-period z)
                                        (proxy [TemporalUnit] []
                                          (getDuration []
                                            (-hour 24)))))))
      (throw (ex-info "Not duration or period" {}))))
  
  (+time [t duration-or-period]
    (.plus t duration-or-period))
  (minutes-of-week [t]
    (+
     (* (.getMinute t))
     (* 60 (.getHour t))
     (* 60 24 (mod (-> t (.getDayOfWeek) (.getValue)) 7))))
  
  (first-date-of-week [t]
    (-> (adjust t (TemporalAdjusters/previous (DayOfWeek/MONDAY)))
        (.truncatedTo (ChronoUnit/DAYS))))

  (before [t duration-or-period]
    (.minus t duration-or-period))

  (after [t duration-or-period]
    (.plus t duration-or-period))) 

(defn day
  ([] (-day 1)) 
  ([i] (-day i)))
(defn hour
  ([] (-hour 1)) 
  ([i] (-hour i)))
(defn minute
  ([] (-minute 1)) 
  ([i] (-minute i)))

(defn until 
  ([^Temporal t]
   (until t *precision*))
  ([^Temporal t ^TemporalUnit tu]
   (.until (datetime) t tu)))

(defn between
  ([^Temporal t0 ^Temporal t1]
   (between *precision* t0 t1))
  ([^TemporalUnit temporal-unit ^Temporal t0 ^Temporal t1]
   (.between temporal-unit t0 t1)))

(defn today []
  (.truncatedTo (datetime) (ChronoUnit/DAYS)))

(defn yesterday []
  (adjust (datetime) (day)))

(defn tomorrow []
  (-> (after (datetime) (day))
      (.truncatedTo (ChronoUnit/DAYS))))

(defn time-seq
  ([delta]
   (time-seq (datetime) delta))
  ([t delta]
   (lazy-seq
    (cons (adjust t delta) (time-seq (before t delta) delta)))))

