(ns silvur.build
  (:require [badigeon.javac :refer (javac)]
            [badigeon.jar :refer (jar)]
            [badigeon.deploy :refer (deploy)]
            [badigeon.sign :refer (sign)]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.xml :as xml]
            [clojure.tools.cli :refer (parse-opts)]
            [taoensso.timbre :as log]))

(defn usage []
  (println "Usage:"))

(def option-schemas
  {:top [["-h" "--help" "This help"]]
   :javac [["-h" "--help" "This help"]
           ["-s" "--source-dir DIRECTORY" "Source directory"]]
   :jar [["-h" "--help" "This help"]
         ["-a" "--artifact-id ID" "ID to be registered"
             :parse-fn symbol]
         ["-v" "--version VERION" "Version"
          :default "0.0.1"]]
   :deploy [["-h" "--help" "This help"]
            ["-a" "--artifact-id ID" "ID to be registered"
             :parse-fn symbol]
            ["-v" "--version VERSION" "Version"]
            [nil "--gpg-key-id" "GPG id to be signed"]
            ["-f" "--file-path PATH" "File to be deployed"]]
   :uberjar [["-h" "--help" "This help"]
             ["-v" "--version VERSION" "Version"
              :default "0.0.1"]]})

(defmulti call-function (fn [op opts] op))

(defmethod call-function :javac [op {:keys [arguments options summary]}]
  (if (:help options)
    (println summary)
    (if-let [src (:source-dir options)]
      (do (println "Compiling" src)
          (javac (:source-dir options)))
      (println "Please specify source directory"))))

(defmethod call-function :jar [op {:keys [arguments options summary]}]
  (if (:help options)
    (println summary)
    (let [opts-map (clojure.set/rename-keys options {:version :mvn/version})
          project-name (-> (io/file ".")
                           (.getCanonicalPath)
                           (io/as-file)
                           (.getName)
                           symbol)]
      (do (println "Packaging jar:" opts-map)
          (when (.exists (io/as-file "pom.xml"))
            (io/delete-file "pom.xml"))
          (let [path (jar (or (:artifact-id options) project-name)
                          opts-map)]
            (println "Completed")
            path)))))

(defmethod call-function :deploy [op {:keys [arguments options summary]}]
  (if (:help options)
    (println summary)
    (let [{:keys [artifact-id file-path version gpg-key-id]} options
          artifacts [{:file-path file-path :extension "jar"}
                     {:file-path "pom.xml" :extension "pom"}]]
      (if (and artifact-id file-path version gpg-key-id)
        (let [s-artifacts (sign artifacts gpg-key-id)]
          (deploy artifact-id version s-artifacts {:id "clojars" :url "https://repo.clojars.org/"}))
        (println "Please specify artifact-id, version, gpg-key-id and file-path" )))))


(defmethod call-function :uberjar [op opts]
  (prn "Uberjar" opts)
  )

(defn print-usage [summary]
  (println)
  (println "Usage:" "<command> [options]")
  (println)  
  (println "- Commands:")
  (println " " (str/join "," (map name (keys (dissoc option-schemas :top)))))
  (println)  
  (println "- Options:")
  (println summary))


(defn -main [& args]
  (log/set-level! :info)
  (let [{:keys [arguments options summary]} (parse-opts args (option-schemas :top))
        op (keyword (first arguments))]
    (if (and (:help options) (nil? op))
      (print-usage summary)
      (call-function op (update (parse-opts args (option-schemas op)) :arguments rest)))))


(defn easy-jar [v]
  (call-function :jar {:options {:artifact-id (symbol "theorems/silvur") :version v}}))

(defn easy-deploy [gpg-key-id v]
  (call-function :deploy {:options {:artifact-id (symbol "theorems/silvur")
                                    :gpg-key-id gpg-key-id
                                    :file-path (easy-jar v)
                                    :version v}}))
