;; *   Silvur
;; *
;; *   Copyright (c) Tsutomu Miyashita. All rights reserved.
;; *
;; *   The use and distribution terms for this software are covered by the
;; *   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; *   which can be found in the file epl-v10.html at the root of this distribution.
;; *   By using this software in any fashion, you are agreeing to be bound by
;; * 	 the terms of this license.
;; *   You must not remove this notice, or any other, from this software.

(ns silvur.util
  (:gen-class)
  (:require [clojure.string :as str :refer [upper-case lower-case]]
            [muuntaja.core :as m]
            [org.httpkit.client :as http]
            [clojure.java.io :as io]
            [postal.core :as postal]
            [nrepl.server :as nrepl]
            [nrepl.core :refer (message client)]
            [nrepl.transport :as transport])
  (:import [java.security MessageDigest]
           [java.io File]
           [java.nio.charset Charset]
           [java.util.stream Collectors]
           [java.nio.file Path Paths Files LinkOption OpenOption]
           [java.nio.file.attribute FileAttribute]
           [java.net URI]
           [javax.net.ssl SNIHostName SNIServerName SSLEngine SSLParameters X509TrustManager]
           [java.util.concurrent LinkedBlockingQueue TimeUnit]))

(defmacro uuid
  ([]
   `(java.util.UUID/randomUUID))
  ([s]
   `(java.util.UUID/fromString ~s)))

(defn sha1sum [& vals]
  (let [sha1 (MessageDigest/getInstance "SHA1")]
    (doseq [v vals]
      (.update sha1 (.getBytes (str v) "UTF-8")))
    (str/join (map #(format "%02x" %) (.digest sha1)))))

(defmacro uk [s]
  `(try (keyword (upper-case (name ~s))) (catch Exception e# nil)))

(defmacro us [s]
  `(try (upper-case (name ~s)) (catch Exception e# nil)))

(defmacro lk [s]
  `(try (keyword (lower-case (name ~s))) (catch Exception e# nil)))

(defmacro ls [s]
  `(try (lower-case (name ~s)) (catch Exception e# nil)))

(defn json->map [x]
  (m/decode "application/json" x))

(defn map->json [x]
  (m/encode "application/json" x))


;; NIO

(extend Path
  io/IOFactory
  {:make-reader (fn [path & {:keys [encoding]}]
                  (Files/newBufferedReader path (Charset/forName encoding)))
   :make-writer (fn [path opts]
                  (Files/newBufferedWriter path (into-array OpenOption opts)))
   :make-input-stream (fn [^Path path opts] (io/make-input-stream (.toFile path) opts))
   :make-output-stream (fn [^Path path opts] (io/make-output-stream (.toFile path) opts))})

(defprotocol PathOperation
  (path [f]))

(extend-protocol PathOperation
  String
  (path [s] (.toPath (io/file s)))
  File
  (path [f] (.toPath f))
  Path
  (path [p] p))


(defn pwd []
  (.toAbsolutePath ^Path (path "")))

(defn exists? [a-path]
  (Files/exists (path a-path) (into-array LinkOption [])))

(defn absolute-path [a-path]
  (.toAbsolutePath ^Path (path a-path)))

(defn readable? [a-path]
  (Files/isReadable (path a-path)))

(defn mkdir [a-path]
  (Files/createDirectories (path a-path) (into-array FileAttribute [])))

(defn dir? [a-path]
  (Files/isDirectory (path a-path) (into-array LinkOption [])))

(defn rm [a-path]
  (Files/deleteIfExists (path a-path)))

(defn path-seq [a-path]
  (-> (Files/list (path a-path))
      (.collect (Collectors/toList))))

(defn size [a-path]
  (Files/size (path a-path)))


;; SSL
(defn sni-configure [^SSLEngine ssl-engine ^URI uri]
  (let [^SSLParameters ssl-params (.getSSLParameters ssl-engine)]
    (.setServerNames ssl-params [(SNIHostName. (.getHost uri))])
    (.setUseClientMode ssl-engine true)
    (.setSSLParameters ssl-engine ssl-params)))

;; When using SNI, it should make a client as below for httpkit
;; (defonce sni-client
;;   ;; SNI (Server Name Indication) is a TLS extension to multiplex multiple SSL termination on a single IP  
;;   (http/make-client {:ssl-configurer ssl/sni-configure}))

(defn engine []
  (let [tm (reify javax.net.ssl.X509TrustManager
             (getAcceptedIssuers [this] (make-array  java.security.cert.X509Certificate 0))
             (checkClientTrusted [this chain auth-type])
             (checkServerTrusted [this chain auth-type]))
        client-context (javax.net.ssl.SSLContext/getInstance "TLSv1.2")]
    (.init client-context nil
           (into-array javax.net.ssl.TrustManager [tm])
           nil)
    (doto (.createSSLEngine client-context)
        (.setUseClientMode true))))

;; SMTP
(defn mail [& {:keys [from to subject host port contents] :or {host "localhost" port 25}}]
  ;; Ex. (mail :host "lebesgue" :from "analysis@theorems.co" :to ["myst3m@gmail.com"] :subject "Test" :contents {:text/plain "Hello"})
  (postal/send-message {:host host}
                       {:from from :to to :subject subject
                        :body (reduce conj [] (map (fn [[type content]]
                                                     (hash-map :type (condp = type
                                                                       :attachment type
                                                                       :inline type
                                                                       (clojure.string/replace (str type "; charset=utf-8") #":" ""))
                                                               :content content))
                                                   contents))}))

;; Net
(defn hex<-text [text-bytes]
  ;; text-bytes [0x38 0x36 0x3e 0x30]
  ;; => 0x86e0
  (str/join "" (map #(format "%x" (- % 0x30)) text-bytes)))

(defn digit<-text [text-bytes]
  ;; text-bytes [0x38 0x36 0x3e 0x30]
  ;; => 34528
  (str (Long/parseLong (hex<-text text-bytes) 16)))

(defn parse-binary-message [format-coll data-coll]
  ;; format-coll: [1 3 3 4 4 4 4 4 4]
  ;;              [[1 text<-bytes] [3 text<-bytes] ....]
  (loop [sf format-coll
         pdata []
         zs data-coll]
    (let [v (first sf)
          [s cf] (cond
                   (number? v) [v]
                   (sequential? v) v
                   :else [])]
      (if-not (seq sf)
        pdata
        (recur (rest sf) (conj pdata ((or cf identity) (take s zs))) (drop s zs))))))



;; nREPL

(defn nrepl-start [& {:keys [ip port cider] :or {ip "0.0.0.0" port 7888}}]
  (letfn [(nrepl-handler []
            (require 'cider.nrepl)
            (ns-resolve 'cider.nrepl 'cider-nrepl-handler))]

    (if cider
      (nrepl/start-server :port port :bind ip :handler (nrepl-handler))
      (nrepl/start-server :port port :bind ip
                          :transport-fn transport/tty
                          :greeting-fn (fn [transport]
                                         (transport/send transport
                                                         {:out (str "\nWelcome to  nREPL !\n\n"
                                                                    "user=> ")}))))
    (println (str "Booted nREPL server on " ip  ":" port (when cider (str " with cider option"))))))


;; (comment
;;   (def c (client (client-transport "http://localhost:12345") 1000))
;;   (message c {:op "eval" :code "(+ 1 2 3)"}))

(defn nrepl-http-transport
  "Returns an nREPL client-side transport to connect to HTTP nREPL
   endpoints implemented by `ring-handler`.

   This fn is implicitly registered as the implementation of
   `nrepl.core/url-connect` for `http` and `https` schemes;
   so, once this namespace is loaded, any tool that uses `url-connect`
   will use this implementation for connecting to HTTP and HTTPS
   nREPL endpoints."
  [url]
  (let [incoming (LinkedBlockingQueue.)
        fill #(when-let [responses (->> (io/reader %)
                                        line-seq
                                        rest
                                        drop-last
                                        (map json->map)
                                        (remove nil?)
                                        seq)]
                (.addAll incoming responses))
        session-cookies (atom nil)
        http (fn [& [msg]]
               (let [{:keys [cookies body] :as resp} @((if msg http/post http/get)
                                                       url
                                                       (merge {:as :stream
                                                               :cookies @session-cookies}
                                                              (when msg {:form-params msg})
                                                              ;;(when http-headers {:headers http-headers})
                                                              ))]
                 (swap! session-cookies merge cookies)
                 (fill body)))]
    (transport/->FnTransport
     (fn read [timeout]
       (let [t (System/currentTimeMillis)]
         (or (.poll incoming 0 TimeUnit/MILLISECONDS)
             (when (pos? timeout)
               (http)
               (recur (- timeout (- (System/currentTimeMillis) t)))))))
     http
     (fn close []))))
