(ns silvur.nrepl
  (:require [nrepl.server :as nrepl]
            [nrepl.core :refer (message client)]
            [nrepl.transport :as transport]
            [silvur.util :as util]
            [org.httpkit.client :as http]
            [clojure.java.io :as io]
            )
  (:import
   (java.util.concurrent LinkedBlockingQueue TimeUnit)))

(defn start [& {:keys [ip port cider] :or {ip "0.0.0.0" port 7888}}]
  (letfn [(nrepl-handler []
            (require 'cider.nrepl)
            (ns-resolve 'cider.nrepl 'cider-nrepl-handler))]

    (if cider
      (nrepl/start-server :port port :bind ip :handler (nrepl-handler))
      (nrepl/start-server :port port :bind ip
                          :transport-fn transport/tty
                          :greeting-fn (fn [transport]
                                         (transport/send transport
                                                         {:out (str "\nWelcome to  nREPL !\n\n"
                                                                    "user=> ")}))))
    (println (str "Booted nREPL server on " ip  ":" port (when cider (str " with cider option"))))))


;; (comment
;;   (def c (client (client-transport "http://localhost:12345") 1000))
;;   (message c {:op "eval" :code "(+ 1 2 3)"}))

(defn nrepl-http-transport
  "Returns an nREPL client-side transport to connect to HTTP nREPL
   endpoints implemented by `ring-handler`.

   This fn is implicitly registered as the implementation of
   `nrepl.core/url-connect` for `http` and `https` schemes;
   so, once this namespace is loaded, any tool that uses `url-connect`
   will use this implementation for connecting to HTTP and HTTPS
   nREPL endpoints."
  [url]
  (let [incoming (LinkedBlockingQueue.)
        fill #(when-let [responses (->> (io/reader %)
                                        line-seq
                                        rest
                                        drop-last
                                        (map util/json->map)
                                        (remove nil?)
                                        seq)]
                (.addAll incoming responses))
        session-cookies (atom nil)
        http (fn [& [msg]]
               (let [{:keys [cookies body] :as resp} @((if msg http/post http/get)
                                                       url
                                                       (merge {:as :stream
                                                               :cookies @session-cookies}
                                                              (when msg {:form-params msg})
                                                              ;;(when http-headers {:headers http-headers})
                                                              ))]
                 (swap! session-cookies merge cookies)
                 (fill body)))]
    (transport/->FnTransport
     (fn read [timeout]
       (let [t (System/currentTimeMillis)]
         (or (.poll incoming 0 TimeUnit/MILLISECONDS)
             (when (pos? timeout)
               (http)
               (recur (- timeout (- (System/currentTimeMillis) t)))))))
     http
     (fn close []))))
