(ns silvur.dpkg
  {:boot/export-tasks true}
  (:require [boot.core :as core]
            [boot.task.built-in :as built-in]
            [radicalzephyr.boot-dpkg :refer :all]
            [boot.pod :as pod]
            [boot.util :as util]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]))

(defonce NL "\n")

(defn gen-systemd-conf [{:keys [command user group app type]}]
  (let [user (or user app)]
    (str
     "[Unit]" 
     NL "  Description=" (str "systemd config file for " app ) 
     NL
     NL "[Service]"
     NL "  User=" user
     NL "  Group=" (or group user)
     NL "  Type=" (or type "simple")
     NL "  WorkingDirectory=" (str "/run/" user)
     NL "  RuntimeDirectory=" user
     NL "  RuntimeDirectoryMode=0755"
     NL "  ExecStart=" command
     NL
     NL "[Install]"
     NL "  WantedBy=multi-user.target"
     NL)))

(defn preinst [{:keys [app user group]}]
  (let [user (or user app)
        group (or group user)]
    (str
     "#/bin/bash"
     NL
     NL "groupadd " group " > /dev/null 2>&1"
     NL "useradd -M -s /usr/sbin/nologin  -d " "/var/lib/" app  " -g adm " user " > /dev/null 2>&1"
     NL
     NL "exit 0"
     NL)))

(defn- -exists? [& paths]
  (str "[[ -f " (str/join "/" paths) " ]] "))

(defn- -chown [o g & paths]
  (let [path (str/join "/" paths)]
    (str (-exists? (pr-str path)) " && " " chown " o "." g " " path)))

(defn- -chown-r [o g & paths]
  (let [path (str/join "/" paths)]
    (str " chown -R " o "." g " " path " >&2")))

(defn- -chmod [mode & paths]
  (let [path (str/join "/" paths)]
    (str (-exists? (pr-str path)) " && " " chmod " mode " " path)))

(defn- -chmod-r [mode & paths]
  (let [path (str/join "/" paths)]
    (str  "chmod -R " mode " " path " >&2 ")))

(defn postinst [{:keys [app user group]}]
  (let [user (or user app)
        group (or group user)
        app-path (str "/usr/bin/" app)]
    (str
     "#!/bin/bash"
     NL 
     NL "case $1 in"
     NL " configure|reconfigure)"
     NL "    " (-chown-r "root" "root" "/etc" app)
     NL "    " (-chown-r "root" "root" "/usr/share/doc" app)
     NL "    " (-chown user group "/etc" app "config.edn")
     NL "    " (-chown-r user group "/usr/share/java" (str app "*.jar"))
     NL "    " (-chmod-r "644" "/usr/share/java" (str app "*.jar"))          
     NL "    " (-chown user group "/etc" app "config.edn")
     NL "    " (-chown user group "/var/log" app)     
     NL "    " (-chown user group "/var/lib" app)     
     NL "    " (-chown user group app-path)
     NL "    " (-chmod "640" "/etc" app "config.edn")
     NL "    " (-chmod "755" "/usr/bin" app)
     NL "    udevadm control -R"
     NL "    udevadm trigger"
     NL "    systemctl daemon-reload"
     NL "    systemctl enable " app
     NL "    ;;"
     NL "esac"
     NL NL "exit 0")))

(defn prerm [{:keys [app user group]}]
  (let [user (or user app)]
    (str
     "#!/bin/bash"
     NL
     NL "case $1 in"
     NL "  purge)"
     NL "    systemctl daemon-reload"
     NL "    systemctl stop " app
     NL "    systemctl disable " app
     NL "    ;;"
     NL "  remove|upgrade|failed-upgrade|abort-install|abort-upgrade|disappear)"
     NL "    ;;"
     NL "esac"
     NL
     NL "exit 0")))

(defn postrm [{:keys [app user group]}]
  (let [user (or user app)
        group (or group user)]
    (str
     "#!/bin/bash"
     NL
     NL "case $1 in"
     NL "purge)"
     NL "  rm -rf /var/run/" app
     NL "  rm -rf /var/log/" app
     NL "  userdel " user " > /dev/null 2>&1"
     NL "  groupdel " group " > /dev/null 2>&1"     
     NL "  ;;"
     NL "remove|upgrade|failed-upgrade|abort-install|abort-upgrade|disappear)"
     NL "  ;;"
     NL "esac"
     NL
     NL "exit 0"
     NL)))

(core/deftask dpkg-prepare [a app APPLICATION str "The name of the application"
                            u user USER str "User"
                            g group GROUP str "Group"]
  (let [dir "resources/DEBIAN/"
        app (or app (core/get-env :project))
        user (or app user)
        group (or group user)]
    (core/with-pre-wrap [fileset]
      (io/make-parents (str dir ".placeholder"))
      (doseq [s ['preinst 'postinst 'prerm 'postrm]]
        (spit (io/file (str dir (name s))) ((ns-resolve *ns* s) {:app app :user user :group group} )))
      fileset)))

(core/deftask dpkg-systemd [a app APPLICATION sym "The name of the application"
                            u user USER str "User"
                            g group USER str "Group"
                            t type TYPE str "The type of boot by systemd as simple, forking etc..."
                            c command COMMAND str "The command to boot"]
  (let [tmp (core/tmp-dir!)
        systemd-service-file (io/file tmp "usr/lib/systemd/system/" (str app ".service"))]

    (core/with-pre-wrap [fileset]
      (io/make-parents systemd-service-file)
      (spit systemd-service-file (gen-systemd-conf {:app app
                                                    :user user
                                                    :group group
                                                    :type type
                                                    :command (or command (str "/usr/bin/" app " -c /etc/" app "/config.edn"))}))
      
      (-> fileset
          (core/add-asset tmp)
          (core/commit!)))))

(core/deftask dpkg-sh-wrapper [a app APPLICATION str "The name of the application"
                               v ver VERSION str "Version"]
  (let [app (name (or app (core/get-env :project)))
        ver (name (or ver (core/get-env :version)))
        sh (str
            "#!/bin/bash"
            NL
            NL "java -jar " "/usr/share/java/" app "-" ver ".jar " "$*"
            NL)
        tmp (core/tmp-dir!)
        f (io/file tmp "usr/bin" app)]
        
        (core/with-pre-wrap [fileset]
          (io/make-parents f)
          (spit f sh)
      
          (-> fileset
              (core/add-asset tmp)
              (core/commit!)))))


(core/deftask dpkg-copy [a app APPLICATION str "The name of the application"
                         b bin BIN #{str} "The files to be placed in /usr/bin"
                         w wrapper WRAPPER BOOL]
  (let [app (or app (core/get-env :project))
        ver (core/get-env :version)
        tmp (core/tmp-dir!)]

    (core/with-pre-wrap [fileset]
      (doseq [[files path] [[["README.md" "CHANGELOG.md" "LICENSE"] (str "usr/share/doc/" app)]
                            [bin "usr/bin"]]]
        (doseq [fname files]
          (let [new-path (io/file tmp path (.getName (io/as-file fname)))]
            (io/make-parents new-path)
            (io/copy (io/file fname) new-path))))
      
      (-> fileset
          (core/add-asset tmp)
          (core/commit!)))))

(core/deftask dpkg-filter []
  (let [app (core/get-env :project)]
    (comp (built-in/sift :include #{#"usr/bin/.*"
                                    #"etc/.*"
                                    #"usr/lib/systemd/system/.*"
                                    #"usr/share/java/.*"
                                    #"usr/share/doc/.*"
                                    #".+\.jar$"
                                    #"^config\..+"
                                    #"DEBIAN/.*"}
                         :move {#"(.+\.jar$)" "usr/share/java/$1"
                                #"config\.edn" (str "etc/" app "/config.edn")
                                #"config\.env" (str "etc/default/" app)}))))







