;; *   Silvur
;; *
;; *   Copyright (c) Tsutomu Miyashita. All rights reserved.
;; *
;; *   The use and distribution terms for this software are covered by the
;; *   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; *   which can be found in the file epl-v10.html at the root of this distribution.
;; *   By using this software in any fashion, you are agreeing to be bound by
;; * 	 the terms of this license.
;; *   You must not remove this notice, or any other, from this software.

(ns silvur.nio
  (:gen-class)
  (:require [clojure.java.io :as io])
  (:import [java.io File]
           [java.nio.charset Charset]
           [java.util.stream Collectors]
           [java.nio.file Path Paths Files LinkOption OpenOption]
           [java.nio.file.attribute FileAttribute]
           [java.net URI]))

(extend Path
  io/IOFactory
  {:make-reader (fn [path & {:keys [encoding]}]
                  (Files/newBufferedReader path (Charset/forName encoding)))
   :make-writer (fn [path opts]
                  (Files/newBufferedWriter path (into-array OpenOption opts)))
   :make-input-stream (fn [^Path path opts] (io/make-input-stream (.toFile path) opts))
   :make-output-stream (fn [^Path path opts] (io/make-output-stream (.toFile path) opts))})

(defn path [f]
  (cond
    (string? f) (.toPath (io/file f))
    (instance? File f) (.toPath ^File f)
    (instance? Path f) f))

(defn exists? [a-path]
  (Files/exists (path a-path) (into-array LinkOption [])))

(defn absolute-path [a-path]
  (.toAbsolutePath ^Path (path a-path)))

(defn readable? [a-path]
  (Files/isReadable (path a-path)))

(defn mkdir [a-path]
  (Files/createDirectories (path a-path) (into-array FileAttribute [])))

(defn dir? [a-path]
  (Files/isDirectory (path a-path) (into-array LinkOption [])))

(defn rm [a-path]
  (Files/deleteIfExists (path a-path)))

(defn path-seq [a-path]
  (-> (Files/list (path a-path))
      (.collect (Collectors/toList))))

(defn size [a-path]
  (Files/size (path a-path)))
