;; *   Silvur
;; *
;; *   Copyright (c) Tsutomu Miyashita. All rights reserved.
;; *
;; *   The use and distribution terms for this software are covered by the
;; *   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; *   which can be found in the file epl-v10.html at the root of this distribution.
;; *   By using this software in any fashion, you are agreeing to be bound by
;; * 	 the terms of this license.
;; *   You must not remove this notice, or any other, from this software.

(ns silvur.datetime
  (:gen-class)
  (:import [java.util Date GregorianCalendar]
           [java.time.format DateTimeFormatter]
           [java.time ZonedDateTime ZoneOffset ZoneId Instant Duration ZoneRegion]))

;; Minimum unit is 'second'
(def UTC (ZoneId/of "UTC"))
(def JST (ZoneId/of "Asia/Tokyo"))
(def NYC (ZoneId/of "America/New_York"))

(def FORMAT {:JP "YYYY/MM/dd HH:mm:ss"
             :US "MM/dd/YYYY HH:mm:ss"})

(defonce ^:dynamic *tz* JST)
(defonce ^:dynamic *precision* 1) 

(declare adjust)

(defn set-default-precision! [x]
  (alter-var-root #'*precision* (constantly (condp = x
                                              :second 1000
                                              :milli 1
                                              1))))
(defn set-default-tz! [tz]
  (alter-var-root #'*tz* (constantly tz)))

(defmulti -datetime (fn [arg & _]  (class arg)))

(defmethod -datetime Integer [arg & rest]
  (apply -datetime (long arg) rest))

(defmethod -datetime Long [arg & rest]
  (if (and (>= 9999 arg 1970) (not (instance? clojure.lang.Keyword (first rest))))
    (ZonedDateTime/of arg (nth rest 0 1) (nth rest 1 1) (nth rest 2 0) (nth rest 3 0) (nth rest 4 0) (nth rest 5 0) *tz*)
    (do
      (ZonedDateTime/ofInstant  (condp = (first rest)
                                  :second (Instant/ofEpochSecond arg)
                                  :milli (Instant/ofEpochMilli arg)
                                  ;; Handle milli sec part 
                                  (Instant/ofEpochMilli (* *precision* arg))) *tz*))))

(defmethod -datetime String [arg & rest]
  (.format ^ZonedDateTime (apply -datetime rest) (DateTimeFormatter/ofPattern arg)))

(defmethod -datetime ZonedDateTime [arg & rest] arg)

(defmethod -datetime clojure.lang.LazySeq [arg & _]
  (map -datetime arg))

(defmethod -datetime clojure.lang.PersistentVector [arg & _]
  (map -datetime arg))

(defmethod -datetime Date [arg & rest]
  (-datetime (.getTime ^Date arg) :milli))

(defmethod -datetime clojure.lang.PersistentArrayMap [arg & _]
  (-datetime (:datetime arg)))

(defmethod -datetime clojure.lang.PersistentHashMap [arg & _]
  (-datetime (:datetime arg)))

(defmethod -datetime clojure.lang.IFn [f & rest]
  (adjust (apply -datetime (or rest [nil])) (f)))


(defmethod -datetime :default [arg & rest]
  (ZonedDateTime/now))


(defn datetime
  ([]
   (-> (ZonedDateTime/now ^ZoneId *tz*)
       (.toInstant)
       (.toEpochMilli)
       (quot *precision*)
       (* *precision*)
       (quot *precision*)
       (-datetime)))
  ([arg & rest]
   (apply -datetime arg (flatten [rest]))))

(defn datetime*
  ([]
   (datetime* (ZonedDateTime/now ^ZoneId *tz*)))
  ([arg & rest]
   (-> ^ZonedDateTime
       (apply datetime arg rest)
       (.toInstant )
       (.toEpochMilli)
       (quot *precision*))))


(defn tz-> ^ZonedDateTime [^ZoneId zid ^ZonedDateTime dt] 
  (.withZoneSameInstant dt zid))

(defn date<- ^Date [^ZonedDateTime zdt]
  (Date/from (.toInstant zdt)))

(defn inst<- ^Instant [^ZonedDateTime zdt]
  (.toInstant zdt))


(defn vec<- [^ZonedDateTime zdt]
  [(.getYear zdt) (.getMonthValue zdt) (.getDayOfMonth zdt)
   (.getHour zdt) (.getMinute zdt) (.getSecond zdt)])

(defn calendar<- ^Instant [^ZonedDateTime zdt]
  (GregorianCalendar/from zdt))



(defmulti +time (fn [v _ & _] (class v)) )
(defmethod +time Long [zdt delta & deltas]
  (datetime* (apply +time (datetime zdt) delta deltas)))

(defmethod +time ZonedDateTime [zdt delta & deltas]
  (.plus ^ZonedDateTime zdt (Duration/ofSeconds (-> *precision*
                                                    (* (+ delta (reduce + deltas)))
                                                    (quot 1000)))))


(defn duration [i]
  (Duration/ofSeconds (quot (* *precision* i) 1000)))

(defprotocol TimeExchange
  (minutes-of-week [zdt])
  (first-date-of-week [zdt])
  (adjust [zdt duration])
  (-day [i])
  (-hour [i])
  (-minute [i])
  (-second [i]))

(extend-protocol TimeExchange
  Integer
  (-day [i]
    (quot (* 1000 60 60 24 i) *precision*))
  (-hour [i]
    (quot (* 1000 60 60 i) *precision* ))
  (-minute [i]
    (quot (*  1000 60 i) *precision*))
  (-second [i]
     (quot (* 1000 i) *precision*))
  (adjust [v duration]
    (let [offset (.getRawOffset (java.util.TimeZone/getTimeZone ^ZoneId *tz*))]
      (datetime* (-  (* duration  (quot (+ v offset) (quot duration *precision*))) offset))))
  
  (first-date-of-week [i]
    (first-date-of-week (datetime i)))
  (minutes-of-week [i]
    (minutes-of-week (datetime i)))
  
  Long
  (-day [i]
    (quot (* 1000 60 60 24 i) *precision*))
  (-hour [i]
    (quot (* 1000 60 60 i) *precision* ))
  (-minute [i]
    (quot (*  1000 60 i) *precision*))
  (-second [i]
     (quot (* 1000 i) *precision*))
  (adjust [v duration]
    (let [offset (.getRawOffset (java.util.TimeZone/getTimeZone ^ZoneId *tz*))]
      (datetime* (-  (* duration  (quot (+ v offset) (quot duration *precision*))) offset))))
  
  (first-date-of-week [i]
    (first-date-of-week (datetime i)))
  (minutes-of-week [i]
    (minutes-of-week (datetime i)))

  ZonedDateTime
  (-year [this]
    (.getYear this))
  (-month [this]
    (.getMonth this))
  (-hour [this]
    (.getHour this))
  (-minute [this]
    (.getMinute this))
  (-second [this]
    (.getSecond this))
  
  (adjust [zdt duration]
    (datetime (adjust (.toEpochMilli (inst<- zdt)) duration )))
  
  (first-date-of-week [zdt]
    (-> zdt
        (.minus ^Duration (duration (-minute (minutes-of-week zdt))) )
        (adjust (-minute 1))))
  
  (minutes-of-week [zdt]
    (+
     (* (.getMinute zdt))
     (* 60 (.getHour zdt))
     (* 60 24 (mod (-> zdt (.getDayOfWeek) (.getValue)) 7))))) ;; Convert Sunday = 0

(defn day
  ([] (-day 1)) 
  ([i] (-day i)))
(defn hour
  ([] (-hour 1)) 
  ([i] (-hour i)))
(defn minute
  ([] (-minute 1)) 
  ([i] (-minute i)))


(defn this-month [& args]
  (apply datetime (concat ((juxt #(.getYear ^ZonedDateTime %)
                                 #(.getMonthValue ^ZonedDateTime %))
                           (datetime))
                          args)))

(defn this-year [& args]
  (apply datetime (concat ((juxt #(.getYear ^ZonedDateTime %)) (datetime)) args)))

(defn today [& args]
  (apply datetime (concat ((juxt #(.getYear ^ZonedDateTime %)
                                 #(.getMonthValue ^ZonedDateTime %)
                                 #(.getDayOfMonth ^ZonedDateTime %))
                           (datetime)) args)))
(defn yesterday [& args]
  (apply datetime (concat ((juxt #(.getYear ^ZonedDateTime %)
                                 #(.getMonthValue ^ZonedDateTime %)
                                 #(.getDayOfMonth ^ZonedDateTime %))
                           (.plusDays ^ZonedDateTime (datetime) -1))
                          args)))

(defmacro this-month* [& args]
  `(datetime* (this-month ~@args)))
(defmacro this-year* [& args]
  `(datetime* (this-year ~@args)))
(defmacro today* [& args]
  `(datetime* (today ~@args)))
(defmacro yesterday* [& args]
  `(datetime* (yesterday ~@args)))

(defn time-seq [latest delta]
  (lazy-seq
   (cons (adjust latest delta)
         (time-seq (+time (adjust latest delta) (- delta)) delta))))


(defn time-period-seq
  ([]
   (time-period-seq (datetime) (minute 1)))
  ([latest]
   (time-period-seq latest (minute 1)))
  ([latest delta]
   (lazy-seq
    (let [end (adjust latest (minute 1))
          start (adjust latest delta)]
      (cons  [start end] (time-period-seq (+time start (- (minute 1))) delta))))))

(defn- -offset [zid  zdt]
  (-> (tz-> zid zdt)
      (.getOffset)
      (.getTotalSeconds)
      (quot (hour 1))))

(defmulti summer-time (fn [z h] (class z)))
(defmethod summer-time java.lang.String [z h]
  (summer-time (ZoneId/of z) h))


(defmethod summer-time :default [zid h]
  (binding [*precision* 1000]
    (+ h (-offset zid (datetime 2014 1 1)) (- (-offset zid (datetime))))))


