(ns moncee.context
  (:require [moncee.util :refer [bson<- bson-> ]])
  (:import [com.mongodb.client MongoClients MongoClient MongoIterable MongoCollection]
           [com.mongodb.client.model Filters Sorts UpdateOptions Indexes Aggregates Accumulators Projections]
           [com.mongodb.client.result UpdateResult]
           [com.mongodb MongoClientSettings]))

(gen-class
 :name moncee.Context
 :state state
 :init init
 :implements [clojure.lang.ISeq
              clojure.lang.Associative
              clojure.lang.Sequential
              clojure.lang.Counted
              java.util.List]
 :constructors {[com.mongodb.client.MongoCollection] []}
 :methods [[insert [java.util.List] void]
           [delete [] java.lang.Long]
           [update [java.lang.Boolean java.lang.Boolean java.util.Map] java.lang.Long]])

(defn- -limit [this]
  (let [context @(.state this)]
    (if-let [x (last (sort (map :v (filter #(= (:op %) :limit) (:operations context)))))]
      x
      0)))

(defn- -skip [this]
  (let [context @(.state this)]
    (apply max (cons 0  (map :v (filter #(= (:op %) :skip) (:operations context)))))))

(defn- -filter [this key]
  (let [context @(.state this)]
    (->> (:operations context)
         (filter #(= (:op %) key))
         (map :v)
         (reduce conj {}))))


(defn -init [collection]
  [[] (atom {:source collection
             :operations []})])


;; ISeq
(defn -seq [this]
  (let [{:keys [source operations]} @(.state this)]
    (if (some #(= :group (:op %)) operations)
      (let [docs (transduce (map (fn [{:keys [op v]}]
                                   (bson<- (hash-map ({:group :$group
                                                       :filter :$match
                                                       :project :$project
                                                       :sort :$sort
                                                       :limit :$limit
                                                       :skip :$skip} op)
                                                     v))))
                            conj
                            []
                            (filter #(not (= (:op %) :transduce)) operations))]

        (-> source
            (.aggregate docs)
            (->> (sequence (apply comp (concat [(map bson->)] (vals (-filter this :transduce))))))
            seq))

      
      (-> source
          (.find (bson<- (-filter this :filter)))
          (.projection (bson<- (-filter this :project)))
          (.sort (bson<- (-filter this :sort)))
          (.limit (-limit this))
          (.skip (-skip this))
          (->> (sequence (apply comp (concat [(map bson->)]
                                             (vals (-filter this :transduce))))))
          (seq)))))

(defn -cons [this obj] (throw (ex-info "Not yet implemented" {})))
(defn -more [this] (throw (ex-info "Not yet implemented" {})))
(defn -next [this] (next (-seq this)))
(defn -first [this]
  (let [{:keys [source operations]} @(.state this)]
    (if (some #(= :group (:op %)) operations)
      (let [docs (transduce (map (fn [{:keys [op v]}]
                                   (bson<- (hash-map ({:group :$group
                                                       :filter :$match
                                                       :project :$project
                                                       :sort :$sort
                                                       :limit :$limit
                                                       :skip :$skip} op)
                                                     v))))
                            conj
                            []
                            (filter #(not (= (:op %) :transduce)) operations))]

        (-> source
            (.aggregate docs)
            (.first)
            (bson->)))

      
      (-> source
          (.find (bson<- (-filter this :filter)))
          (.projection (bson<- (-filter this :project)))
          (.sort (bson<- (-filter this :sort)))
          (.limit (-limit this))
          (.skip (-skip this))
          (.first)
          (bson->)))))

(defn -equiv [this obj] (throw (ex-info "Not yet implemented" {})))
(defn -empty [this] (throw (ex-info "Not yet implemented" {})))
(defn -count [this]
  (let [{:keys [source]} @(.state this)]
    (if (< 0 (-limit this) Long/MAX_VALUE)
      (count (-> this seq))
      (.countDocuments source (bson<- (-filter this :filter))))))

;; Associative
(defn -containsKey [this key]
  (contains? (keys @(.state this)) key))

(defn -entryAt [this key] (@(.state this) key))

(defn -valAt [this key] (@(.state this) key))

(defn -toArray [this]
  (to-array (-seq this)))

(defn -assoc [this key val]
  (when (contains? @(.state this)  key)
    (swap! (.state this) assoc key val))
  this)


;; Operations
(defn -insert [this docs]
  (let [ctx @(.state this)]
    (.insertMany (:source ctx) (map bson<- docs))))

(defn -delete [this]
  (let [ctx @(.state this)]
    (-> (:source ctx) (.deleteMany  (bson<- (-filter this :filter))) (.getDeletedCount))))

(defn -update [this upsert? only-one? m]
  (let [ctx @(.state this)
        result (if only-one?
                 (.updateOne (:source ctx)
                             (bson<- (-filter this :filter))
                             (bson<- m)
                             (doto (UpdateOptions.) (.upsert upsert?)))
                 (.updateMany (:source ctx)
                              (bson<- (-filter this :filter))
                              (bson<- m)
                              (doto (UpdateOptions.) (.upsert upsert?))))]
    (cond
      (and only-one? upsert?) (-> result (.getUpsertedId) (.getValue))
      (and only-one? (not upsert?)) (-> result (.getModifiedCount))
      (and (not only-one?) upsert?) (let [modified (-> result (.getModifiedCount))]
                                      (if (= 0 modified) 1 modified))
      (and (not only-one?) (not upsert?)) (-> result (.getModifiedCount)) )))
