(ns moncee.core
  (:require [moncee.util :refer [bson-> bson<-]])
  (:require [taoensso.timbre :as log])
  (:import [com.mongodb.client MongoClients MongoClient MongoIterable MongoCollection]
           [com.mongodb.client.model Filters Sorts UpdateOptions Indexes]
           [com.mongodb MongoClientSettings])
  (:import [org.bson.codecs.configuration CodecRegistries CodecRegistry])
  (:import [clojure.lang Named])
  (:import [moncee Context])
  (:require [silvur.datetime :refer [datetime* datetime]]))



(defonce ^:dynamic *mongo* nil)
(defonce ^:dynamic *db* nil)

(def mongo-ops {< "$lt" > "$gt" <= "$lte" >= "$gte"})

(defn bound-client []
  *mongo*)

(defprotocol MongoFundamental
  (-mongo [x])
  (-database [client x]))

(extend-protocol MongoFundamental
  clojure.lang.PersistentArrayMap
  (-mongo [{:keys [host port user password name] :or {host "localhost" port 27017}}]
    (let [uri (str "mongodb://" (cond-> ""
                                  (and user password) (str user ":" password "@")
                                  host (str host)
                                  port (str ":" port)
                                  name (str "/" name)))]
      (log/debug "Mongo URI:" uri)
      (MongoClients/create uri)))
  clojure.lang.Keyword
  (-mongo [host]
    (-mongo {:host (name host) :port 27017}))
  String
  (-mongo [host]
    (-mongo {:host (name host) :port 27017}))
  (-database [host db-name]
    (-database (-mongo host) db-name))
  com.mongodb.client.MongoClient
  (-database [client db-name]
    (-> client (.getDatabase (name db-name)))))

(defn mongo
  ([]
   (-mongo "localhost"))
  ([& opts]
   (apply -mongo opts)))

(defn database
  ([]
   (-database "localhost" "test"))
  ([{:keys [name] :as m}]
   (-database (-mongo m) name))
  ([arg0 arg1]
   (-database arg0 arg1)))

(defn set-default-client! [client]
  (alter-var-root #'*mongo* (constantly client)))

(defmacro with-db-binding [db & body]
  `(binding [*db* ~db]
     ~@body))


(defn set-default-db!
  "Set default database."
  ([]
   (set-default-db! (mongo :localhost) :test))
  ([db-name]
   (if *mongo*
     (set-default-db! *mongo* db-name)
     (do
       (set-default-client! (mongo))
       (set-default-db! *mongo* db-name))))
  ([client db-name]
   (alter-var-root #'*db* (constantly (database client db-name))))
  ([user password db-name & [host port]]
   (set-default-client! (mongo {:user user :password password :name db-name :host host :port port}))
   (set-default-db! db-name)))

(defmacro switch!
  "Syntax sugar of set-default-db"
  [db-name]
  `(if *mongo* (set-default-db! *mongo* ~db-name) (ex-info "Client not configured" {})))

(defn collection [x]
  (-> *db* (.getCollection (name x))))


(defprotocol MongoRequestContextIdentify
  (requester [this] ))

(extend-protocol MongoRequestContextIdentify
  Named
  (requester [this] (Context. (collection this)))

  Context
  (requester [this] this))



(log/merge-config! {:ns-blacklist ["org.mongodb.driver.*"]})
