(ns theladders.ephphp.unserialize
  (:require [clojure.string :as string]))

(defn str->keyword [obj]
  (if (string? obj) 
    (keyword obj) 
    obj))

(defmulti unserialize-blob (fn [blob key-fn] (str (first blob))))

(defn get-string-length [blob]
  (Integer/parseInt (get (re-find (re-pattern "s:(\\d+):.*") blob) 1)))

(defn get-string-body [blob length]
   (re-find (re-pattern (format "s:%d:\"(.{%d})\";(.*)" length length)) blob))

(defn get-string [blob length]
  (->>
   (get-string-body blob length)
   (apply #(vector %2 %3))))

(defmethod unserialize-blob "s" [blob key-fn]
  (get-string blob (get-string-length blob)))

(defn get-integer [blob]
  (->>
   (re-find (re-pattern "i:(\\d+);(.*)") blob)
   (apply #(vector (Integer/parseInt %2) %3))))

(defmethod unserialize-blob "i" [blob key-fn]
  (get-integer blob))

(defn sequential-array? [array]
  (let [k (keys array)]
    (and
     (every? integer? k)
     (= (sort k) (range (count array))))))

(defn maybe-sequential-array->vector
  "Turns the array into a vector if the keys are sequential integers from 0 to
  one less than the length of the array."
  [array]
  (if (sequential-array? array)
    (into [] (map #(get array %1) (range (count array))))
    array))

(defn get-array-length [blob]
  (Integer/parseInt (get (re-find (re-pattern "a:(\\d+):.*") blob) 1)))

(defn unserialize-key-value-pair [blob key-fn]
  (let [[key value-blob] (unserialize-blob blob key-fn)
        [value remainder] (unserialize-blob value-blob key-fn)]
    [[(key-fn key) value] remainder]))

(defn unserialize-key-value-pairs 
  ([blob length key-fn]
     (unserialize-key-value-pairs [] blob length key-fn))
  ([pairs blob length key-fn]
     (case length 
       0 [pairs blob]
       (let [[pair remainder] (unserialize-key-value-pair blob key-fn)]
         (recur (conj pairs pair) remainder (dec length) key-fn)))))

(defn strip-array-end [blob]
  (get (re-find (re-pattern "\\}(.*)") blob) 1))

(defn get-key-value-pairs [blob length]
   (get (re-find (re-pattern (format "a:%d:\\{(.*)" length)) blob) 1))

(defn get-array [blob length key-fn]
  (->
   (get-key-value-pairs blob length)
   (unserialize-key-value-pairs length key-fn)
   (#(vector (maybe-sequential-array->vector (into {} (first %1)))
             (strip-array-end (second %1))))))

(defmethod unserialize-blob "a" [blob key-fn]
  (get-array blob (get-array-length blob) key-fn))

(defn get-null [blob]
  [nil (get (re-find (re-pattern "N;(.*)") blob) 1)])

(defmethod unserialize-blob "N" [blob key-fn]
  (get-null blob))

(defn parse-int->bool [int]
  (case int
    "0" false
    true))

(defn get-bool [blob]
  (->>
   (re-find (re-pattern "b:(\\d);(.*)") blob)
   (apply #(vector (parse-int->bool %2) %3))))

(defmethod unserialize-blob "b" [blob key-fn]
  (get-bool blob))

(defn- unserialize-
  ([obj blob key-fn]
     (case blob 
       "" obj
       (let [[new-obj remainder] (unserialize-blob blob key-fn)]
         (unserialize- (conj obj new-obj) remainder key-fn)))))

(defn unserialize 
  [blob & {:as options}]
     (let [key-fn (or (:key-fn options) identity)]
     (unserialize- [] blob key-fn)))
