(ns nim.member
  (:require-macros
    [tailrecursion.javelin.macros :refer [cell]])
  (:require
    [hlisp.reactive               :as r]
    [hlisp.wigwam                 :as w]
    [tailrecursion.javelin        :as j]))

(def menus          (cell nil)) 
(def brand          (atom -1)) 
(def loading        w/loading) 
(def page           (cell '{})) 
(def state          (cell nil)) 
(def error          (cell nil)) 

(def current-cycle  (cell 0))
(def current-day    (cell 0))

(let [padvec          (fn padvec ([c n] (padvec c n nil)) 
                          ([c n p] (vec (take n (concat c (repeat p))))))
      safe-name       #(try (name %) (catch js/Error e))

      m-info          js/NiM.API.Member.clientInfo
      m-login         js/NiM.API.Member.login
      m-logout        js/NiM.API.Member.logout
      m-menu          js/NiM.API.Member.clientMenu
      m-choose        js/NiM.API.Member.chooseMeal

      all-mealcats    [:Breakfast :Lunch :Dinner]

      r-day           (fn [days i] [(cell (if (= i current-day) :active))])
      r-cycle         (fn [cycles i]
                        [(cell (let [c (nth cycles i)]
                                 (cond (and c (= i current-cycle)) :active
                                       c :pending)))])
      r-avail         (fn [avail i] [(cell (nth avail i))])

      meal-map        (cell (:meal-info menus))
      menu-map        (cell (dissoc menus :meal-info))

      get-meals       #(if %1 (get (:meals %1) (keyword (:sqldate %2)))) 
      get-info        #(if %1 (nth (second (get (vec %1) %2)) %3)) 
      get-selected    #(cell (->> (nth %1 %2) (get %3) keyword (get %4)))
      get-menu        #(->> (nth %1 %2) (get %3) (mapv keyword))
      get-choosers    #(let [meal-info (get %2 %1)]
                         (assoc meal-info :selected (= meal-info %3)))

      cur-day-info    (cell (get-info menu-map current-cycle current-day))
      cur-day-meals   (cell (get-meals state cur-day-info))
      cur-day-menu    (cell (:meals cur-day-info))
      cur-sqldate     (cell (:sqldate   cur-day-info))
      cur-cycle-id    (cell (safe-name (nth (keys menu-map) current-cycle)))
      mk-cur-choosers #(cell
                         (let [menu     (get-menu %1 %2 cur-day-menu)
                               meal-map (repeat meal-map)
                               selected (repeat %3)]
                           (padvec (mapv get-choosers menu meal-map selected) 30))) 

      r-mealcat       (fn [mealcats i]
                        (let [selected-info
                              (get-selected mealcats i cur-day-meals meal-map)]
                          [(cell (name (nth mealcats i)))
                           selected-info
                           (r/thing-looper
                             (mk-cur-choosers mealcats i selected-info)
                             r-avail)]))
      
      async           (fn [c f & args] (w/async (apply f args) c error))]

  (def cycles         (cell (padvec (keys menu-map) 7)))
  (def days           (cell (padvec '[] 7)))

  (def meals?         (cell (seq (get menus :meal-info))))
  (def current-date   (cell (:date      cur-day-info)))

  (def loop-cycles    (r/thing-looper cycles r-cycle)) 
  (def loop-days      (r/thing-looper days   r-day)) 
  (def loop-mealcats  (r/thing-looper (cell all-mealcats) r-mealcat))

  (defn login! [email pass]
    (async state m-login email pass @brand)) 

  (defn logout! [& _]
    (async state m-logout)) 

  (defn select-cycle! [i]
    (when (and (not= @current-cycle i) (nth @cycles i)) 
      (reset! current-cycle i)
      (reset! current-day 0))) 

  (defn select-day! [i]
    (reset! current-day i)) 

  (defn choose-meal! [meal-info]
    (async state m-choose
           (get-in @state [:client :clientID])
           @cur-cycle-id
           @cur-sqldate
           (:meal_category_id meal-info)
           (:id meal-info)))

  (defn init [b]
    (reset! brand b)
    (reset! w/keywordize true) 
    (w/async (m-info) state) 
    (cell (let [cid (get-in state [:client :clientID])]
            (#(if (< 0 %) (async menus m-menu %)) cid))) 
    (cell (#(do (reset! login-email "") (reset! login-pass "")) state))))
