(ns nim.member
  (:require-macros
    [tailrecursion.javelin.macros   :refer [cell]])
  (:require
    [tailrecursion.javelin          :as j]
    [tailrecursion.hoplon.util      :as u]
    [tailrecursion.hoplon.reactive  :as r]
    [hlisp.wigwam                   :as w]
    [clojure.string                 :as string]))

(def menus          (cell nil)) 
(def brand          (atom -1)) 
(def loading        w/loading) 
(def page           (cell '{})) 
(def state          (cell nil)) 
(def error          (cell nil)) 
(def notice         (cell nil)) 
(def warning        (cell nil)) 

(def current-cycle  (cell 0))
(def current-day    (cell 0))

(let [padvec            (fn padvec
                          ([c n] (padvec c n nil)) 
                          ([c n p] (vec (take n (concat c (repeat p))))))
      safe-name         #(try (name %) (catch js/Error e))
      async             #(fn [& args] (w/async (apply %2 args) %1 error))
      kw-route          #(->> (string/split % "/")
                           (remove string/blank?)
                           rest
                           (mapv keyword))

      m-login           (async state js/NiM.API.Member.login)
      m-logout          (async state js/NiM.API.Member.logout)
      m-menu            (async menus js/NiM.API.Member.clientMenu)
      m-choose          (async state js/NiM.API.Member.chooseMeal)
      m-update-email    (async state js/NiM.API.Member.updateEmail)
      m-update-passwd   (async state js/NiM.API.Member.updatePassword)
      m-update-promote  (async state js/NiM.API.Member.updatePromote)
      m-update-billing  (async state js/NiM.API.Member.updateBilling)
      m-contact-us      (async state js/NiM.API.Member.contactUs)

      all-mealcats      [:Breakfast :Lunch :Dinner :Snack (keyword "Dessert Snack")
                         (keyword "Starch Side") (keyword "Vegetable Side")]

      r-day             (fn [days i] [(cell (if (= i current-day) :active))])
      r-cycle           (fn [cycles i]
                          [(cell (let [c (nth cycles i)]
                                   (cond (and c (= i current-cycle)) :active
                                         c :pending)))])
      r-avail           (fn [avail i] [(cell (nth avail i))])

      cid               (cell (get-in state [:client :clientID]))
      meal-map          (cell (:meal-info menus))
      payments          (cell (vec (take 10 (padvec (vec (:payments menus)) 10))))
      menu-map          (cell (dissoc menus :meal-info :payments))

      get-meals         #(if %1 (get (:meals %1) (keyword (:sqldate %2)))) 
      get-info          #(if %1 (nth (second (get (vec %1) %2)) %3)) 
      get-selected      #(cell (->> (nth %1 %2) (get %3) keyword (get %4)))
      get-menu          #(->> (nth %1 %2) (get %3) (mapv keyword))
      get-choosers      #(let [meal-info (get %2 %1)]
                           (assoc meal-info :selected (= meal-info %3)))

      cur-day-info      (cell (get-info menu-map current-cycle current-day))
      cur-day-meals     (cell (get-meals state cur-day-info))
      cur-day-menu      (cell (:meals cur-day-info))
      cur-sqldate       (cell (:sqldate cur-day-info))
      cur-cycle-id      (cell (safe-name (nth (keys menu-map) current-cycle)))
      mk-cur-choosers   #(cell
                           (let [menu     (get-menu %1 %2 cur-day-menu)
                                 meal-map (repeat meal-map)
                                 selected (repeat %3)]
                             (padvec (mapv get-choosers menu meal-map selected) 30))) 

      r-mealcat         (fn [mealcats i]
                          (let [selected-info
                                (get-selected mealcats i cur-day-meals meal-map)]
                            [(cell (name (nth mealcats i)))
                             selected-info
                             (r/thing-looper
                               (mk-cur-choosers mealcats i selected-info)
                               r-avail)]))
      r-bill-history    (fn [history-items i]
                          (let [item (cell (get history-items i))]
                            [(cell (get item :date))
                             (cell (get item :label))
                             (cell (get item :amt))
                             (cell (get item :descr))]))]

  (def route            (cell (kw-route '(j/route* 50 "#/"))))

  (def display-name     (cell (u/capitalize-name
                                (str (get-in state [:client :firstName]) " "
                                     (get-in state [:client :lastName])))))

  (def cycles           (cell (padvec (sort (keys menu-map)) 7)))
  (def days             (cell (padvec '[] 7)))

  (def meals?           (cell (seq (get menus :meal-info))))
  (def current-date     (cell (:date cur-day-info)))
  (def cutoff-date      (cell (:cutoff cur-day-info))) 
  (def cur-nutrition    (cell (get-in state [:nutrition (keyword cur-sqldate)])))
  (def can-choose?      (cell (<= (get-in state [:client :cutoff]) cur-sqldate))) 

  (def loop-cycles      (r/thing-looper cycles r-cycle)) 
  (def loop-days        (r/thing-looper days   r-day)) 
  (def loop-mealcats    (r/thing-looper (cell all-mealcats) r-mealcat))
  (def loop-billing-history (r/thing-looper payments r-bill-history))

  (defn login! [email pass]
    (m-login email pass @brand)) 

  (defn logout! [& _]
    (m-logout)) 

  (defn select-cycle! [i]
    (when (and (not= @current-cycle i) (nth @cycles i)) 
      (reset! current-cycle i)
      (reset! current-day 0))) 

  (defn select-day! [i]
    (reset! current-day i)) 

  (defn choose-meal! [meal-info]
    (let [cat (:meal_category_id meal-info)
          id  (:id meal-info)]
      (m-choose @cid @cur-cycle-id @cur-sqldate cat id)))

  (defn update-email! [passwd email email2]
    (m-update-email @cid passwd email email2))

  (defn update-passwd! [passwd new-passwd new-passwd2]
    (m-update-passwd @cid passwd new-passwd new-passwd2))

  (defn update-promote! [promote]
    (m-update-promote @cid promote))

  (defn update-billing! [addr1 addr2 city state zip ccname ccnum exp-m exp-y cvv]
    (m-update-billing @cid addr1 addr2 city state zip ccname ccnum exp-m exp-y cvv))

  (defn contact-us! [message]
    (m-contact-us @cid message))

  (defn init [b]
    (reset! brand b)
    (reset! w/keywordize true)
    (w/async (js/NiM.API.Member.clientInfo) state) 
    (cell (#(if (< 0 %) (m-menu %)) cid))))
