(ns ds
  "Simple namespace with an abbreviated name for quick use during debugging. The
  namespace is split among four separate files, similar to the clojure.core
  source. The source files rely on the requires in this primary file, but those
  files can't use the dependency aliases using `:as`, so most of the aliases are
  omitted below."
  (:require
   [clojure.core.cache]
   [clojure.inspector]
   [clojure.pprint :as pprint]
   [clojure.string :as str]
   [clojure.walk :as w]
   [hashp.core :as hashp]
   [incanter.charts]
   [incanter.core]
   [puget.printer]
   [rhizome.viz]))

(defn current-trace []
  (let [{:keys [ns fn line]}
        (->> (hashp/current-stacktrace)
             (filter :clojure)
             (remove (comp (partial = "ds") :ns))
             (remove (every-pred
                      (comp (partial = "clojure.core") :ns)
                      (comp not-empty :fn)
                      (comp #(str/starts-with? % "eval") :fn)))
             first)]
    (if (and ns fn line)
      (format "[%s/%s:%s]" ns fn line)
      "[location unknown]")))

(defn filter-form
  "If `form` already contains macroexpanded pretty-print output, filter out the
  macro expanded content to just return the original form for prettier printing.

  Example of code where a data-scope reader macro is nested in another and might
  require filtering:
     #pp (->> [1 2 3]
              #pp->> (map inc)
              (apply +))

  The above example has a #pp->> macro that results in macroexpanded output to
  #pp, so we try to filter that noise out from the #pp output."
  [form]
  (cond
    ;; if your form isn't a collection, we don't need to worry about
    ;; filtering out noise
    (not (coll? form))
    form

    ;; threaded printed items will have a form that looks like:
    ;;   `((ds/pprint-wrapper (quote original-form)))`
    (and (= 1 (count form))
         (coll? (first form))
         (= 'ds/pprint-wrapper (ffirst form)))
    (second (second (first form)))

    ;; regular pretty-printed items look like:
    ;;   `(ds/pprint-wrapper (quote original-form))`
    (= 'ds/pprint-wrapper (first form))
    (second (second form))

    ;; if the form is a collection but isn't wrapped with pprint, then the
    ;; form doesn't have any nested pretty-print macros
    :else
    form))

(defn location-line-and-form
  "Prints the namespace, fn, and line where the literal was placed, and also
  prints the original form that is being inspected if it's different than the
  result of evaluating that form."
  [form evaluated]
  (if (= evaluated form)
    `~(println (current-trace) "=>")
    (let [orig-form# (->> form
                          (w/prewalk filter-form)
                          pprint/pprint
                          with-out-str)]
      `~(println (current-trace)
                 (str/trim-newline (or orig-form# "")) ;; nil-safe
                 "=>"))))

(load "data_scope/charts")
(load "data_scope/graphs")
(load "data_scope/inspect")
(load "data_scope/pprint")
