(ns template.handler
  (:require [compojure.core :refer :all]
            [compojure.route :as route]
            [ring.middleware.defaults :refer [wrap-defaults site-defaults]]
            [ring.middleware.reload :refer [wrap-reload]]
            [ring.middleware.params :refer [wrap-params]]
            [org.httpkit.server :as http]
            [template.email :as email]
            [template.view :as view]
            [hiccup.core :refer :all]
            [clojure.java.io :as io]))

(defn process-email-notification [record]
  (let [kind (:kind record)
        tenant (:tenant record)
        context (get-in record [:body :context])
        locale (or (:locale record) "en_US")
        payload (email/generate-payload kind tenant locale context)]
    ;; (println "tenant-key:" tenant)
    ;; (println "payload:" payload)
    payload))

;; return html
(defn generate-template
  [record]
  (let [{:keys [kind type tenant ext locale]} record
        template-file (email/template-file kind type ext tenant locale)]
    (email/render-resource template-file record)))

;; put template under one of the following folder
;; (format "templates/notification/%s/%s/%s.%s.%s" tenant locale kind type ext)
;; (format "templates/notification/%s/%s.%s.%s" tenant kind type ext)
;; (format "templates/notification/%s.%s.%s" kind type ext)
;; kind is template-name
;; extension is either .html or .txt
(defroutes app-routes
  (GET "/" [] "Hello Worlds")
  (GET "/:template-name/:type/:ext" [type ext template-name :as request]
       (println "req: " request)
       (println "params" (:params request))
    (let [params (:params request)]
      (generate-template (merge {:kind template-name
                                 :type type
                                 :ext ext
                                 :tenant nil
                                 :locale nil}
                                params))))
  (GET "/:tenant-or-locale/:template-name/:type/:ext" [type ext template-name tenant-or-locale :as request]
       (let [params (:params request)]
         (or (generate-template (merge {:kind template-name
                                        :type type
                                        :ext ext
                                        :tenant nil
                                        :locale tenant-or-locale}
                                       params))
             (generate-template (merge {:kind template-name
                                        :type type
                                        :ext ext
                                        :tenant tenant-or-locale
                                        :locale nil}
                                       params)))))
  (GET "/:tenant-name/:locale/:template-name/:type/:ext" [type ext template-name tenant-name locale :as request]
    (let [params (:params request)]
      (generate-template (merge {:kind template-name
                                 :type type
                                 :ext ext
                                 :tenant tenant-name
                                 :locale locale}
                                params))))
  (GET "/template-list" []
    (let [directory (io/resource "templates")
          files (file-seq (io/as-file directory))]
      (html [:ul
             (map #(vector :li %) files)])))
  (GET "/views/*" [:as req]
       (let [path (subs (:uri req) 1)
             template-path (format "%s.html" path)
             context-path (format "%s.edn" path)]
         (view/generate-view template-path context-path)))
  (route/not-found "Not Found"))

(def app
  (-> #'app-routes
      (wrap-params)
      (wrap-reload)))

(defn -main  [& args]
  (let [port (try
               (Integer/parseInt (first args))
               (catch Exception _
                 9090))]
    (http/run-server app {:port port})
    (printf "Server started on port %s....%n" port)))
