(ns pine.core
  (:require [clojure.string :as string]))

(declare match-subpath)

(defn match-route
  ;; Routes acts as a stack for DFS search
  [path routes]
  (let [route (peek routes)
        remaining (pop routes)]
    (if-let [{remaining-path :next-subpath} (match-subpath path (:path route))]
      (if (= 0 (count remaining-path))
        {:route (:route route)}
        (if (:children route)
          ;; TODO consider changing underlying route structure so all of this
          ;; iterable-type shuffling can be removed
          (recur remaining-path (apply list (concat (:children route) remaining)))
          (recur path remaining)))
      (recur path remaining))))


(defmulti match-subpath
  "Check to see if a subpath matches a test.

   If the subpath matches the test, break down the test
   params and return an object with a next-subpath and
   the params."
  ;; TODO Create record for match
  (fn [test subpath] (type test)))

(defmethod match-subpath js/String
  [test subpath]
  (when (string/starts-with? test subpath)
    {:next-subpath (string/replace-first test subpath "")}))

(defmethod match-subpath cljs.core/PersistentVector
  [test subpath]
  (str "vec: " test))



;; UI Components
(defn view
  "Insert a component into the component tree.

  `route` is the route name for this component.
  `component` is either a keyword identifier for the component,
  or the component itself.
  `props` are the props to be passed to the component.

  e.g.
  (view :home :home)
  (view :home home)
  (view :home :home {:title \"Welcome\"})
  "
  [route component props])


(defn multi-view
  "Inserts one of many components into the component tree.

  Accepts a hash-map keyed by route, with values of either components
  or hash-maps defining components and properties.

  e.g.
  (multi-view :home :home
              :login login
              :welcome [:welcome]
              :about [about {:title \"hi\"}]
              :portfolio {:component :portfolio
                          :transformer identity})"
  [components])
