(ns tech.vault-clj.core
  (:require [clojure.java.io :as io]
            [tech.config.core :refer [get-config]]
            [cheshire.core :refer :all]
            [org.httpkit.client :as http]))

(defonce loaded-token (atom nil))

(defn force-load-vault-token
  "Attempts to load the vault token from the config.  Attempts users
  .vault-token file if the config has no token"
  []
  (let [config-token (get-config :vault-token)
        home-dir (System/getProperty "user.home")
        token (if (< (.length config-token) 5)
                (->> (io/file home-dir ".vault-token")
                     (slurp))
                config-token)]
    (reset! loaded-token token)))

(defn get-vault-token
  []
  (if @loaded-token
    @loaded-token
    (force-load-vault-token)))

(defn- vault-raw-request
  [verb token addr args]
  (let [url (str (get-config :vault-addr) "/v1/" addr)
        verb-fn (condp = verb
                  :get http/get
                  :post http/post)]
    (->> (update-in args [:headers] assoc "X-Vault-Token" token)
         (verb-fn url)
         (deref))))

(defn vault-request
  [verb token addr args]
  (-> (vault-raw-request verb token addr args)
      :body
      (parse-string)))

(defn vault-get
  [addr & {:keys [token]
           :or {token (get-vault-token)}} ]
  (vault-request :get token addr {}))

(defn vault-post
  [addr body-map & {:keys [token]
                    :or {token (get-vault-token)}}]
  ;;Arguments to vault are sent via json in the body of the post
  (vault-request :post token addr {:body (generate-string body-map)}))

(defn read-credentials
  [addr]
  (vault-get addr))

(defn create-token
  [lease-seconds]
  (-> (vault-post "auth/token/create"
                  {"lease" (str lease-seconds "s")})
      (get-in ["auth" "client_token"])))
