(ns tech.smile.utils
  (:require [tech.datatype :as dtype])
  (:import [java.lang.reflect Constructor]))


(def keyword->class-types
  {:float64 Double/TYPE
   :int32 Integer/TYPE
   :boolean Boolean/TYPE
   :int32-array (Class/forName "[I")
   :float64-array (Class/forName "[D")
   :float64-array-array (Class/forName "[[D")
   :object-array (Class/forName "[Ljava.lang.Object;")})


(defmulti option->class-type
  (fn [option]
    (:type option)))

(defmethod option->class-type :default
  [option]
  (if-let [retval (keyword->class-types (:type option))]
    retval
    (throw (ex-info "Failed to find primitive class type"
                    {:cls-type (:type option)
                     :available (keys keyword->class-types)}))))


(defmethod option->class-type :enumeration
  [option]
  (:class-type option))


(defn class-name->class
  ^Class [pkgname ^String clsname]
  (Class/forName
   (if-not (.contains clsname ".")
     (str pkgname "." clsname)
     clsname)))


(defmulti option-value->value
  (fn [option-desc option-val]
    (:type option-desc)))


(defmethod option-value->value :default
  [option-desc option-val]
  (dtype/cast option-val (:type option-desc)))


(defmethod option-value->value :enumeration
  [option-desc option-val]
  (if-let [retval (get-in option-desc [:lookup-table option-val])]
    retval
    (throw (ex-info "Failed to find option"
                    {:option-descriptor option-desc
                     :option-value option-val}))))


(defmethod option-value->value :boolean
  [option-desc option-val]
  (boolean option-val))


(defmethod option-value->value :int32-array [_ val] val)
(defmethod option-value->value :float64-array [_ val] val)
(defmethod option-value->value :float64-array-array [_ val] val)
(defmethod option-value->value :object-array [_ val] val)


(defmethod option-value->value :int32-array
  [option-desc option-val]
  (let [option-val (if (number? option-val)
                     [option-val]
                     option-val)]
    (dtype/make-array-of-type :int32 option-val)))


(defn set-setter-options!
  [setter-options item options]
  (let [model-cls (.getClass item)]
    (->> setter-options
         (map (fn [{:keys [setter default name] :as setter-option}]
                (let [option-val (->> (get options name default)
                                      (option-value->value setter-option))
                      method (.getMethod model-cls setter (into-array [(option->class-type setter-option)]))]
                  (.invoke method item (object-array [option-val])))))
         dorun)))


(defn construct
  [class-metadata package-name options]
  (let [metadata-options (get class-metadata :options)
        setter-options (seq (filter :setter metadata-options))
        metadata-options (remove :setter metadata-options)
        full-class (class-name->class package-name (:class-name class-metadata))
        ^Constructor constructor (let [arg-types (->> metadata-options
                                                      (map #(option->class-type %))
                                                      seq)]
                                   (if arg-types
                                     (.getConstructor full-class (into-array ^Class arg-types))
                                     (.getConstructor full-class nil)))
        arguments (->> metadata-options
                       (map #(->> (get options
                                       (:name %)
                                       (:default %))
                                  (option-value->value %)))
                       object-array)
        retval (try
                 (.newInstance constructor arguments)
                 (catch Throwable e
                   (throw (ex-info "Failed to construct class"
                                   {:class-metadata class-metadata
                                    :options options
                                    :error e}))))]
    (when setter-options
      (set-setter-options! setter-options retval options))
    retval))

(defn get-option-value
  [metadata option-name options]
  (get options option-name
       (->> metadata
            :options
            (group-by :name)
            option-name
            first
            :default)))
